//===========================================================================
//
// Copyright (c)  2002-2010 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.activeDirectory;
import java.io.FileInputStream;
import java.io.IOException;

import javax.naming.NamingException;

import com.entrust.toolkit.User;
import com.entrust.toolkit.credentials.CredentialReader;
import com.entrust.toolkit.credentials.FilenameProfileReader;
import com.entrust.toolkit.credentials.PKCS12Reader;
import com.entrust.toolkit.exceptions.UserFatalException;
import com.entrust.toolkit.util.HttpsLdapServer;
import com.entrust.toolkit.util.ManagerTransport;
import com.entrust.toolkit.util.SecureStringBuffer;
import com.entrust.toolkit.x509.LdapDirectory;
import com.entrust.toolkit.x509.directory.JNDIDirectory;

/**************************************************************************************
 * ActiveDirectoryServer is an example of an SSL-enabled proxy to MS Active Directory
 * <p>
 * Note: While debugging, use the following command-line parameter to the JVM,
 *       to get detail on the exception messages from JNDIDirectory;
 * </p>
 * <pre>
 *   java -Dcom.entrust.toolkit.x509.directory.JNDIDirectory.trace=1 ActiveDirectoryServer
 * </pre>
 * While testing or debugging, you might want to run this SSL proxy with other
 * kinds of X.500 Directories -- simply omit the optional command line parameters
 * that specify the MS Windows user name and login password.  In fact, while testing
 * or debugging your system, you can use this proxy as a simple SSL-enabled web 
 * server that returns HTML pages to a web browser.
 **************************************************************************************/

public class ActiveDirectoryServer
{
    public static final int SERVER_LIFETIME=3600;
    
    // default port numbers
    public static final int LDAP_PORT=389;
    public static final int CMP_PORT=829;
  
    /**
     * main method starts the proxy
     */
    public static void main(String[] args)
    throws UserFatalException, NamingException, IOException
    {
        // Check the command-line arguments.  If they're not there, print a usage message.
        if (args.length < 3) {
            System.out.println();
            System.out.println("ActiveDirectoryServer - serves LDAP over HTTP or HTTPS");
            System.out.println();
            System.out.println("usage: ActiveDirectoryServer <propertiesfile> <Entrust profile> <Entrust profile password> " +
                                " [ -dir <Directory IP address> ] " +
                                " [ -port <Directory port> ] " +
                                " [ -AD <MS Windows username> <MS Windows login password> ] " +
                                " [ -pki <Authority IP address> ]" + 
                                " [ -pki_port <Authority port> ]");
            System.out.println();
            System.exit(0);
        }
        
        int index = 0;

        // The proxy server properties file
        String propertiesFile = args[index++];

        // a user whose credentials will be used to do SSL
        String profile = args[index++];
        String password = args[index++];

        // The Windows 2000 login name of a user on the Active Directory
        // Note: This is usually NOT the user's X.500 Distinguished Name.
        String WindowsUserName = null;

        // Windows 2000 login password of that User.  Note: This is the
        // user's Windows 2000 login password, not the Entrust profile password.
        SecureStringBuffer WindowsPassword = null;

        // The IP addresses of the PKI's Authority and Directory components
        String directoryIPAddress = null ;
        String authorityIPAddress = null ;
        int directoryPort = LDAP_PORT;
        int authorityPort = CMP_PORT;
        
        while (args.length > index) {

            if (args[index].equals("-dir")) {
                directoryIPAddress = args[++index] ;
            }

            if (args[index].equals("-port")) {
                directoryPort = Integer.parseInt(args[++index]) ;
            }

            if (args[index].equalsIgnoreCase("-ad")) {

                // need these for Simple Authentication to Active Directory
                WindowsUserName = args[++index];
                WindowsPassword = new SecureStringBuffer(new StringBuffer(args[++index]));
            }
            if (args[index].equals("-pki")) {
                authorityIPAddress = args[++index];
            }
            if (args[index].equals("-pki_port")) {
                authorityPort = Integer.parseInt(args[++index]) ;
            }
            ++index;
        }
        
        // Display the warning about Simple Authentication
        showWarning();
        
        // Get a connection to Active Directory for the Proxy to use
        JNDIDirectory dir = null;
        if(directoryIPAddress != null) {
            
            dir = new JNDIDirectory(directoryIPAddress, directoryPort);
            
            // throw an exception if the proxy itself cannot read the Directory
            checkAuthentication(dir, WindowsUserName, WindowsPassword);
        }
        
        // Optional:  Set a connection to the Entrust Authority, 
        // for key management of the user who represents the AD proxy
        ManagerTransport emt = null ;
        if(authorityIPAddress != null) {
            emt = new ManagerTransport(authorityIPAddress, authorityPort);
        }

        // Log in, so we can acquire SSL credentials
        SecureStringBuffer profilePassword = new SecureStringBuffer(new StringBuffer(password));
        User user = login(profile, profilePassword, dir, emt);

        // Initialize an SSL-enabled proxy
        HttpsLdapServer server = new HttpsLdapServer(propertiesFile, user);

        // Provide a Directory instance for the server to proxy.  In this example,
        // it proxies the Directory where its own certificates reside.
        server.setDirectory(dir);

        /* Optional:  If the X.500 Directory is a Microsoft Active Directory, the proxy
         * must do "Simple Authentication" to it, in order to gain 'read' access.
         * But in this particular example, the JNDIDirectory instance was already
         * AD-enabled above, so it can access Active Directory.  Otherwise, do this;
         *
         * server.initActiveDirectory("simple", WindowsUserName, WindowsPassword);
         */
        
        
        /* Optional:  If the main Directory contains LDAP referrals to other Directories,
         * you should attach those Directories.  The Toolkit will automatically follow 
         * LDAP referrals contained in the main Directory and retrieve objects that reside
         * in the referred Directories.  If a referred Directory is a Microsoft Active Directory,
         * provide its authentication credentials here.   Only "simple" authentication is
         * supported, so the Windows login password will be sent in the clear;  therefore,
         * you should configure that Windows user so it has minimal privileges.  E.g.:
         *
         * JNDIDirectory refDir = new JNDIDirectory("myLdapDirectory.mydomain.com", 389);
         * refDir.setAuthentication("simple",
         *                         "NameOfWindowsUserOnReferredDirectory",
         *                          new SecureStringBuffer("PasswordOfWindowsUserOnReferredDirectory"));
         * dir.attachReferredDirectory(refDir);  // attach refDir to the main directory
         * refDir.resetAuthentication();
         */

        // Test the proxy's connection to the X.500 Directory
        server.connect();

        // Finished with the Active Directory password
        if(WindowsPassword != null){
            WindowsPassword.wipe();
        }

        // Display the server configuration
        server.printConfiguration();

        // Start the proxy
        server.start();
        
        // As a demo, let the server run for one hour, then stop it.
        // A real application would probably let it run "forever".
        try {
            Thread.sleep(1000 * SERVER_LIFETIME);
        }
        catch (InterruptedException e)
        {
        }

        // Just as an example, how to stop the server.
        System.out.println("Stopping the server...");
        // server.stop();

        System.out.println("Done.");
    }
    
    
    
  /**
   * Warn the user about the Simple Authentication procedure the proxy uses
   * to acquire read access to Active Directory.  That protocol sends the 
   * MS Windows login password in the clear to Active Directory.
   */
    private static void showWarning()
    throws IOException
    {
        System.out.println("\nWarning: This proxy uses Simple Authentication to Active Directory.\n" + 
                            "Run this proxy on same computer as Active Directory or on a computer \n" + 
                            "that has a secure network connection to it.\n\n" + 
                            "Continue (y/n) ? ");
        int response = System.in.read(); 
        if(response != 'y') {
            System.out.println("Exiting.");
            System.exit(0);
        }
        System.out.println();
    }


  /**
   * This private helper method confirms that the proxy itself can read
   * from the Active Directory.
   *
   * @param dir the JNDIDirectory instance
   * @param WindowsUserName The MS Windows user name of the user who is the proxy
   * @param WindowsPassword The Windows login password.
   */
    private static void checkAuthentication(JNDIDirectory dir,
                                       String WindowsUserName,
                                       SecureStringBuffer WindowsPassword)
    throws UserFatalException                                                  
    {
        
        // If it's not an Active Directory, nothing to do.
        if(WindowsUserName == null)
            return;

        System.out.println("\"" + WindowsUserName + "\"" 
                                + " is attempting to connect to Active Directory...");
                                
        // Try Simple Authentication.
        dir.setAuthentication("simple", WindowsUserName, WindowsPassword);
        if (!dir.isAvailable()) {
            
            // Simple Authentication has failed, so this user cannot be a proxy.
            WindowsPassword.wipe();
            System.out.println("\"" + WindowsUserName + "\"" 
                                + " failed to authenticate to Active Directory !");
                                
            System.out.println("Attempting to connect without authentication...");

            // Just to check the configuration, try to connect without authentication.
            dir.setAuthentication("none", WindowsUserName, new SecureStringBuffer(new StringBuffer("")));
            if (!dir.isAvailable()) {
                                
                // Can't connect to Active Directory at all (is it running at this IP ?)
                throw new UserFatalException("Windows user \"" + WindowsUserName + "\"" 
                                    + " cannot access Active Directory at all !"); 
            }
                
            // Connected without authentication.  This confirms the Directory
            // is running, IP and port are correct, etc.;  however, without 
            // authentication, this particular Windows user cannot read the  
            // User's Area in an Active Directory, so cannot be a Proxy.
            
            System.out.println("Connected.");
            
            throw new UserFatalException("Windows user \"" + WindowsUserName + "\"" + 
                      " can only connect to AD without authentication," + 
                      " so is not permitted to read the Users Area !"); 
        }
        System.out.println("Windows user \"" + WindowsUserName + "\"" 
                            + " has authenticated to Active Directory."); 
    }
    
  /**
   * Logs in to an Entrust user profile.
   *
   * @param profile The name of the Entrust user profile (e.g. "user.epf").
   * @param password The password.
   */
    private static User login(String profile,
                              SecureStringBuffer password,
                              LdapDirectory dir,
                              ManagerTransport emt)
    throws UserFatalException
    {
        try {
            System.out.println("Logging in to: " + profile);

            CredentialReader credreader = null ;

            if(profile.endsWith(".epf")) {
                credreader = new FilenameProfileReader(profile);
            }
            else if (profile.endsWith(".p12") || profile.endsWith(".pfx")) {
                credreader = new PKCS12Reader(new FileInputStream(profile));
            }

            User user = new User() ;

            // Set connections to the Directory and the Authority
            if(dir != null) {
                user.setConnections(dir, emt);
            }

            user.login(credreader, password);
            return user ;
        }
        catch (Exception e) {
            throw new UserFatalException("Error logging in: " + e.getMessage());
        }
    }
}
