//===========================================================================
//
//  Copyright (c)  1998-2010 Entrust.  All rights reserved.
//
//===========================================================================

package com.entrust.toolkit.examples.applet;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import iaik.utils.Util;

/**
 * The class represents a message header used in messages sent between the
 * client and the server.
 * 
 * <p>
 * A message header consists of the following:
 * </p>
 * 
 * <pre>
 *     sentinal | message type | message length
 * </pre>
 * 
 * <p>
 * The sentinal is a fixed set of bytes; it is used to help verify the validity
 * of received message.  If the received sentinal does not match the know
 * sentinal value, the message header is invalid.
 * </p>
 * 
 * <p>
 * The message type is a single byte that indicates whether the message is a 
 * real message or a test message.  A real message will consist of encrypted
 * data being transferred from the Client to the Server.  A test message is
 * data being sent in the clear to the Client from the Server and back from the
 * Server to the Client.  It is used to test the connection.
 * </p>
 * 
 * <p>
 * The message length is a 4 byte component that indicates the length of the 
 * message that is associated with the message header.
 * </p>
 */
class MessageHeader {

    //*****************************************************************************
    //                                 CONSTANTS
    //*****************************************************************************

    /** Message type that indicates a test message. */
    static final byte MESSAGE_TYPE_TEST = 0;

    /** Message type that indicates a real message. */
    static final byte MESSAGE_TYPE_REAL = 1;

    /** The message header sentinal; used to help ensure the message header is
     *  valid. */
    private static final byte[] SENTINAL = Util.toASCIIBytes("AppletDemo");

    //*****************************************************************************
    //                                   FIELDS
    //*****************************************************************************

    /** The message type component. */
    private byte m_messageType;

    /** The message length component. */
    private int m_messageLength;

    //*****************************************************************************
    //                                CONSTRUCTORS
    //*****************************************************************************

    /**
     * Creates a message header with the indicated message length and message
     * type components.
     * 
     * @param messageLength
     *      the message length
     * 
     * @param messageType
     *      the message type
     */
    MessageHeader(byte messageType, int messageLength) {
        m_messageType = messageType;
        m_messageLength = messageLength;
    }

    /**
     * Creates a message header by parsing it from an input stream.
     * <p></p>
     * 
     * @param firstHeaderByte
     *      the first byte of the message header
     *      
     * @param is
     *      the input stream from which the rest of the message header is to
     *      be read
     * 
     * @throws IOException
     *      if an error occurs while parsing the message header, or the parsed
     *      message header is invalid
     */
    MessageHeader(byte firstHeaderByte, InputStream is) throws IOException {
        // Read the entire header
        int sentinalSize = SENTINAL.length;
        int headerSize = sentinalSize + 1 + 4;
        byte[] headerBytes = new byte[headerSize];
        headerBytes[0] = firstHeaderByte;
        int bytesToRead = headerSize - 1;
        while (bytesToRead > 0)
            bytesToRead -= is.read(headerBytes, headerSize - bytesToRead, bytesToRead);

        // Verify the sentinal
        int i;
        for (i = 0; i < sentinalSize; i++) {
            if (SENTINAL[i] != headerBytes[i])
                throw new IOException("Invalid message header; incorrect sentinal");
        }

        // Ensure the message type is valid
        m_messageType = headerBytes[i++];
        if (m_messageType != MESSAGE_TYPE_TEST && m_messageType != MESSAGE_TYPE_REAL)
            throw new IOException("Invalid message type; supported [0, 1], found " + m_messageType);

        // Read the message length
        m_messageLength = headerBytes[i++] << 24;
        m_messageLength |= (headerBytes[i++] & 0xff) << 16;
        m_messageLength |= (headerBytes[i++] & 0xff) << 8;
        m_messageLength |= headerBytes[i] & 0xff;
        if (m_messageLength < 0)
            throw new IOException("Invalid message length; negative message length not permitted, found "
                + m_messageLength);
    }

    //*****************************************************************************
    //                                    APIS
    //*****************************************************************************

    /**
     * Returns the message type component from the message header.
     * <p></p>
     * 
     * @return
     *      the message type
     */
    byte getMessageType() {
        return m_messageType;
    }

    /**
     * Returns the message length component from the message header.
     * <p></p>
     * 
     * @return
     *      the message length
     */
    int getMessageLength() {
        return m_messageLength;
    }

    /**
     * Writes the message header to an output stream.
     * <p></p>
     * 
     * @param os
     *      the output stream the message header will be written to
     * 
     * @throws IOException
     *      if an error occurs while writing the message header
     */
    void writeTo(OutputStream os) throws IOException {
        // Construct the message header
        int sentinalSize = SENTINAL.length;
        int headerSize = sentinalSize + 1 + 4;
        byte[] headerBytes = new byte[headerSize];
        System.arraycopy(SENTINAL, 0, headerBytes, 0, sentinalSize);
        headerBytes[sentinalSize] = m_messageType;
        headerBytes[sentinalSize + 1] = (byte) (m_messageLength >> 24);
        headerBytes[sentinalSize + 2] = (byte) (m_messageLength >> 16);
        headerBytes[sentinalSize + 3] = (byte) (m_messageLength >> 8);
        headerBytes[sentinalSize + 4] = (byte) m_messageLength;

        // Write the message header
        os.write(headerBytes);
        os.flush();
    }
}