//===========================================================================
//
// Copyright (c)  1998-2010 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.applet;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.ServerSocket;
import java.net.Socket;

import com.entrust.toolkit.PKCS7DecodeStream;
import com.entrust.toolkit.User;
import com.entrust.toolkit.credentials.CredentialReader;
import com.entrust.toolkit.credentials.FilenameProfileReader;
import com.entrust.toolkit.security.provider.Initializer;
import com.entrust.toolkit.util.IniFile;
import com.entrust.toolkit.util.ManagerTransport;
import com.entrust.toolkit.util.SecureStringBuffer;
import com.entrust.toolkit.x509.directory.JNDIDirectory;

/**
 * This class represents a Server that is designed to communicate securely with 
 * a corresponding Client, using PKCS#7 to secure communications.
 * 
 * <p>
 * The Server is able to receive and decrypt information that was encrypted
 * and sent by a Client over a socket.  The same Entrust User is used by the
 * Client to encrypt messages and by the Server to decrypt messages.
 * </p>
 * 
 * <p>
 * This is intended to provide an example of how the Toolkit can be used to
 * facilitate secure communication in a Client/Server environment.  This Server
 * is single-threaded and not intended to operate in a production environment.
 * </p>
 */
public class ServerApplication {

    //*****************************************************************************
    //                                  CONSTANTS
    //*****************************************************************************    

    /** The maximum block size used when receiving a message from the client. */
    private static final int MAX_RECEIVE_BLOCKSIZE = 1024 * 16;

    /** Receive connection timeout. */
    private static final int RECEIVE_CONNECTION_TIMEOUT = 5000;

    //*****************************************************************************
    //                                APPLICATION
    //*****************************************************************************    

    /**
     * The server application.
     * 
     * <p>
     * Run the program with the "-?" console argument for more information.
     * </p>
     */
    public static void main(String[] args) {

        // Check if the user requested help
        if (getBooleanArg(args, "-?", false) || getBooleanArg(args, "-h[elp]", false)) {
            displayHelpAndExit(1);
        }

        // Load the INI file specified in the arguments, or if not provided, use
        // the default one
        String iniFileName = getStringArg(args, "-f[ile]", "data/applet/Server.ini");
        IniFile testData = null;
        try {
            testData = new IniFile(iniFileName);
        } catch (FileNotFoundException e) {
            System.out.println("(!)ERROR: " + e.toString());
            return;
        }

        // Initialize parameters with values from the INI file
        String profileName = testData.getString("Server", "defaultProfile", null);
        String profilePass = testData.getString("Server", "defaultProfilePassword", null);
        String directoryIP = testData.getString("Server", "defaultDirectoryIP", null);
        int directoryPort = testData.getInt("Server", "defaultDirectoryPort", 389);
        String managerIP = testData.getString("Server", "defaultManagerIP", null);
        int managerPort = testData.getInt("Server", "defaultManagerPort", 829);
        int serverPort = testData.getInt("Server", "defaultServerPort", 1234);

        // Check for parameters passed in by user as arguments; when present, these
        // values are used instead of the values from the INI file
        profileName = getStringArg(args, "-prof[ile]", profileName);
        profilePass = getStringArg(args, "-pass[word]", profilePass);
        directoryIP = getStringArg(args, "-dirIP", directoryIP);
        managerIP = getStringArg(args, "-manIP", managerIP);
        directoryPort = Integer.parseInt(getStringArg(args, "-dirPort", String.valueOf(directoryPort)));
        managerPort = Integer.parseInt(getStringArg(args, "-manPort", String.valueOf(managerPort)));
        serverPort = Integer.parseInt(getStringArg(args, "-P[ort]", String.valueOf(serverPort)));
        String serverIP = "localhost";

        //  Display the actual parameters we will use if requested.
        System.out.println("Server configuration data was loaded from: " + iniFileName + "\n");
        System.out.println("Server Configuration Data");
        System.out.println("------------------------------------------------------------");
        System.out.println("  Test Profile Location       : " + profileName);
        System.out.println("  Directory IP Address        : " + directoryIP);
        System.out.println("  Directory Port              : " + directoryPort);
        System.out.println("  Security Manager IP Address : " + managerIP);
        System.out.println("  Security Manager Port       : " + managerPort);
        System.out.println("  Server Will Use IP Address  : " + serverIP);
        System.out.println("  Server Will Use Port        : " + serverPort);
        System.out.println("------------------------------------------------------------\n");

        // Initialize the Toolkit
        System.out.print("Initializing the Entrust Toolkit... ");
        long time = System.currentTimeMillis();
        Initializer.getInstance().setProviders(Initializer.MODE_NORMAL);
        System.out.println("DONE [" + (System.currentTimeMillis() - time) + " ms]");

        // Login to the User
        System.out.print("Logging in to the User (allows secure communication)... ");
        time = System.currentTimeMillis();
        User user = new User();
        try {
            if (directoryIP != null && managerIP != null && directoryIP.equals("") && managerIP.equals("")) {
                // To log in online, connections to the PKI infrastructure are 
                // required; the initialization data contains the necessary
                // connection information
                JNDIDirectory directory = new JNDIDirectory(directoryIP, directoryPort);
                ManagerTransport manager = new ManagerTransport(managerIP, managerPort);
                user.setConnections(directory, manager);
            }
            CredentialReader cr = new FilenameProfileReader(profileName);
            SecureStringBuffer securePassword = new SecureStringBuffer(new StringBuffer(profilePass));
            user.login(cr, securePassword);
            securePassword.wipe();
            System.out.println("DONE [" + (System.currentTimeMillis() - time) + " ms]");
        } catch (Exception e) {
            System.out.println("FAILED");
            System.out.println("(!)ERROR: " + e.toString());
            return;
        }

        // Start the Server
        ServerSocket serverSocket = null;
        try {
            System.out.print("Starting the Server...");
            time = System.currentTimeMillis();
            serverSocket = new ServerSocket(serverPort);
            System.out.println("DONE [" + (System.currentTimeMillis() - time) + " ms]\n");
        } catch (Exception e) {
            System.out.println("FAILED");
            System.out.println("(!)ERROR: " + e.toString());
            return;
        }
        System.out.println("-- Server Started");

        // Handle client connection
        clientConnection: while (true) {
            // Wait for client connections
            System.out.print("\nWaiting for client connection... ");
            Socket socket = null;
            InputStream is = null;
            OutputStream os = null;
            try {
                socket = serverSocket.accept();
                is = socket.getInputStream();
                os = socket.getOutputStream();
                System.out.println("DETECTED");
            } catch (IOException e) {
                System.out.println("FAILED");
                System.out.println("(!)ERROR: " + e.toString());
                return;
            }
            String clientAddress = socket.getInetAddress().toString();
            System.out.println("[" + clientAddress + "] Client connected");

            // Handle client transmission
            clientTransmission: while (true) {

                // Wait for transmission
                System.out.print("\nWaiting for client transmission... ");
                byte firstHeaderByte;
                try {
                    // Blocks until the client sends a byte or the connection is
                    // terminated
                    socket.setSoTimeout(0);
                    firstHeaderByte = (byte) is.read();
                    System.out.println("DETECTED");
                } catch (IOException e) {
                    System.out.println("ABORTED");
                    System.out.println("[" + clientAddress + "] Client disconnected");
                    continue clientConnection;
                }
                // Handle client disconnect
                if (firstHeaderByte == -1) {
                    System.out.println("ABORTED");
                    System.out.println("[" + clientAddress + "] Client disconnected");
                    continue clientConnection;
                }

                // Read the header
                System.out.println("[" + clientAddress + "] Client transmission detected");
                System.out.print("[" + clientAddress + "]   Receiving message header... ");
                MessageHeader header = null;
                try {
                    socket.setSoTimeout(RECEIVE_CONNECTION_TIMEOUT);
                    header = new MessageHeader(firstHeaderByte, is);
                    System.out.println("DONE");
                } catch (IOException e) {
                    System.out.println("FAILED");
                    System.out.println("[" + clientAddress + "]   (!)ERROR: " + e.toString());
                    System.out.println("[" + clientAddress + "] Connection dropped by Server");
                    continue clientConnection;
                }

                // Read the message
                int messageLength = header.getMessageLength();
                System.out.print("[" + clientAddress + "]   Receiving '" + messageLength + "' message bytes... ");
                byte[] receivedMessage = new byte[messageLength];
                try {
                    // Read message in reasonable size chunks
                    int bytesReceived = 0;
                    int blockSize = MAX_RECEIVE_BLOCKSIZE;
                    while (bytesReceived < messageLength) {
                        blockSize = messageLength - bytesReceived;
                        if (blockSize > MAX_RECEIVE_BLOCKSIZE)
                            blockSize = MAX_RECEIVE_BLOCKSIZE;
                        bytesReceived += is.read(receivedMessage, bytesReceived, blockSize);
                    }
                    System.out.println("DONE");
                } catch (Exception e) {
                    System.out.println("FAILED");
                    System.out.println("[" + clientAddress + "]   (!)ERROR: " + e.toString());
                    System.out.println("[" + clientAddress + "] Connection dropped by Server");
                    continue clientConnection;
                }

                // Process message
                try {
                    if (header.getMessageType() == MessageHeader.MESSAGE_TYPE_TEST) {
                        // Handle a test message                
                        System.out.println("[" + clientAddress + "]   Received a test message");
                        System.out.print("[" + clientAddress + "]   Sending test message back to client... ");
                        os.write(receivedMessage);
                        os.flush();
                        System.out.println("DONE");
                    } else {
                        // Decrypt message
                        System.out.println("[" + clientAddress + "]   Received an encrypted message");
                        System.out.print("[" + clientAddress + "]   Decrypting message... ");
                        PKCS7DecodeStream decrypter = new PKCS7DecodeStream(user, new ByteArrayInputStream(
                            receivedMessage));
                        ByteArrayOutputStream decryptedMessage = new ByteArrayOutputStream();
                        byte[] buffer = new byte[MAX_RECEIVE_BLOCKSIZE];
                        int bytesRead;
                        while ((bytesRead = decrypter.read(buffer)) > 0)
                            decryptedMessage.write(buffer, 0, bytesRead);
                        decrypter.close();
                        System.out.println("DONE");
                        System.out.println("---- Message (Begin) ----");
                        System.out.println(new String(decryptedMessage.toByteArray()));
                        System.out.println("---- Message (End)   ----");
                    }
                } catch (Exception e) {
                    System.out.println("FAILED");
                    System.out.println("[" + clientAddress + "]   (!)ERROR: " + e.toString());
                    System.out.println("[" + clientAddress + "] Connection dropped by Server");
                    continue clientConnection;
                }

                System.out.println("[" + clientAddress + "] Client transmission completed");

            }
        }
    }

    //*****************************************************************************
    //                           INTERNAL HELPER METHODS
    //*****************************************************************************

    /**
     * Displays help message to the console and terminates the application.
     * <p></p>
     *
     * @param exitCode 
     *      the code to call <code>System.exit</code> with
     */
    private static void displayHelpAndExit(int exitCode) {
        System.out.println("usage: java Server [options]");
        System.out.println("");
        System.out.println("where options include:");
        System.out.println("    -prof[ile] <path>     specify a profile to test");
        System.out.println("    -pass[word] <pw>      specify the profile password");
        System.out.println("    -P[ort] <port>        specify a port for the server to use");
        System.out.println("    -dirIP <ip>           specify the directory server IP address");
        System.out.println("    -dirPort <port>       specify the directory server port");
        System.out.println("    -manIP <ip>           specify the manager server IP address");
        System.out.println("    -manPort <port>       specify the manager server port");
        System.out.println("    -f[ile] <file>        use a different INI file - default 'data/applet/Server.ini'");
        System.out.println("    -h[elp] -?            display this screen");
        System.exit(exitCode);
    }

    /**
     * Searches an array of <code>String</code> arguments, looking for the specified 
     * argument.
     * <p></p>
     * 
     * @param argList
     *      the array of arguments to be searched
     * 
     * @param arg 
     *      the name of the argument whose value is to be returned
     * 
     * @param defaultValue 
     *      the value to return if the argument is not among the parameters
     * 
     * @return
     *      <code>true</code> if the argument exists; otherwise the specified 
     *      default value is returned
     */
    private static boolean getBooleanArg(String[] argList, String arg, boolean defaultValue) {
        int argCount = argList.length;
        if (arg.indexOf("[") >= 0) {
            String shortArg = arg.substring(0, arg.indexOf("["));
            String longArg = arg.substring(arg.indexOf("[") + 1, arg.lastIndexOf("]"));
            for (int i = 0; i < argCount; i++) {
                if (shortArg.equalsIgnoreCase(argList[i]))
                    return true;
                if (longArg.equalsIgnoreCase(argList[i]))
                    return true;
            }
        } else {
            for (int i = 0; i < argCount; i++) {
                if (arg.equalsIgnoreCase(argList[i]))
                    return true;
            }
        }
        return defaultValue;
    }

    /**
     * Searches an array of <code>String</code> arguments, looking for the specified 
     * argument.
     * <p></p>
     * 
     * @param argList
     *      the array of arguments to be searched
     * 
     * @param arg 
     *      the name of the argument whose value is to be returned
     * 
     * @param defaultValue 
     *      the value to return if the argument is not among the parameters
     * 
     * @return 
     *      the value of the argument or the specified default if the argument 
     *      cannot be found
     */
    private static String getStringArg(String[] argList, String arg, String defaultValue) {
        int argCount = argList.length;
        if (arg.indexOf("[") >= 0) {
            String shortArg = arg.substring(0, arg.indexOf("["));
            String longArg = arg.substring(arg.indexOf("[") + 1, arg.lastIndexOf("]"));
            for (int i = 0; i < argCount; i++) {
                if (shortArg.equalsIgnoreCase(argList[i]))
                    return argList[i + 1];
                if (longArg.equalsIgnoreCase(argList[i]))
                    return argList[i + 1];
            }
        } else {
            for (int i = 0; i < argCount; i++) {
                if (arg.equalsIgnoreCase(argList[i]))
                    return argList[i + 1];
            }
        }
        return defaultValue;
    }
}