//===========================================================================
//
// Copyright (c) 2004-2010 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.cache;

import iaik.x509.X509Certificate;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.security.cert.CRLException;

import com.entrust.toolkit.User;
import com.entrust.toolkit.credentials.CredentialReader;
import com.entrust.toolkit.credentials.FilenameProfileReader;
import com.entrust.toolkit.credentials.UserStatus;
import com.entrust.toolkit.exceptions.CertificationException;
import com.entrust.toolkit.exceptions.UserException;
import com.entrust.toolkit.exceptions.UserFatalException;
import com.entrust.toolkit.security.provider.Initializer;
import com.entrust.toolkit.util.SecureStringBuffer;
import com.entrust.toolkit.x509.certstore.ArchiveCertCache;
import com.entrust.toolkit.x509.revocation.ArchiveCRLCache;

/**
 * Example to demonstrate how to use cache files to log in
 * offline. Some situations where this might be required are when
 * a user is in a multiple-level CA hierarchy and does not
 * have all of their certificates in the epf file, or requires 
 * revocation information but does not have access to a Directory.
 * 
 * <p>
 * Usage:
 * <pre>
 * OfflineLoginUsingCache path_to_epf password
 * </pre>
 * <dl>
 * <dt>path_to_epf</dt><dd>Path to EPF file. For example, data/userdata/RSAUser1.epf</dd>
 * <dt>password</dt><dd>The password for the EPF. For example, ~Sample7~</dd>
 * </dl>
 * </p>
 */
public class OfflineLoginUsingCache
{
    /**
     * The main driver for the sample.
     * 
     * @param args
     *     Program arguments. See the class documentation for the usage.
     */
    public static void main(String[] args)
    {
        if (args.length != 2)
        {
            System.out.println("Usage: java OfflineLoginUsingCache <path_to_epf> <password>");
            return;
        }

        // Make sure the Entrust Provider is installed so that crypto operations
        // will work.
        Initializer.getInstance().setProviders(Initializer.MODE_NORMAL);

        OfflineLoginUsingCache cache = new OfflineLoginUsingCache();
        try
        {
            cache.run(args[0], args[1]);
            System.out.println("Test ran successfully");
        }
        catch (Exception e)
        {
            System.err.println("Sample failed to run successfully");
            e.printStackTrace();
        }
    }

    /**
     * The actual code that runs the sample. This does the following:
     * <ul>
     * <li>Reads the cross-certificate (.xcc) and user certificate (.cch)
     * cache files, if available.</li>
     * <li>Logs in to the supplied epf file</li>
     * <li>Turns on the requirement for revocation information, then
     * tries to validate a certificate. This should fail as there
     * is no revocation information available.</li>
     * <li>Reads the certificate and authority revocation list cache
     * files (.crl and .arl), if available.</li>
     * <li>Tries to verify the certificate again. If the cache files
     * were loaded, this verification should work.</li>
     * <li>Writes the cache files.</li>
     * <li>Logs out.</li>
     * </ul>
     * 
     * @param epfFileName
     *     the epf file to log in to
     * 
     * @param password
     *     the password for the epf file
     * 
     * @throws Exception
     *     in case of errors. Error handling is very coarse to keep the code simpler.
     */
    public void run(String epfFileName, String password) throws Exception
    {
        User user = new User();

        try
        {
            // Read certificate cache files
            readCertCacheFiles(epfFileName);

            // log in to the epf file.
            System.out.println("Attempting to log in to " + epfFileName);
            CredentialReader cr = new FilenameProfileReader(epfFileName);
            user.login(cr, new SecureStringBuffer(new StringBuffer(password)));

            // Display the status returned from logging in.
            UserStatus status = user.getStatus();
            System.out.println("Status returned by login:");
            System.out.println(status);

            // Turn on the requirement for revocation information
            // when validating  certificates.
            user.requireCRL(true);

            X509Certificate userCert = user.getVerificationCertificate();
            if (userCert != null)
            {
                try
                {
                    user.validate(userCert);
                    throw new Exception("Validation should have failed");
                }
                catch (CertificationException e)
                {
                    // This is expected, as there are no CRLs available.
                }

                // Read the revocation list cache for the given user.
                readCrlCacheFiles(user, epfFileName);

                // Try validating the certificate again. If valid CRLs were
                // loaded, this should work.
                user.validate(userCert);
            }

            // Write the cache files out
            writeUserCacheFiles(user, epfFileName);

        }
        finally
        {
            // Always make sure to log the user out.
            if (user.isLoggedIn())
            {
                user.logout();
            }
        }
    }

    /**
     * Read the certificate cache files associated with an epf file.
     * This reads the cache files from a file, and adds them to
     * the in-memory cache used during certificate validation.
     * 
     * @param epfFileName
     *     the name of the epf file for which the associated cache files will
     * be loaded.
     * 
     * @throws IOException
     *     if there is a problem reading the files.
     * 
     * @throws UserFatalException
     *     if there is a problem decrypting the cache files.
     */
    public void readCertCacheFiles(String epfFileName) throws IOException, UserFatalException
    {
        System.out.println("Attempting to read cache files for " + epfFileName);
        String userName = getUserNameFromPath(epfFileName);

        // Load the cross certificate cache.
        String xccFileName = userName + ".xcc";
        File xccFile = new File(xccFileName);
        if (xccFile.exists() && xccFile.canRead() && !xccFile.isDirectory())
        {
            System.out.println("Reading cross-certificate cache " + xccFileName);
            InputStream is = new FileInputStream(xccFile);
            ArchiveCertCache xccCache = new ArchiveCertCache(is);
            // Call cleanup to remove any expired certificates from the cache.
            xccCache.cleanup();

            // Copy the values read from disk in to memory.
            xccCache.initMemoryCache();
        }

        // Load the end-user certificate cache.
        String cchFileName = userName + ".cch";
        File cchFile = new File(cchFileName);
        if (cchFile.exists() && cchFile.canRead() && !cchFile.isDirectory())
        {
            System.out.println("Reading end-user certificate cache " + cchFileName);
            InputStream is = new FileInputStream(cchFile);
            ArchiveCertCache cchCache = new ArchiveCertCache(is);
            // Call cleanup to remove any expired certificates from the cache.
            cchCache.cleanup();

            // Copy the values read from disk in to memory.
            cchCache.initMemoryCache();
        }
    }

    /**
     * Read the CRL cache files for a user. This a chicken-and-egg problem, as
     * the toolkit requires the CRLs to be validated, but they can't be validated
     * without access to a root CA certificate, which can't be retrieved until
     * the <code>User.login()</code> call has completed.  So the user has to
     * log in without revocation information, after which the cached
     * versions can be loaded.
     * 
     * @param user
     *     the User to load the cache files for.
     *  
     * @param epfFileName
     *     the name of the user's epf file.
     * 
     * @throws IOException
     *     if there is an error reading from the streams. 
     * 
     * @throws UserException
     *     if there is an error decoding the data in the cache files, or the user is not
     * logged in.
     */
    public void readCrlCacheFiles(User user, String epfFileName) throws IOException, UserException
    {
        String userName = getUserNameFromPath(epfFileName);

        // Try reading the certificate revocation list cache.
        String arlFileName = userName + ".arl";
        File arlFile = new File(arlFileName);
        if (arlFile.exists() && arlFile.canRead() && !arlFile.isDirectory())
        {
            System.out.println("Reading revocation list cache " + arlFileName);
            InputStream is = new FileInputStream(arlFile);
            ArchiveCRLCache arlCache = new ArchiveCRLCache(is, user.getCertVerifier());

            // Remove expired ARLs from the cache
            arlCache.cleanup();

            // Copy the cached revocation lists in to the user's in-memory
            // cache that will be used during validation
            arlCache.initMemoryCache(user.getRevocationStore().getMemoryCRLCache());
        }

        // Try reading the certificate revocation list cache.
        String crlFileName = userName + ".crl";
        File crlFile = new File(crlFileName);
        if (crlFile.exists() && crlFile.canRead() && !crlFile.isDirectory())
        {
            System.out.println("Reading revocation list cache " + crlFileName);
            InputStream is = new FileInputStream(crlFile);
            ArchiveCRLCache crlCache = new ArchiveCRLCache(is, user.getCertVerifier());

            // Remove expired CRLs from the cache
            crlCache.cleanup();

            // Copy the cached revocation lists in to the user's in-memory
            // cache that will be used during validation
            crlCache.initMemoryCache(user.getRevocationStore().getMemoryCRLCache());
        }
    }

    /**
     * Write the user's cache files out to different streams.
     * 
     * @param user
     *     the User whose cache files are to be written.
     * 
     * @param epfFileName
     *     the epf file that was used when logging in to the user.
     * 
     * @throws CRLException
     *     if CRLs in the CRL cache cannot be encoded.
     * 
     * @throws IOException
     *     if there's a problem writing to the streams.
     * 
     * @throws UserException
     *     if the user is not logged in, or there is a problem
     * protecting the cache files.
     */
    public void writeUserCacheFiles(User user, String epfFileName)
        throws CRLException, IOException, UserException
    {
        System.out.println("Writing cache files for " + epfFileName);

        // There's no point writing the caches out to disk in this example.
        // There was no connection to a Directory, so new certificates
        // and CRLs could not have been retrieved. This just dumps the
        // caches to a ByteArrayOutputStream, then displays that
        // to System.out. The procedure for writing to files is similar,
        // with file names created as shown in readCertCacheFiles and
        // readCrlCacheFiles 

        // Write the ARL cache.
        ByteArrayOutputStream os = new ByteArrayOutputStream();
        ArchiveCRLCache arlCache = new ArchiveCRLCache(user.getCertVerifier());
        arlCache.addMemoryCache(user.getRevocationStore().getMemoryCRLCache());
        arlCache.cleanup();
        arlCache.write(os, ArchiveCRLCache.ARL_ONLY);
        os.close();
        System.out.println("The user's ARL cache:");
        os.writeTo(System.out);
        System.out.println("----------------------------------------");
        System.out.println();

        // Write the CRL cache. ARLs and CRLs are kept in the same cache
        // object, so creating a new ArchiveCRLCache would be redundant.
        // The separation of CRL and ARL is purely for external purposes
        // and compatibility with other toolkits.
        os = new ByteArrayOutputStream();
        arlCache.write(os, ArchiveCRLCache.CRL_ONLY);
        os.close();
        System.out.println("The user's CRL cache:");
        os.writeTo(System.out);
        System.out.println("----------------------------------------");
        System.out.println();

        // Write the cross certificate cache.
        ArchiveCertCache certCache = new ArchiveCertCache();
        certCache.addMemoryCache();
        certCache.cleanup();
        os = new ByteArrayOutputStream();
        certCache.write(os, ArchiveCertCache.CROSS_CERT_ONLY);
        os.close();
        System.out.println("The user's cross certificate cache:");
        os.writeTo(System.out);
        System.out.println("----------------------------------------");
        System.out.println();

        // Similar to the CRL/ARL case, certificates are not split
        // up internally based on being cross-certificates or not.
        os = new ByteArrayOutputStream();
        certCache.write(os, ArchiveCertCache.USER_CERT_ONLY);
        os.close();
        System.out.println("The user's end-user certificate cache:");
        os.writeTo(System.out);
        System.out.println("----------------------------------------");
        System.out.println();
    }

    /**
     * Retrieves the part of a string before the epf extension. This
     * can be used to get the part of the file name to append
     * other extensions to when reading or writing cache files.
     * 
     * @param epfFileName
     *     the string from which to strip the ".epf" extension.
     * @return
     *     the string without the ".epf" extension.
     * 
     * @throws IllegalArgumentException
     *     if the given epf file name does not end in .epf.
     */
    private String getUserNameFromPath(String epfFileName) throws IllegalArgumentException
    {
        int epfPos = epfFileName.lastIndexOf(".epf");
        if (epfPos <= 0)
        {
            throw new IllegalArgumentException("The given file name does not represent a valid epf path");
        }
        return epfFileName.substring(0, epfPos);

    }
}