//===========================================================================
//
// Copyright (c)  2019 Entrust.  All rights reserved.
//
//===========================================================================

package com.entrust.toolkit.examples.cms;

import iaik.asn1.CodingException;
import iaik.asn1.ObjectID;
import iaik.asn1.structures.AlgorithmID;
import iaik.cms.AuthenticatedData;
import iaik.cms.AuthenticatedDataStream;
import iaik.cms.CMSException;
import iaik.cms.CompressedDataStream;
import iaik.cms.CompressionProvider;
import iaik.cms.ContentInfoStream;
import iaik.cms.KeyTransRecipientInfo;
import iaik.cms.PasswordRecipientInfo;
import iaik.cms.RecipientInfo;
import iaik.security.spec.PBEKeyAndParameterSpec;
import iaik.smime.SMimeEncrypted;
import iaik.utils.Util;

import java.io.*;
import java.security.GeneralSecurityException;
import java.security.InvalidAlgorithmParameterException;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.security.spec.AlgorithmParameterSpec;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.InvalidParameterSpecException;

import javax.activation.DataHandler;
import javax.mail.MessagingException;
import javax.mail.util.ByteArrayDataSource;

import com.entrust.toolkit.User;
import com.entrust.toolkit.credentials.FilenameProfileReader;
import com.entrust.toolkit.security.provider.Entrust;
import com.entrust.toolkit.security.provider.Initializer;
import com.entrust.toolkit.util.SecureStringBuffer;
import iaik.x509.X509Certificate;


/**
 * This sample shows how to use the Java Toolkit's Cryptographic Message Syntax (CMS) (RFC 2630) functionality
 * to generate a secured file that can be decrypted by Entrust Entelligence Security Provider (ESP) 9.3 or higher.
 * </p>
 *
 * In this sample, the generated encrypted file is secured by both a password and a user's encryption certificate.
 * ESP can then be used to decrypt the file using either the password or the user's Entrust profile (.epf)
 * (which contains the encryption certificate).
 * </p>
 *
 * The sample shows two scenarios:
 * </p>
 * - One scenario where you are encrypting the file for yourself (that is, encrypting it using
 * your own encryption certificate which is in your Entrust user profile (.epf)).
 * </p>
 * - And, another scenario where you are encrypting the file for another ESP user (using their encryption certificate)
 * </p>
 * Additionally, in both scenarios, the file is also encrypted with a password.
 * </p>
 *
 * To generate a secured file in an ESP 9.3 format, the following is done:
 * </p>
 *
 * First, the sample creates the secured CMS <code>EnvelopedData</code> with a <code>PasswordRecipientInfo</code> (for
 * password based encryption for a recipient) and a <code>KeyTransRecipientInfo</code> (for certificate based
 * encryption for a specific user).
 * </p>
 *
 * Then, the sample takes the enveloped data and transforms it into a format that Entelligence Security Provider (ESP)
 * uses.  (Specifically, it wraps the enveloped data in a s/mime layer and then compresses it.)
 * ESP uses a .pp7m file extension for its format (when it creates a file that is secured by both a password and
 * an encryption certificate).  This sample creates the output file with that extension as well (by default).
 * </p>
 *
 * Entelligence Security Provider (ESP) 9.3 or higher can then be used to decrypt this file using either the shared
 * password or the user's Entrust profile (epf) (that is, the user profile which contains the encryption certificate
 * that the file was encrypted for).
 */
public class ESPEncodeMessage {

    /**
     * The main program.
     *
     * @param args
     *        Program arguments. See the help below for the expected command line arguments for this main method.
     * @throws Exception
     */
    public static void main(String args[]) throws Exception {

        // For the intended recipient, either the Entrust user profile and the profile's password can be provided (to
        // be able to extract the encryption certificate), or the recipient's X509 encryption certificate can be directly
        // supplied. Therefore, to determine how to interpret the provided command line arguments, the number of arguments
        // provided must be determined.
        if (args.length < 4 || args.length > 5) {
            System.out.println("Usage: EncodeESPMessage <input filename> <password> {<Entrust profile> <profile password> | <encryption certificate>} <output filename>");
            System.out.println("   where <input filename> is the original plaintext file");
            System.out.println("   where <password> is the password that will be used to protect the encrypted output file\n" +
                               "       (used for the Password Recipient)");
            System.out.println("   where <Entrust profile> is a profile (.epf file) whose encryption certificate will be used\n" +
                               "       to also protect the encrypted output file (used for the Key Transport Recipient)");
            System.out.println("   where <profile password> is the password for the user profile (.epf file)");
            System.out.println("   where <encryption certificate> is file containing the X509 encryption certificate that will be\n" +
                               "       used to protect the encrypted output file (used for the Key Transport Recipient)");
            System.out.println("   where <output filename> is the .pp7m output file where the encrypted contents will be written to,\n" +
                               "       and which Entelligence Security Provider can process\n");
            System.out.println("Note, for the Key Transport Recipient information, either the <Entrust profile> and <profile password>\n" +
                               "must be supplied OR the <encryption certificate>, but not both.\n");
            return;
        }

        Initializer.getInstance().setProviders(Initializer.MODE_NORMAL);

        // The first argument always specifies the input file (which contains the plaintext)
        String filename = args[0];

        // The next argument always specifies the password that will be used for the password recipient
        SecureStringBuffer password = new SecureStringBuffer(new StringBuffer(args[1]));
        // The password can only be used once, and then it gets wiped, so make a copy of the
        // password since it will be used twice (later on)
        SecureStringBuffer passwordCopy = new SecureStringBuffer(new StringBuffer(args[1]));

        // Interpreting the remaining command line arguments depends on how many were supplied.
        // The recipient's encryption certificate is either provided via the user profile (epf) and associated profile
        // password (i.e. two arguments), or directly via a certificate file (i.e. one argument).
        // The final argument is always the output file.
        String epfFile;
        String outFile;
        X509Certificate encryptionCertificate;

        if (args.length == 5) {
            // The user's Entrust profile (epf) and associated profile password were provided.

            // User profile of recipient
            epfFile = args[2];

            // Password to the user profile
            SecureStringBuffer profilePass = new SecureStringBuffer(new StringBuffer(args[3]));

            // The final argument is always the output file.
            outFile = args[4];

            // Retrieve the user's encryption certificate from the user profile.
            // To retrieve the certificate a login is first required.
            User user = new User();
            FilenameProfileReader reader = new FilenameProfileReader(epfFile);
            user.login(reader, profilePass);
            System.out.println("Login into user profile successful");
            encryptionCertificate = user.getEncryptionCertificate();
            System.out.println("Recipient's encryption certificate retrieved from user profile " + epfFile);
        } else {
            // The recipient's encryption certificate was provided
            String encryptionCertificateFilename = args[2];

            // The final argument is always the output file.
            outFile = args[3];

            // Read the certificate file and obtain the encryption certificate
            FileInputStream fis = new FileInputStream(new File(encryptionCertificateFilename));
            encryptionCertificate = new X509Certificate(fis);
            System.out.println("Recipient's encryption certificate obtained from file " + encryptionCertificateFilename);
        }


        // Generate the encrypted contents (in the ESP 9.3 format)
        byte[] output = encodeEnvelopedDataStreamFromFile(filename, password, passwordCopy, encryptionCertificate);


        // Write the data (which is encrypted) to the output file.
        // Add a pp7m extension, if the provided filename doesn't contain an extension
        // (This is what ESP uses.)
        if (!outFile.endsWith(".pp7m")) outFile = outFile + ".pp7m";
        FileOutputStream fos = new FileOutputStream(outFile);
        fos.write(output);
        fos.close();
        System.out.println("\nEncrypted file created: " + outFile);
    }


    /**
     * Code that does the work of decoding the ESP generated message
     *
     * @param fileData the InputStream containing the original (plaintext) file
     * @param pass The Password protecting the file
     * @param encryptionCertificate The recipient's Encryption Certificate
     * @throws Exception
     */
    private static byte[] encodeEnvelopedDataStreamFromFile(String fileData, SecureStringBuffer pass, SecureStringBuffer passCopy, X509Certificate encryptionCertificate) throws Exception {

        // ESP first Mimes the File data
        FileInputStream fis = new FileInputStream(fileData);
        InputStream is = createMimeLayerFromInputStream(fileData,"text/plain", fis);

        // Then ESP Compresses the Data
        CompressedDataStream compressed = new CompressedDataStream(is,CompressionProvider.zlib_compress,CompressedDataStream.IMPLICIT);

        // Then wrap in the content info Stream
        ContentInfoStream cis = new ContentInfoStream(compressed);
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        cis.writeTo(bos);

        InputStream cInput = new ByteArrayInputStream(bos.toByteArray());

        // Now wrap the CompressedData
        // See RFC 2633 (S/MIME v3 Message Specification) for details on the content type
        is = createMimeLayerFromInputStream(fileData,"application/pkcs7-mime; name=\"smime.p7m\"; smime-type=\"compressed-data\"",cInput);

        // Generate the SecretKey
        byte[] authdata = createAuthenticatedData(com.entrust.toolkit.asn1.structures.AlgorithmID.MacAlgs.id_hmacWithSHA256,AlgorithmID.sha512, is, pass, encryptionCertificate);
        ByteArrayInputStream bis = new ByteArrayInputStream(authdata);

        // Now wrap the AuthenticatedData in the mime layers
        // See RFC 2633 (S/MIME v3 Message Specification) for details on the content type
        is = createMimeLayerFromInputStream(fileData,"application/pkcs7-mime; name=\"smime.p7m\"; smime-type=\"authenticated-data\"",bis);

        return createEnvelopedDataStream(is, com.entrust.toolkit.asn1.structures.AlgorithmID.CipherAlgs.id_aes256_CBC, passCopy, encryptionCertificate);
    }


    /**
     * Create the EnvelopedDataStream (The encrypted layer in the ESP message).
     * @param is The InputStream containing the data to encrypt
     * @param id The AlgorithmIdentifier of the Algorithm to use for the symmetric encryption
     * @param pass The Password to use for the PasswordRecipientInfo
     * @param encryptionCertificate The recipient's Encryption Certificate
     * @return A byte array representing the ASN1 Der encoding of the EnvelopedDataStream
     * @throws CMSException
     * @throws CodingException
     * @throws IOException
     * @throws GeneralSecurityException
     */
    private static byte[] createEnvelopedDataStream(InputStream is, com.entrust.toolkit.asn1.structures.AlgorithmID id, SecureStringBuffer pass, X509Certificate encryptionCertificate) throws CMSException, CodingException, IOException, GeneralSecurityException  {
        // Create the SMimeEncrypt object.  This is basically an EnvelopedDataStream with an extra object
        // Identifier in front of it saying it is an enveloped data stream
        SMimeEncrypted enveloped = new SMimeEncrypted(is, id.toIAIKAlgorithmID(),id.getSymmetricKeySize());

        // EnvelopedDataStream enveloped = new EnvelopedDataStream(is, id.toIAIKAlgorithmID(),id.getSymmetricKeySize());

        // Get the recipients
        RecipientInfo trans = getKeyTransRecipient(encryptionCertificate);
        RecipientInfo passrecip = getPasswordRecipient(pass);
        // add the recipient information:
        enveloped.addRecipientInfo(trans);
        enveloped.addRecipientInfo(passrecip);

        // write the EnvelopedData to a stream thereby performing the
        // content encryption:
        ByteArrayOutputStream encryptedData = new ByteArrayOutputStream();
        enveloped.writeTo(encryptedData);

        System.out.println("\nEncrypted Data: " + Util.toString(encryptedData.toByteArray()));

        return encryptedData.toByteArray();
    }

    /**
     * Create a Mime Layer from an InputStream.  The Name is the name used in the mime layer
     * @param fileName the fileName to add in the Mime header
     * @param contentType The String holding the content type parameters
     * @param source The InputStream which is being wrapped by this Mime layer
     * @return An inputSTream containing the
     * @throws MessagingException
     * @throws IOException
     */
    private static InputStream createMimeLayerFromInputStream(String fileName, String contentType,  InputStream source) throws MessagingException, IOException {
        EncodeCustomMimeHeaders message = new EncodeCustomMimeHeaders(null);
        message.removeHeader("MIME-Version");

        message.addHeader("Content-Disposition", "attachment; filename=\"" + fileName + "\"");

        ByteArrayDataSource bds = new ByteArrayDataSource(source,contentType);
        DataHandler handler = new DataHandler(bds);
        message.setDataHandler(handler);

        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        // message.addHeader("Content-Transfer-Encoding","binary");
        message.writeTo(bos);

        return new ByteArrayInputStream(bos.toByteArray());
    }



    /**
     * Create an AuthenticatedDataStream using an AlgoirthmID and Password
     * @param emacAlgorithm
     * @param password
     * @return the byteArray Representation of an AuthenticationDataStream
     */
    private static byte[] createAuthenticatedData(com.entrust.toolkit.asn1.structures.AlgorithmID emacAlgorithm, AlgorithmID digestAlgorithm, InputStream is, SecureStringBuffer password, X509Certificate encryptionCertificate) {

        // the content type
        ObjectID contentType = ObjectID.cms_data;

        // the mac algorithm to be used
        AlgorithmID macAlgorithm = emacAlgorithm.toIAIKAlgorithmID();
        // the length of the mac key to be generated
        int macKeyLength = 256;
        // we do not need mac algorithm parameters
        AlgorithmParameterSpec macParams = null;
        // we want to include authenticated attributes and therefore need a digest algorithm
        //AlgorithmID digestAlgorithm = (AlgorithmID)AlgorithmID.sha1.clone();
        // the transmission mode (either AuthenticatedDataStream.IMPLICIT or AuthenticatedDataStream.EXPLICIT)
        int mode = AuthenticatedData.IMPLICIT;
        // create the AuthenticatedDataStream object:
        try {
            AuthenticatedDataStream authenticatedData = new AuthenticatedDataStream(contentType,
                    is,  macAlgorithm,macKeyLength,macParams,null,mode);
            // SMimeAuthenticatedData authenticatedData = new SMimeAuthenticatedData(contentType,
            //        is,  macAlgorithm,macKeyLength,macParams,null,mode);
            ByteArrayOutputStream bos = new ByteArrayOutputStream();
            RecipientInfo passrecipient = getPasswordRecipient(password);
            RecipientInfo transrecipient = getKeyTransRecipient(encryptionCertificate);
            authenticatedData.setRecipientInfos(new RecipientInfo[]{passrecipient,transrecipient});

            // Add ContentType header information in the ASN1 object
            ContentInfoStream cis = new ContentInfoStream(authenticatedData);
            cis.writeTo(bos);
            authenticatedData.writeTo(bos);

            System.out.println("\nOutput Authenticated data: " + Util.toString(bos.toByteArray()));
            System.out.println("\nMAC: " + Util.toString(authenticatedData.getMac()));
            return bos.toByteArray();
        } catch (CMSException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        } catch (IOException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        } catch (NoSuchAlgorithmException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        } catch (InvalidKeySpecException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        } catch (InvalidAlgorithmParameterException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        } catch (InvalidParameterSpecException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        } catch (CodingException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        } catch (GeneralSecurityException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }
        return null;
    }

    /**
     * Helper method to create the TransRecipientInfo recipient for Certificate based decryption of
     * the secret key.  This recipient always uses rsaEncryption to encrypt the secret.
     * Note:  This just sets up the recipient, it does not encrypt the secret.
     *
     * @param encryptionCertificate The recipient's Encryption Certificate
     * @return the Key Transport Recipient info
     */
    private static RecipientInfo getKeyTransRecipient(X509Certificate encryptionCertificate) {
        AlgorithmID keyEncryptionAlgorithmId = AlgorithmID.rsaEncryption;

        // The recipient's encryption certificate is needed to create the KeyTransRecipientInfo.
        KeyTransRecipientInfo trans = new KeyTransRecipientInfo(encryptionCertificate, keyEncryptionAlgorithmId);

        return trans;
    }

    /**
     * Helper method to create the PasswordRecipientInfo for password based encryption/decryption of
     * the secret key.  This recipient always uses rsaEncryption to encrypt the secret.
     * Note:  This just sets up the recipient, it does not encrypt the secret.
     *
     * Get a Password Recipient as a recipient
     * @param pass The password for the Password Recipient which is used to derive the key for
     * encryption
     * @return The PasswordRecipient
     * @throws CodingException
     * @throws CMSException
     * @throws GeneralSecurityException
     */
    private static RecipientInfo getPasswordRecipient(SecureStringBuffer pass) throws CodingException, CMSException, GeneralSecurityException {
        // For Security, generate a random salt
        byte[] salt = new byte[8];
        SecureRandom random = Entrust.getDefaultSecureRandomInstance();
        random.nextBytes(salt);

        // AlgorithmID keyEncryptionAlgorithmId = AlgorithmID.des_EDE3_CBC;
        AlgorithmID keyEncryptionAlgorithmId = AlgorithmID.aes_256_CBC;
        PasswordRecipientInfo pwri = new PasswordRecipientInfo(pass, AlgorithmID.pbkdf2,
                new PBEKeyAndParameterSpec(pass.toByteArray(), salt, 10000, (keyEncryptionAlgorithmId.toEntrustAlgorithmID().getSymmetricKeySize() / 8)),
                keyEncryptionAlgorithmId, null);
        return pwri;
    }


    /**
     * An example of how to create a Mime Layer from an InputStream.  The Name is the name used in the mime layer
     * (Note, this method is not used by the rest of the sample, but is provided for reference.)
     *
     * @param fileName the fileName to add in the Mime header
     * @param contentType The String holding the content type parameters
     * @param source The InputStream which is being wrapped by this Mime layer
     * @return An inputStream containing the Mime layer
     * @throws MessagingException
     * @throws IOException
     */
    private static InputStream createMimeLayerStreamFromInputStream(String fileName, String contentType,  InputStream source) throws MessagingException, IOException {
        EncodeCustomMimeHeaders message = new EncodeCustomMimeHeaders(null);
        message.removeHeader("MIME-Version");

        message.addHeader("Content-Disposition", "attachment; filename=\"" + fileName + "\"");

        ByteArrayDataSource bds = new ByteArrayDataSource(source,contentType);
        DataHandler handler = new DataHandler(bds);
        message.setDataHandler(handler);

        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        // message.addHeader("Content-Transfer-Encoding","binary");
        message.writeTo(bos);

        return new ByteArrayInputStream(bos.toByteArray());
    }

    /**
     * This is for debugging.  It allows writing out the contents of the inputStream to a File, and
     * returns the InputStream so it can continue to be read.  It is useful to peel off the mime
     * layers.
     * (Note, this method is not used by the rest of the sample, but is provided for reference.)
     *
     * @param stream The InputStreams whose contents will be written
     * @param message The System.out textual description
     * @param filename The Filename (optional).  If not null, then bytes will also be written to a file.
     * @return The InputStream that can be read again.
     * @throws IOException if an error occurs
     */
    private static InputStream writeOutput(InputStream stream, String message, String filename) throws IOException {

        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        byte[] buf = new byte[1024];
        int r;
        while ((r = stream.read(buf)) > 0) {
            // do something useful
            bos.write(buf, 0, r);
        }
        stream.close();
        bos.flush(); bos.close();
        System.out.println(message + ": " + Util.toString(bos.toByteArray()));
        if (filename != null) {
            FileOutputStream fos = new FileOutputStream(filename);
            fos.write(bos.toByteArray());
        }
        return new ByteArrayInputStream(bos.toByteArray());
    }
}