//===========================================================================
//
// Copyright (c)  2003-2012 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.crypto;

import iaik.utils.Util;

import java.security.AlgorithmParameters;

import javax.crypto.Cipher;
import javax.crypto.KeyGenerator;
import javax.crypto.SecretKey;

import com.entrust.toolkit.security.provider.Initializer;

/**
 * This example demonstrates how to use Entrust's 'AES' implementation of a Cipher 
 * algorithm from the Java Cryptography Architecture (JCA).
 * 
 * <p>
 * It shows how to generate an AES key, and then how to use this key in 
 * conjunction with an AES Cipher to encrypt and decrypt data.
 * </p>
 * 
 * <p>
 * When using JRE 1.7 it is also possible to use the GCM mode.  To use GCM mode,
 * comment out the CBC section and uncomment the GCM section.
 * </p>
 * 
 * <p>
 * The JCA refers to a framework for accessing and developing cryptographic 
 * functionality for the Java platform.  For further information on how the JCA
 * is used, please refer to Java's 
 * <a href=http://download.oracle.com/javase/1.5.0/docs/guide/security/index.html>
 * Security Documentation</a> documentation.
 * </p>
 */
public class CipherExampleAES {

    /** Test data. */
    private static final byte[] TEST_DATA = new byte[] { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 };

    /**
     * The main program.
     * 
     * @param args
     *      program arguments
     */
    public static void main(String[] args) {

        // Ensure that the arguments are valid
        int size = args.length;
        int strength = 128;
        if (size == 1) {
            try {
                strength = Integer.parseInt(args[0]);
            } catch (NumberFormatException e) {
                // First argument does not represent an integer strength.. ignore
                // and use default strength
            }
        } else if (size > 1) {
            System.out.println("Usage: CipherExampleAES [<strength>]");
            return;
        }

        // Ensure that Entrust's providers have been installed
        System.out.print("Installing providers... ");
        Initializer.getInstance().setProviders(Initializer.MODE_NORMAL);
        System.out.println("DONE");
        System.out.println();

        // Comment out this block when using GCM mode
        try {
            // Request an instance of the AES cipher and key generation 
            // algorithms from the Entrust provider.  The requested AES cipher
            // will operate in CBC mode, and use PKCS5 padding.
            Cipher cipher = Cipher.getInstance("AES/CBC/PKCS5Padding", "Entrust");
            KeyGenerator keyGen = KeyGenerator.getInstance("AES", "Entrust");

            // Generate a key that can be used with the AES cipher
            System.out.print("Generating key... ");
            keyGen.init(strength);
            SecretKey key = keyGen.generateKey();
            System.out.println("DONE");
            System.out.println();

            // Initialize the cipher algorithm for encryption and encrypt
            // some data.  Symmetric ciphers used in CBC mode all require an
            // initialization vector.  If not provided, it is automatically
            // generated by the cipher.  In order for the decryption to 
            // succeed, this same initialization vector must be supplied.
            System.out.print("Encrypting data... ");
            cipher.init(Cipher.ENCRYPT_MODE, key);
            byte[] ciphertext = cipher.doFinal(TEST_DATA);
            System.out.println("DONE");
            AlgorithmParameters params = cipher.getParameters();
            System.out.println("input plaintext:   " + Util.toString(TEST_DATA));
            System.out.println("output ciphertext: " + Util.toString(ciphertext));
            System.out.println(params);

            // Initialize the cipher algorithm for decryption and decrypt the
            // ciphertext.  The same parameters that were used during encryption
            // are also used during decryption (initialization vector)
            System.out.print("Decrypting data... ");
            cipher.init(Cipher.DECRYPT_MODE, key, params);
            byte[] plaintext = cipher.doFinal(ciphertext);
            System.out.println("DONE");
            System.out.println("input ciphertext: " + Util.toString(ciphertext));
            System.out.println("output plaintext: " + Util.toString(plaintext));
            System.out.println(params);
        } catch (Exception e) {
            e.printStackTrace();
        }
        
        // Uncomment this section to use GCM mode with JRE 1.7 or later.
//        try {
//            // Request an instance of the AES cipher and key generation 
//            // algorithms from the Entrust provider.  The requested AES cipher
//            // will operate in GCM mode, and use no padding.
//            Cipher cipher = Cipher.getInstance("AES/GCM/NoPadding", "Entrust");
//            KeyGenerator keyGen = KeyGenerator.getInstance("AES", "Entrust");
//
//            // Generate a key that can be used with the AES cipher
//            System.out.print("Generating key... ");
//            keyGen.init(strength);
//            SecretKey key = keyGen.generateKey();
//            System.out.println("DONE");
//            System.out.println();
//            
//            byte[] aadData = new String("Additional Authentication Data").getBytes();
//
//            // Initialize the cipher algorithm for encryption and encrypt
//            // some data.  Symmetric ciphers used in GCM mode all require an
//            // initialization vector and authentication tag length (in bits).  
//            // If not provided, they are automatically generated by the cipher.
//            // In order for the decryption to succeed, the same initialization 
//            // vector and authentication tag length must be supplied.  The 
//            // authentication tag data will be appended to the ciphertext.
//            System.out.print("Encrypting data... ");
//            cipher.init(Cipher.ENCRYPT_MODE, key);
//            
//            // Provide the Additional Authentication Data first 
//            cipher.updateAAD(aadData);
//            byte[] ciphertext = cipher.doFinal(TEST_DATA);
//            System.out.println("DONE");
//            AlgorithmParameters params = cipher.getParameters();
//            System.out.println("input plaintext:   " + Util.toString(TEST_DATA));
//            System.out.println("output ciphertext: " + Util.toString(ciphertext));
//            System.out.println("IV and tLen parameters: " + params);
//            
//            // Initialize the cipher algorithm for decryption and decrypt the
//            // ciphertext.  The same parameters that were used during encryption
//            // are also used during decryption (initialization vector, 
//            // authentication tag length)
//            System.out.print("Decrypting data... ");
//            cipher.init(Cipher.DECRYPT_MODE, key, params);
//
//            // Provide the Additional Authentication Data first 
//            cipher.updateAAD(aadData);
//            byte[] plaintext = cipher.doFinal(ciphertext);
//            System.out.println("DONE");
//            System.out.println("input ciphertext: " + Util.toString(ciphertext));
//            System.out.println("output plaintext: " + Util.toString(plaintext));
//            System.out.println("IV and tLen parameters: " + params);
//        } catch (Exception e) {
//            e.printStackTrace();
//        }
    }
}