//===========================================================================
//
// Copyright (c)  2003-2010 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.crypto;

import iaik.utils.Util;

import java.security.MessageDigest;

import com.entrust.toolkit.security.provider.Initializer;

/**
 * This example demonstrates how to use Entrust's 'SHA1' implementation of a 
 * Message Digest algorithm from the Java Cryptography Architecture (JCA).
 * 
 * <p>
 * It shows how to use the SHA1 message digest algorithm to create a message
 * digest (hash).
 * </p>
 * 
 * <p>
 * The JCA refers to a framework for accessing and developing cryptographic 
 * functionality for the Java platform.  For further information on how the JCA
 * is used, please refer to Java's 
 * <a href=http://download.oracle.com/javase/1.5.0/docs/guide/security/index.html>
 * Security Documentation</a> documentation.
 * </p>
 */
public class MessageDigestExampleSHA1 {

    /** Test data. */
    private static final byte[] TEST_DATA = new byte[] { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 };

    /**
     * The main program.
     * <p></p>
     * 
     * @param args
     *      not used
     */
    public static void main(String[] args) {

        // Ensure that Entrust's providers have been installed
        System.out.print("Installing providers... ");
        Initializer.getInstance().setProviders(Initializer.MODE_NORMAL);
        System.out.println("DONE");
        System.out.println();

        try {
            // Request an instance of the SHA1 message digest algorithm from the 
            // Entrust provider
            MessageDigest messageDigest = MessageDigest.getInstance("SHA1", "Entrust");

            // Calculate the digest of some data
            System.out.print("Calculating digest... ");
            messageDigest.update(TEST_DATA);
            byte[] digest1 = messageDigest.digest();
            System.out.println("DONE");
            System.out.println("digest: " + Util.toString(digest1));
            System.out.println();

            // Re-calculate the mac over the same data
            System.out.print("Re-calculating digest... ");
            byte[] digest2 = messageDigest.digest(TEST_DATA);
            System.out.println("DONE");
            System.out.println("digest: " + Util.toString(digest2));
        } catch (Exception e) {
            e.printStackTrace();
        }
    }
}