//===========================================================================
//
// Copyright (c)  2003-2010 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.crypto;

import iaik.utils.Util;

import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.Signature;

import com.entrust.toolkit.security.provider.Initializer;

/**
 * This example demonstrates how to use Entrust's 'RSA' implementation of a 
 * Signature algorithm from the Java Cryptography Architecture (JCA).
 * 
 * <p>
 * It shows how to generate an RSA public/private key pair, and then how to use
 * this key pair in conjunction with an RSA Signature to sign data and verify
 * signatures.
 * </p>
 * 
 * <p>
 * The JCA refers to a framework for accessing and developing cryptographic 
 * functionality for the Java platform.  For further information on how the JCA
 * is used, please refer to Java's 
 * <a href=http://download.oracle.com/javase/1.5.0/docs/guide/security/index.html>
 * Security Documentation</a> documentation.
 * </p>
 */
public class SignatureExampleRSA {

    /** Test data. */
    private static final byte[] TEST_DATA = new byte[] { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 };

    /**
     * The main program.
     * <p></p>
     * 
     * @param args
     *      program arguments
     */
    public static void main(String[] args) {

        // Ensure that the arguments are valid
        int size = args.length;
        int strength = 1024;
        if (size == 1) {
            try {
                strength = Integer.parseInt(args[0]);
            } catch (NumberFormatException e) {
                // First argument does not represent an integer strength.. ignore
                // and use default strength
            }
        } else if (size > 1) {
            System.out.println("Usage: SignatureExampleRSA [<strength>]");
            return;
        }

        // Ensure that Entrust's providers have been installed
        System.out.print("Installing providers... ");
        Initializer.getInstance().setProviders(Initializer.MODE_NORMAL);
        System.out.println("DONE");
        System.out.println();

        try {
            // Request an instance of the RSA signature and key pair generation 
            // algorithms from the Entrust provider.  The requested RSA signature
            // algorithm will use the SHA message digest algorithm. 
            Signature signer = Signature.getInstance("SHA/RSA", "Entrust");
            KeyPairGenerator keyGen = KeyPairGenerator.getInstance("RSA", "Entrust");

            // Generate a public/private key pair that can be used with the 
            // RSA signature algorithm
            System.out.print("Generating key pair... ");
            keyGen.initialize(strength);
            KeyPair keyPair = keyGen.generateKeyPair();
            System.out.println("DONE");
            System.out.println();
            
            // Initialize the signature algorithm for signing and sign some data
            System.out.print("Signing data... ");
            signer.initSign(keyPair.getPrivate());
            signer.update(TEST_DATA);          
            byte[] signature = signer.sign();
            System.out.println("DONE");
            System.out.println("input plaintext:  " + Util.toString(TEST_DATA));            
            System.out.println("output signature: " + Util.toString(signature));
            System.out.println();

            // Initialize the signature algorithm for verification and verify the
            // signature against the signed data
            System.out.print("Verifying signature against signed data... ");
            signer.initVerify(keyPair.getPublic());
            signer.update(TEST_DATA);
            boolean verified = signer.verify(signature);
            System.out.println("DONE");            
            System.out.println("signature verified: " + verified);
            System.out.println();
            
            // Now attempt verifying the signature against the signed data after
            // only a single byte has been modified. This should fail.
            System.out.print("Verifying signature against modified signed data... ");
            TEST_DATA[0] = 1;
            signer.update(TEST_DATA);
            verified = signer.verify(signature);
            System.out.println("DONE");            
            System.out.println("signature verified: " + verified);            
        } catch (Exception e) {
            e.printStackTrace();
        }
    }
}