//===========================================================================
//
// Copyright (c)  2005-2010 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.pkcs10;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;

import iaik.pkcs.pkcs10.CertificateRequest;
import iaik.x509.X509Certificate;

import com.entrust.toolkit.User;
import com.entrust.toolkit.credentials.CredentialReader;
import com.entrust.toolkit.credentials.EntrustP10CertReqInfo;
import com.entrust.toolkit.credentials.EntrustP10CertRetriever;
import com.entrust.toolkit.credentials.FilenameProfileReader;
import com.entrust.toolkit.security.provider.Initializer;
import com.entrust.toolkit.util.AuthorizationCode;
import com.entrust.toolkit.util.ManagerTransport;
import com.entrust.toolkit.util.SecureStringBuffer;

/**
 * This example demonstrates a PKCS 10 CertificateRequest can be used to 
 * retrieve a certificate from an Entrust Authority Security Manager (EASM).
 * 
 * <p>PKCS 10 certificate retrieval from a EASM requires special configuration
 * of the EASM; refer to the class documentation of 
 * <code>EntrustP10CertRetriever</code> for more details.
 * 
 * <p>To run this sample, you need an Entrust PKI, the digital identity and 
 * password of a PKI administrator, a reference number and authorization code 
 * issued from the PKI, a certificate definition name that the request 
 * correspond to, the name of a file containing a DER encoded PKCS 10 
 * certificate request, and the name of a file where the certificate will be 
 * written to in DER encoded X.509 format.</p>
 *  
 * <p>For example, from the examples directory:</p>
 * <pre>
 * javac -sourcepath source -d classes -classpath ../lib/enttoolkit.jar source/com/entrust/toolkit/examples/pkcs10/RetrieveCertificate.java
 * java -classpath ..\lib\enttoolkit.jar;classes com.entrust.toolkit.examples.pkcs10.RetrieveCertificate
 * 127.0.0.1 829 Admin.epf Password! 51997123 FAKE-AUTH-CODE certReq.p10 Verification MyUser.p10 MyUser.cer
 * </pre>
 */
public class RetrieveCertificate {
    public static void main(String[] args) {

        // Ensure that the required command-line arguments were provided
        if (args.length != 9) {
            System.out.println("\nRetrieveCertificate - Retrieve a certificate using PKCS 10");
            System.out
                .println("\nUsage:  java RetrieveCertificate <EASM IP> <EASM port> <Admin EPF> <Admin password> <refnum> <authcode> <certdefn name> <P10 certReq input file> <X.509 cert output file>");
            return;
        }

        // Ensure that Entrust's providers have been installed
        System.out.print("Installing JCA providers... ");
        Initializer.getInstance().setProviders(Initializer.MODE_NORMAL);
        System.out.println("DONE");
        
        // Retrieve the certificate
        System.out.print("Retrieving certificate... ");
        try {
            // Extract all of the command-line arguments.
            int index = 0;
            String managerIP = args[index++];
            int managerPort = Integer.parseInt(args[index++]);
            String adminEPF = args[index++];
            SecureStringBuffer adminPassword = new SecureStringBuffer(new StringBuffer(args[index++]));
            SecureStringBuffer secureRefNum = new SecureStringBuffer(new StringBuffer(args[index++]));
            AuthorizationCode secureAuthCode = new AuthorizationCode(new StringBuffer(args[index++]));
            String certDefnName = args[index++];
            File p10File = new File(args[index++]);
            File certFile = new File(args[index++]);

            // Read in P10 request
            FileInputStream fis = new FileInputStream(p10File);
            CertificateRequest p10CertReq = new CertificateRequest(fis);
            fis.close();
            
            // Login Administrator
            CredentialReader adminCR = new FilenameProfileReader(adminEPF);
            User adminUser = new User();
            adminUser.login(adminCR, adminPassword);

            // Request the certificate from the EASM
            ManagerTransport managerTransport = new ManagerTransport(managerIP, managerPort);
            EntrustP10CertRetriever certRetriever = new EntrustP10CertRetriever(managerTransport, adminUser);
            EntrustP10CertReqInfo certReqInfo = new EntrustP10CertReqInfo(p10CertReq, certDefnName);
            certRetriever.retrieveUserCert(secureRefNum, secureAuthCode, certReqInfo);
            System.out.println("DONE");

            // Output X.509 certificate
            System.out.println();
            X509Certificate cert = certReqInfo.getUserCertificate();
            FileOutputStream fos = new FileOutputStream(certFile);
            fos.write(cert.toByteArray());
            fos.close();
            System.out.println("\nCertificate in X.509 format written to: " + certFile.getAbsolutePath());
            System.out.println("\nReceived Certificate:\n");
            System.out.println(cert.toString(true));
        } catch (Exception e) {
            // An error occurred
            System.out.println("Failed");
            e.printStackTrace();
        }
    }
}