//===========================================================================
//
// Copyright (c)  2003-2010 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.pkcs8;

import iaik.asn1.DerCoder;
import iaik.asn1.structures.AlgorithmID;
import iaik.pkcs.pkcs8.EncryptedPrivateKeyInfo;
import iaik.utils.Util;

import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.PrivateKey;

import com.entrust.toolkit.security.provider.Initializer;

/**
 * This example demonstrates how to protect a private key using PKCS #8 and
 * the EncryptedPrivateKeyInfo ASN1 structure.
 */
public class EncryptedPrivateKeyExample {

    /**
     * The main program. Exactly one command line argument is required, the
     * password to encrypt and decrypt the private key.
     * 
     * @param args
     *      program arguments. 
     */
    public static void main(String[] args) {

        // Ensure that the arguments are valid
        if (args.length != 1) {
            System.out.println("Usage: EncryptedPrivateKeyExample <password>");
            return;
        }
        String password = args[0];

        // Ensure that Entrust's providers have been installed
        System.out.print("Installing providers... ");
        Initializer.getInstance().setProviders(Initializer.MODE_NORMAL);
        System.out.println("DONE");
        System.out.println();

        try {
            // Generate an RSA public/private key pair
            KeyPairGenerator keyGen = KeyPairGenerator.getInstance("RSA", "Entrust");
            System.out.print("Generating key-pair... ");
            keyGen.initialize(1024);
            KeyPair keyPair = keyGen.generateKeyPair();
            System.out.println("DONE");
            System.out.println("DER encoded PrivateKeyInfo structure:");
            System.out.println(Util.toString(keyPair.getPrivate().getEncoded()));
            System.out.println();

            // Encrypt the private key in an EncryptedPrivateKeyInfo structure,
            // then DER encode that structure.  This DER encoding can then be safely 
            // stored or transported.
            EncryptedPrivateKeyInfo encryptedPrivateKeyInfo = new EncryptedPrivateKeyInfo(keyPair.getPrivate());
            System.out.print("Encrypting/Encoding the private key... ");
            encryptedPrivateKeyInfo.encrypt(password, AlgorithmID.pbeWithMD5AndDES_CBC, null);
            byte[] derEncoding = DerCoder.encode(encryptedPrivateKeyInfo.toASN1Object());
            System.out.println("DONE");
            System.out.println("DER encoded EncryptedPrivateKeyInfo structure:");
            System.out.println(Util.toString(derEncoding));
            System.out.println();

            // DER decode the EncryptedPrivateKeyInfo structure and decrypt
            // the private key.  The private-key can now be used in any 
            // cryptographic algorithm.
            System.out.print("Decoding/Decrypting the private key... ");
            encryptedPrivateKeyInfo = new EncryptedPrivateKeyInfo(derEncoding);
            PrivateKey privateKey = encryptedPrivateKeyInfo.decrypt(password);
            System.out.println("DONE");
            System.out.println("DER encoded PrivateKeyInfo structure:");
            System.out.println(Util.toString(privateKey.getEncoded()));
        } catch (Exception e) {
            e.printStackTrace();
        }
    }
}