//===========================================================================
//
// Copyright (c)  2003-2010 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.pkits;

import iaik.asn1.ASN1Type;
import iaik.asn1.CodingException;
import iaik.asn1.ObjectID;
import iaik.asn1.structures.DistributionPoint;
import iaik.asn1.structures.GeneralName;
import iaik.asn1.structures.GeneralNames;
import iaik.asn1.structures.Name;
import iaik.asn1.structures.RDN;
import iaik.x509.V3Extension;
import iaik.x509.X509CRL;
import iaik.x509.X509Certificate;
import iaik.x509.X509ExtensionException;
import iaik.x509.extensions.CRLDistributionPoints;
import iaik.x509.extensions.IssuingDistributionPoint;

import java.io.FileInputStream;
import java.io.IOException;
import java.security.cert.CRLException;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.List;

import com.entrust.toolkit.exceptions.UserFailureException;
import com.entrust.toolkit.security.provider.Initializer;
import com.entrust.toolkit.x509.CertVerifier;
import com.entrust.toolkit.x509.certstore.CertificateGraph;
import com.entrust.toolkit.x509.policies.ClientSettings;
import com.entrust.toolkit.x509.revocation.ArchiveCRLCache;
import com.entrust.toolkit.x509.revocation.DistPointAndCRL;

import com.entrust.toolkit.examples.pkits.PKITSTestConfiguration;

/**
 * A class that runs an individual PKITS test. Instances are created by
 * providing a test configuration, then the test is run by calling the
 * <code>runTest</code> method. The result of running a test will be either a
 * path validation success, or an exception being thrown. This class does not
 * decide if the result actually means the test passed or failed.
 * <p>
 * This class makes use of the toolkit's certificate validation APIs. The test
 * is run offline, so that anyone can download the test suite and run it without
 * setting up an LDAP Directory.
 * </p>
 * 
 * @see PKITSTestRunner
 */
public class PKITSTest {
	/*
	 * Make sure that the toolkit has been properly initialized.
	 */
	static {
		Initializer.getInstance().setProviders(Initializer.MODE_NORMAL);
	}

	/** Extra verbose debugging flag. */
	private boolean m_debug = true;

	/** The test configuration. */
	private PKITSTestConfiguration m_config;
	
	/**
	   * Test Mode online or offline
	   * In online mode we get the crls and intermediate certs from a directory
	   * In offline mode we use local data
	   */
	  static boolean offline = true;


	/**
	 * Create a PKITSTest object with the given configuration.
	 * 
	 * @param configuration
	 *            the test configuration. Must have been parsed externally, and
	 *            should be valid.
	 * 
	 * @param debug
	 *            <code>true</code> to display extra verbose information about
	 *            what is going on, <code>false</code> otherwise.
	 */
	public PKITSTest(PKITSTestConfiguration configuration, boolean debug) {
		if (configuration.isValidTest() == false) {
			throw new IllegalArgumentException(
					"Invalid configuration specified");
		}
		m_config = configuration;
		m_debug = debug;
	}


	  /**
	   * Run the test. This method loads all certificates and CRLs from the
	   * data files provided by NIST, adds them to the correct places in
	   * memory, sets the initial inputs to the path validation, and
	   * then runs the test. After the test is run, all the data that
	   * was added to memory is removed, so one test will not have any
	   * effect on subsequent tests.
	   *
	   * @throws IOException
	   *     If there is a problem reading from the data files, or as a
	   * way to indicate a problem with the test that is not related to
	   * the actual validation.
	   *
	   * @throws java.security.cert.CertificateException
	   *     If there is a problem parsing the certificates from the data files,
	   * or the certificates are outside their validity periods.
	   *
	   * @throws com.entrust.toolkit.exceptions.CertificationException
	   *     If the target certificate fails to validate.
	   *
	   * @throws CRLException
	   *     If there is a problem parsing the CRLs from the data files.
	   *
	   * @throws X509ExtensionException
	   *     If there is a problem reading the extensions from the certificates.
	   * This exception should not occur during a normal run.
	   */
	  public void runTest() throws
	      IOException,
	      java.security.cert.CertificateException,
	      com.entrust.toolkit.exceptions.CertificationException,
	      CRLException,
	      X509ExtensionException {
	    // The certificates that make up the two ends of the chain.
	    X509Certificate trustAnchorCert = null;
	    X509Certificate endCertificate = null;
	    
	    // Load the trust anchor certificate
	    FileInputStream certIn = new FileInputStream(m_config.
	                                                 getTrustRootFile());
	    trustAnchorCert = new X509Certificate(certIn);
	    certIn.close();
	    trustAnchorCert.checkValidity();

	    // Load the end entity certificate, this is what will be validate
	    certIn = new FileInputStream(m_config.getTargetCertificateFile());
	    endCertificate = new X509Certificate(certIn);
	    certIn.close();
	    endCertificate.checkValidity();

	    // The client settings object controls the initial inputs to
	    // path validation that relate to policy.
	    ClientSettings clientSettings = new ClientSettings();

	    // Set policy requirements.
	    Iterator it = m_config.getAcceptablePolicies().iterator();
	    try {
	      while (it.hasNext()) {
	        String policyOidString = (String) it.next();
	        ObjectID policyOid = new ObjectID(policyOidString);
	        clientSettings.addAcceptablePolicy(policyOid);
	      }
	      clientSettings.setInhibitPolicyMapping(m_config.
	                                             getInhibitPolicyMapping());
	      clientSettings.setInhibitAnyPolicy(m_config.
	                                         getInhibitAnyPolicy());
	      clientSettings.setRequireExplicitPolicy(m_config.
	                                              getRequireExplicitPolicy());
	    }
	    catch (UserFailureException e) {
	      // Will never happen, the ClientSettings are not associated with a User
	    }

	    // Create the object that will do certificate validation

	    CertVerifier certVerifier = null;

	    if (m_config.getDirectory() == null) {
	      // No Directory connection is provided, so all validation is performed
	      // offline using cached information.
	      certVerifier = new CertVerifier(trustAnchorCert, null, clientSettings);

	      // Make sure revocation information is required.  The CertVerifier
	      // has no directory connection which means, by default,
	      // it does not require revocation information.
	      certVerifier.getRevocationStore().requireCRL(true);

	      // Initialize for offline tests (load crls and certs into cache)
	      initOffline(m_config, certVerifier, endCertificate);
	    }
	    else {

	      certVerifier = new CertVerifier(trustAnchorCert,
	    		                   m_config.getDirectory(),
	                                      clientSettings);
	    }

	    try {
	      // This is the line we've been waiting for, and what all the
	      // previous code was leading up to.  Note that this call both
	      // builds and validates the chain, there is no need to
	      // do this separately.

	      certVerifier.validate(endCertificate);
	    }
	    finally {
	      // The certificate graph is static, clean it out now so that it won't
	      // affect any other tests that may be run.
	      CertificateGraph.Graph().clear();
	    }
	  }

	private static void initOffline(PKITSTestConfiguration testConfiguration,
			CertVerifier certVerifier, X509Certificate endCertificate)
			throws IOException, java.security.cert.CertificateException,
			CRLException, X509ExtensionException {
		// Lists of certificates and revocation lists to be used during
		// validation.
		List certPathCerts = new ArrayList();
		List extraCerts = new ArrayList();
		List revocationLists = new ArrayList();

		// Load any certificates on the cert path
		Iterator it = testConfiguration.getCertPathFiles().iterator();
		while (it.hasNext()) {
			String intermediateCertFile = (String) it.next();
			FileInputStream intermediateCertIn = new FileInputStream(
					intermediateCertFile);
			X509Certificate intermediateCert = new X509Certificate(
					intermediateCertIn);
			intermediateCertIn.close();
			intermediateCert.checkValidity();
			certPathCerts.add(intermediateCert);
		}

		// Load any extra certificates. These may be required
		// to validate revocation lists.
		it = testConfiguration.getExtraCertFiles().iterator();
		while (it.hasNext()) {
			String extraCertFile = (String) it.next();
			FileInputStream extraCertIn = new FileInputStream(extraCertFile);
			X509Certificate extraCert = new X509Certificate(extraCertIn);
			extraCertIn.close();
			extraCert.checkValidity();
			extraCerts.add(extraCert);
		}

		// Load any revocation lists
		it = testConfiguration.getRevocationListFiles().iterator();
		while (it.hasNext()) {
			String rlFile = (String) it.next();
			FileInputStream rlIn = new FileInputStream(rlFile);
			X509CRL revocationList = new X509CRL(rlIn);
			rlIn.close();
			revocationLists.add(revocationList);
		}

		// Add all intermediate certificates to the Graph object.
		it = certPathCerts.iterator();
		while (it.hasNext()) {
			X509Certificate cert = (X509Certificate) it.next();
			boolean wasAdded = CertificateGraph.Graph().addCertificate(cert);
			if (wasAdded != true) {
				// This is not strictly necessary, but the graph should be empty
				// for each test. Failure to add the certificate likely
				// indicates
				// a problem with the test configuration.
				throw new IOException("Certificate for "
						+ cert.getSubjectDN().getName()
						+ " not added to graph!");
			}
		}

		// Add all extra certificates to the Graph object.
		it = extraCerts.iterator();
		while (it.hasNext()) {
			X509Certificate cert = (X509Certificate) it.next();
			boolean wasAdded = CertificateGraph.Graph().addCertificate(cert);
			if (wasAdded != true) {
				// The same comment applies as for the intermediate
				// certificates.
				throw new IOException("Certificate for "
						+ cert.getSubjectDN().getName()
						+ " not added to graph!");
			}
		}

		// Load all of the CRLs in to cache so they can be used during the
		// validation
		ArchiveCRLCache myCrlCache = new ArchiveCRLCache(certVerifier);
		it = revocationLists.iterator();
		while (it.hasNext()) {
			X509CRL crl = (X509CRL) it.next();
			ASN1Type crlName = null;
			// This method of adding CRLs to the cache doesn't work in general,
			// e.g. it won't work with URIs. However, the PKITS tests are
			// based on directories, so it does work in this specific case.
			// This code also requires some knowledge of how CRLs are retrieved,
			// it roughly duplicates the way a revocation list for a given
			// certificate should be looked up.
			V3Extension v3Extension = crl
					.getExtension(IssuingDistributionPoint.oid);
			if (v3Extension != null) {
				// The CRL has an issuingDistributionPoint extension, which will
				// either say where it should be looked up from, or will
				// indicate that the CRL should be found at the issuer DN.
				IssuingDistributionPoint idp = (IssuingDistributionPoint) v3Extension;
				ASN1Type asn1DpName = idp.getDistributionPointName();
				if (asn1DpName == null) {
					crlName = (Name) crl.getIssuerDN();
				} else {
					crlName = getDirectoryName(asn1DpName, (Name) crl
							.getIssuerDN());
				}
			} else if (crl.getIssuerDN().equals(endCertificate.getIssuerDN())
					&& endCertificate.getExtension(CRLDistributionPoints.oid) != null) {
				// This is the case when the CRL has no issuing distribution
				// point, but the CRL and certificate issuers match, and the
				// end entity certificate has the CRLDistributionPoints
				// extensions. In this case, extract one of the distribution
				// point names and use that for the location of the CRL.
				CRLDistributionPoints crlDp = (CRLDistributionPoints) endCertificate
						.getExtension(CRLDistributionPoints.oid);
				// This code is a bit lazy, as it only uses the first
				// element of the enumerations. Also, some of the casts used
				// are not guaranteed to work in general, but it is
				// sufficient to run the PKITS tests.
				Enumeration dps = crlDp.getDistributionPoints();
				DistributionPoint dp = (DistributionPoint) dps.nextElement();
				Enumeration gns = ((GeneralNames) dp.getDistributionPointName())
						.getNames();
				GeneralName gn = (GeneralName) gns.nextElement();
				crlName = (Name) gn.getName();
			} else {
				// As a last resort, the revocation information will be
				// retrieved from the issuing CA's directory entry.
				crlName = (Name) crl.getIssuerDN();
			}
			DistPointAndCRL dpAndCrl = new DistPointAndCRL(crlName, crl);
			myCrlCache.addCRL(dpAndCrl);
		}

		// The following line copies the CRL entries from cache over to the
		// revocation store
		// used during certificate validation
		myCrlCache.initMemoryCache(certVerifier.getRevocationStore()
				.getMemoryCRLCache());
	}

	/**
	 * Returns the full directory name of the given DistributionPointName. The
	 * ASN.1 syntax if DistributionPointName is
	 * 
	 * <pre>
	 *  DistributionPointName ::= CHOICE {
	 *     fullName                [0] GeneralNames,
	 *     nameRelativeToCRLIssuer [1] RelativeDistinguishedName }
	 * </pre>
	 * 
	 * The way the full Directory name is formed depends on the choice. If the
	 * choice is fullName, the code merely extracts the first <code>Name</code>
	 * from the given distribution point name. If the choice is
	 * nameRelativeToCRLIssuer, a new <code>Name</code> is created by joining
	 * the nameRelativeToCRLIssuer and the issuer name.
	 * 
	 * @param asn1DpName
	 *            the choice of DistributionPointName, either GeneralNames or
	 *            RDN.
	 * 
	 * @param issuer
	 *            the name of the CRL issuer, only relevant if the choice of
	 *            syntax is nameRelativeToCRLIssuer.
	 * 
	 * @return the full Directory name of the given DistributionPointName.
	 */
	static Name getDirectoryName(ASN1Type asn1DpName, Name issuer) {
		Name name = null;

		if (asn1DpName instanceof GeneralNames) {
			// Look for the first Name.
			for (Enumeration n = ((GeneralNames) asn1DpName).getNames(); n
					.hasMoreElements();) {
				GeneralName generalName = (GeneralName) n.nextElement();
				if (generalName.getType() == GeneralName.directoryName) {
					name = (Name) generalName.getName();
					break;
				}
			}
		} else if (asn1DpName instanceof RDN) {
			try {
				// Make a copy of the issuer, if this is not done it may
				// modify the internal structure of wherever the name
				// came from.
				name = new Name(issuer.toASN1Object());
				name.addRDN((RDN) asn1DpName);
			} catch (CodingException e) {
				// This exception will never actually be thrown.
			}
		}
		return name;
	}
}
