//===========================================================================
//
// Copyright (c)  2003-2010 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.pkits;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import com.entrust.toolkit.x509.directory.JNDIDirectory;

/**
 * This class contains all the information needed to run an individual PKITS test.
 * The configuration needs to be read externally, and all properties in this class
 * set.
 * <p>
 * This class is very simple, all the interesting work is done elsewhere.
 * </p>
 */
public class PKITSTestConfiguration
{
    /**
     * Optional file prefix, used if the test files are installed in a
     * different location from the code.
     */
    private String m_filePrefix = null;

    private static final int PATH_SHOULD_NOT_VALIDATE = 0;
    private static final int PATH_SHOULD_VALIDATE = 1;
    private static final int PATH_VALIDATION_UNKNOWN = -1;

    /** The test title. */
    private String m_title;

    /** Whether or not the path is supposed to validate successfully. */
    private int m_shouldValidate = PATH_VALIDATION_UNKNOWN;

    /** A short description of the expected result. */
    private String m_description;

    /** An optional comment explaining the test result. */
    private String m_comment;


    /** The file containing the trust root certificate. */
    private String m_trustRootFile;

    /** The file containing the target certificate. */
    private String m_targetCertificateFile;

    /** Files that contain certificates to be used in the certification path. */
    private List m_certificationPathFiles = new ArrayList();

    /** Files that contain extra certificates to be used in the certificate validation. */
    private List m_extraCertificateFiles = new ArrayList();

    /** Files that contain revocation lists to be used in the certificate validation. */
    private List m_revocationListFiles = new ArrayList();

    /** A List of Strings containing acceptable policy oids. */
    private List m_acceptablePolicies = new ArrayList();

    // Booleans controlling initial inputs to the certificate validation.
    private boolean m_inhibitAnyPolicy = false;
    private boolean m_requireExplicitPolicy = false;
    private boolean m_inhibitPolicyMapping = false;

    /** Name (DN or IP) of the test directory. */
    private JNDIDirectory m_directory;

    /**
     * Construct a basic test configuration.
     *
     */
    public PKITSTestConfiguration()
    {
    }

    /**
     * Construct a test configuration, adding the given prefix to all file
     * locations.
     *
     * @param filePrefix
     *     A prefix to add to all file locations.
     */
    public PKITSTestConfiguration(String filePrefix)
    {
        m_filePrefix = filePrefix;
    }

    // Obvious getter/setter methods follow
    
    public String getTitle()
    {
        return m_title;
    }
    public void setTitle(String title)
    {
        m_title = title;
    }


    /**
     * Return whether or not the path is expected to validate.
     *
     * @return whether or not the path is expected to validate.
     */
    boolean shouldValidate()
    {
        if (m_shouldValidate == PATH_SHOULD_VALIDATE)
        {
            return true;
        }
        if (m_shouldValidate == PATH_SHOULD_NOT_VALIDATE)
        {
            return false;
        }
        else
        {
            // Should not get to this point, but fail if things do.
            throw new RuntimeException();
        }
    }

    /**
     * Set whether or not the path should validate
     */
    void setShouldValidate(boolean shouldValidate)
    {
        if (shouldValidate)
        {
            m_shouldValidate = PATH_SHOULD_VALIDATE;
        }
        else
        {
            m_shouldValidate = PATH_SHOULD_NOT_VALIDATE;
        }
    }
    
    public String getDescription()
    {
        return m_description;
    }
    public void setDescription(String description)
    {
        m_description = description;
    }

    public String getComment()
    {
        return m_comment;
    }
    public void setComment(String comment)
    {
        m_comment = comment;
    }


    public String getTrustRootFile()
    {
        return m_trustRootFile;
    }
    public void setTrustRootFile(String trustRootFile)
    {
        if (m_filePrefix == null)
        {
            m_trustRootFile = trustRootFile;
        }
        else
        {
            m_trustRootFile = m_filePrefix + trustRootFile;
        }
    }

    public String getTargetCertificateFile()
    {
        return m_targetCertificateFile;
    }
    public void setTargetCertificateFile(String targetCertificateFile)
    {
        if (m_filePrefix == null)
        {
            m_targetCertificateFile = targetCertificateFile;
        }
        else
        {
            m_targetCertificateFile = m_filePrefix + targetCertificateFile;
        }
    }

    public List getCertPathFiles()
    {
        return m_certificationPathFiles;
    }
    public void addCertPathFiles(List certPathFiles)
    {
        addListItems(m_certificationPathFiles, certPathFiles, true);
    }

    public List getExtraCertFiles()
    {
        return m_extraCertificateFiles;
    }
    public void addExtraCertFiles(List extraCertFiles)
    {
        addListItems(m_extraCertificateFiles, extraCertFiles, true);
    }

    public List getRevocationListFiles()
    {
        return m_revocationListFiles;
    }
    public void addRevocationListFiles(List revocationListFiles)
    {
        addListItems(m_revocationListFiles, revocationListFiles, true);
    }

    public List getAcceptablePolicies()
    {
        return m_acceptablePolicies;
    }
    public void addAcceptablePolicies(List policyOids)
    {
        addListItems(m_acceptablePolicies, policyOids, false);
    }

    /**
     * Private helper method to join two lists
     */
    private void addListItems(List addTo, List addFrom, boolean isFileType)
    {
        Iterator it = addFrom.iterator();
        while (it.hasNext())
        {
            if (isFileType && m_filePrefix != null)
            {
                String fileName = (String) it.next();
                addTo.add(m_filePrefix + fileName);
            }
            else
            {
                addTo.add(it.next());
            }
        }
    }

    // More obvious get/set methods
    public boolean getInhibitAnyPolicy()
    {
        return m_inhibitAnyPolicy;
    }
    public void setInhibitAnyPolicy(boolean inhibitAnyPolicy)
    {
        m_inhibitAnyPolicy = inhibitAnyPolicy;
    }

    public boolean getRequireExplicitPolicy()
    {
        return m_requireExplicitPolicy;
    }
    public void setRequireExplicitPolicy(boolean requireExplicitPolicy)
    {
        m_requireExplicitPolicy = requireExplicitPolicy;
    }

    public boolean getInhibitPolicyMapping()
    {
        return m_inhibitPolicyMapping;
    }
    public void setInhibitPolicyMapping(boolean inhibitPolicyMapping)
    {
        m_inhibitPolicyMapping = inhibitPolicyMapping;
    }

    //directory connection
    public JNDIDirectory getDirectory() {
        return m_directory;
    }
    public void setDirectory(String directoryName, int port) {
        m_directory = new JNDIDirectory (directoryName, port);
    }


    /**
     * Perform some sanity checks to make sure the test looks okay.
     *
     * @return whether or not this test is valid.
     */
    public boolean isValidTest()
    {
        if (m_shouldValidate == PATH_VALIDATION_UNKNOWN)
        {
            return false;
        }
        if (m_trustRootFile == null)
        {
            return false;
        }
        if (m_targetCertificateFile == null)
        {
            return false;
        }
        return true;
    }

    public String toString()
    {
        StringBuffer sb = new StringBuffer();
        sb.append("Should validate: " + m_shouldValidate + "\r\n");
        sb.append("Acceptable policies: ");
        Iterator it = m_acceptablePolicies.iterator();
        while (it.hasNext())
        {
            sb.append(it.next().toString() + " ");
        }
        sb.append("\r\nInhibit policy mapping: " + m_inhibitPolicyMapping + "\r\n");
        sb.append("Require explicit policy: " + m_requireExplicitPolicy + "\r\n");
        sb.append("Inhibit any policy: " + m_inhibitAnyPolicy + "\r\n");
        sb.append("Target Certificate file: " + m_targetCertificateFile + "\r\n");
        sb.append("Trust root file: " + m_trustRootFile + "\r\n");
        sb.append("Cert path files: ");
        it = m_certificationPathFiles.iterator();
        while (it.hasNext())
        {
            sb.append(it.next().toString() + " ");
        }
        sb.append("\r\nExtra cert files: ");
        it = m_extraCertificateFiles.iterator();
        while (it.hasNext())
        {
            sb.append(it.next().toString() + " ");
        }
        sb.append("\r\nRevocation list files: ");
        it = m_revocationListFiles.iterator();
        while (it.hasNext())
        {
            sb.append(it.next().toString() + " ");
        }
        return sb.toString();
    }
}
