//===========================================================================
//
// Copyright (c)  2003-2010 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.pkits;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.security.cert.CRLException;

/**
 * This is a convenience class for storing results from running individual
 * <code>PKITSTest</code>s. Test results can be divided up in to three
 * categories:
 * <ul>
 * <li>Success: either the certificate was supposed to validate and did,
 *     or it was not supposed to validate and did not.</li>
 * <li>Failure: either the certificate was supposed to validate and did not,
 *     or it was not supposed to validate and did.</li>
 * <li>Possible failure: if some exception not related to certificate
 *     validation was thrown. This most likely indicates a problem with the
 *     test setup.</li>
 * </ul>
 * 
 */
class PKITSTestResult
{
    public static final int RESULT_SUCCESS = 0;
    public static final int RESULT_CERTVAL_EXCEPTION = 1;
    public static final int RESULT_OTHER_EXCEPTION = 2;

    private int m_resultCode;
    private String m_testTitle;
    private String m_comment;
    private boolean m_shouldValidate;
    private Exception m_exception;

    /**
     * Constructor for PKITSTestResult
     * 
     * @param testTitle
     *     the test title.
     * 
     * @param comment
     *     an optional comment to explain the test result, <code>null</code>
     * if there is no comment. 
     * 
     * @param shouldValidate
     *     <code>true</code> if the path was supposed to validate
     * successfully, <code>false</code> otherwise.
     * 
     * @param exception
     *     the <code>Exception</code> that occurred during certificate
     * validation, or <code>null</code> if there was no exception.
     */
    PKITSTestResult(String testTitle, String comment, boolean shouldValidate, Exception exception)
    {
        m_testTitle = testTitle;
        m_comment = comment;
        m_exception = exception;
        m_shouldValidate = shouldValidate;
        if (exception == null)
        {
            m_resultCode = RESULT_SUCCESS;
        }
        else if (
            exception instanceof CRLException
                || exception instanceof java.security.cert.CertificateException
                || exception instanceof com.entrust.toolkit.exceptions.CertificationException)
        {
            m_resultCode = RESULT_CERTVAL_EXCEPTION;
        }
        else
        {
            m_resultCode = RESULT_OTHER_EXCEPTION;
        }

    }

    /**
     * Return <code>true</code> if the test is a success,
     * <code>false</code> otherwise.
     * 
     * @return <code>true</code> if the test is a success,
     * <code>false</code> otherwise.
     */
    boolean isSuccess()
    {
        if (m_shouldValidate && m_resultCode == RESULT_SUCCESS)
        {
            return true;
        }
        else if (!m_shouldValidate && m_resultCode == RESULT_CERTVAL_EXCEPTION)
        {
            return true;
        }
        else
        {
            return false;
        }
    }

    /**
     * Return <code>true</code> if the test is a failure,
     * <code>false</code> otherwise.
     * 
     * @return <code>true</code> if the test is a failure,
     * <code>false</code> otherwise.
     */
    boolean isFailure()
    {
        if (m_shouldValidate && m_resultCode == RESULT_CERTVAL_EXCEPTION)
        {
            return true;
        }
        else if (!m_shouldValidate && m_resultCode == RESULT_SUCCESS)
        {
            return true;
        }
        else
        {
            return false;
        }
    }

    /**
     * Return <code>true</code> if the test cannot be determined
     * to be either a true success or failure, <code>false</code>
     * if it can.
     * 
     * @return <code>true</code> if the test cannot be determined
     * to be either a true success or failure, <code>false</code>
     * if it can.
     */
    boolean isPossibleFailure()
    {
        if (m_resultCode == RESULT_OTHER_EXCEPTION)
            return true;
        return false;
    }

    /**
     * Returns the result of the test, either pass, fail, or possible
     * failure.
     * 
     * @param explainFailures
     *     If the test failed, either give the stack trace, or explain what
     * went wrong.
     * 
     * @return
     *     the result of the test.
     */
    String getResultString(boolean explainFailures)
    {
        StringBuffer sb = new StringBuffer();
        if (isSuccess())
        {
            sb.append("Success");
            if (m_comment != null)
            {
                sb.append("\r\nComment: " + m_comment);
            }
        }
        else if (isFailure())
        {
            sb.append("FAIL");
            if (m_comment != null)
            {
                sb.append("\r\nComment: " + m_comment);
            }
            if (explainFailures)
            {
                sb.append("\r\n");
                if (m_shouldValidate)
                {
                    StringWriter result = new StringWriter();
                    PrintWriter writer = new PrintWriter(result);
                    m_exception.printStackTrace(writer);
                    writer.flush();
                    writer.close();
                    sb.append(result.toString());
                }
                else
                {
                    sb.append("Expected validation failure, but validation succeeded.");
                }
            }
        }
        else if (isPossibleFailure())
        {
            sb.append("Possible Failure");
            if (m_comment != null)
            {
                sb.append("\r\nComment: " + m_comment);
            }
            if (explainFailures)
            {
                sb.append("\r\n");
                StringWriter result = new StringWriter();
                PrintWriter writer = new PrintWriter(result);
                m_exception.printStackTrace(writer);
                writer.flush();
                writer.close();
                sb.append(result.toString());
            }
        }
        else
        {
            sb.append("Unknown Result");
        }
        return sb.toString();

    }

    /**
     * Return the title of the test.
     * 
     * @return the title of the test.
     */
    public String getTestTitle()
    {
        return m_testTitle;
    }

    /**
     * Query if there is an exception message.
     *  
     * @return
     *     <code>true</code>if there is an exception message associated with
     * this test, <code>false</code> otherwise.
     */
    public boolean hasExceptionMessage()
    {
        return m_exception != null;
    }

    /**
     * Returns the exception associated with this test.
     * 
     * @return the exception associated with this test.
     */
    public Exception getException()
    {
        return m_exception;
    }

}
