//===========================================================================
//
// Copyright (c)  2003-2010 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.serverLogin;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.security.cert.CertificateException;
import java.util.Properties;

import com.entrust.toolkit.User;
import com.entrust.toolkit.credentials.CredentialReader;
import com.entrust.toolkit.credentials.FilenameProfileReader;
import com.entrust.toolkit.credentials.FilenameProfileWriter;
import com.entrust.toolkit.credentials.UALCredentialReader;
import com.entrust.toolkit.exceptions.UserBadPasswordException;
import com.entrust.toolkit.exceptions.UserFatalException;
import com.entrust.toolkit.util.ManagerTransport;
import com.entrust.toolkit.x509.directory.JNDIDirectory;


/**
 * Utility class for logging in to the Toolkit using Server Login.
 *
 * <p>
 *
 * If all required login information is already available, one may call
 * the static method {@link #logIn(String,String,String,int,String,int)}
 * directly.
 *
 * <p>
 *
 * To retrieve the login parameters from a Properties, use
 * {@link #logIn(Properties)}.  This method will look for the following
 * information in the Properties:
 *
 * <p>
 *
 * <table border="1">
 *   <tr>
 *     <th>Parameter</th>
 *     <th>Default property name</th>
 *     <th>Presence</th>
 *   </tr>
 *   <tr>
 *     <td>EPF file name</td>
 *     <td><tt>profile.epf</tt></td>
 *     <td>Required</td>
 *   </tr>
 *   <tr>
 *     <td>UAL file name</td>
 *     <td><tt>profile.ual</tt></td>
 *     <td>
 *       Optional -- the default is derived from the EPF file name by
 *       forcing a ".ual" extension
 *     </td>
 *   </tr>
 *   <tr>
 *     <td>CA host</td>
 *     <td><tt>ca.host</tt></td>
 *     <td>Required</td>
 *   </tr>
 *   <tr>
 *     <td>CA port</td>
 *     <td><tt>ca.port</tt></td>
 *     <td>Optional</td>
 *   </tr>
 *   <tr>
 *     <td>Directory host</td>
 *     <td><tt>directory.host</tt></td>
 *     <td>Required</td>
 *   </tr>
 *   <tr>
 *     <td>Directory port</td>
 *     <td><tt>directory.port</tt></td>
 *     <td>Optional</td>
 *   </tr>
 * </table>
 *
 * <p>
 *
 * The property names may be changed using various methods in this class.
 */
public class ServerLogin
{

    public static final int DEFAULT_LDAP_PORT = 389;
    public static final int DEFAULT_PKIX_CMP_PORT = 829;

    private static final String EPF_SETTING = "profile.epf";
    private static final String UAL_SETTING = "profile.ual";
    private static final String CA_HOST_SETTING = "ca.host";
    private static final String CA_PORT_SETTING = "ca.port";
    private static final String DIRECTORY_HOST_SETTING = "directory.host";
    private static final String DIRECTORY_PORT_SETTING = "directory.port";

    private String      m_epfFileSetting = EPF_SETTING;
    private String      m_ualFileSetting = UAL_SETTING;
    private String      m_caHostSetting = CA_HOST_SETTING;
    private String      m_caPortSetting = CA_PORT_SETTING;
    private String      m_directoryHostSetting = DIRECTORY_HOST_SETTING;
    private String      m_directoryPortSetting = DIRECTORY_PORT_SETTING;


	/**
     * Logs in to a profile using settings from the provided Properties.
     * The login is online, not offline.
     *
     * @param settings The Properties containing EPF file name, etc.
     *
     * @return A logged in User instance.
     *
     * @throws IOException
     * @throws UserFatalException
     * @throws UserBadPasswordException
     * @throws CertificateException
     * @throws NumberFormatException
     *              If <code>settings</code> contains an illegally formatted
     *              integer for a port number
     * @throws IllegalArgumentException
     *              If <code>settings</code> is missing a required setting.
     */
    public User
    logIn( Properties settings )
            throws IOException, UserFatalException,
                   UserBadPasswordException,
                   CertificateException,
                   NumberFormatException, IllegalArgumentException
    {
        //*** Fetch EPF file name
        String epfFile = settings.getProperty( m_epfFileSetting );
        if( epfFile == null ) {
            throw new IllegalArgumentException( missing( m_epfFileSetting ) );
        }

        //*** Fetch UAL file name
        String ualFile = settings.getProperty( m_ualFileSetting );
        if( ualFile == null ) {
            ualFile = assumeExtension( epfFile, ".ual" );
        }

        //*** Fetch CA host name
        String caHost = settings.getProperty( m_caHostSetting );
        if( caHost == null ) {
            throw new IllegalArgumentException( missing( m_caHostSetting ) );
        }

        //*** Fetch CA port (defaults to 829)
        int caPort = DEFAULT_PKIX_CMP_PORT;         // default
        String caPortStr = settings.getProperty( m_caPortSetting );
        if( caPortStr != null ) {
            caPort = Integer.parseInt( caPortStr );
        }

        //*** Fetch directory host name
        String directoryHost = settings.getProperty( m_directoryHostSetting );
        if( directoryHost == null ) {
            throw new IllegalArgumentException(
                    missing( m_directoryHostSetting ) );
        }

        //*** Fetch directory port (defaults to 389)
        int directoryPort = DEFAULT_LDAP_PORT;      // default
        String directoryPortStr = settings.getProperty( m_directoryPortSetting );
        if( directoryPortStr != null ) {
            directoryPort = Integer.parseInt( directoryPortStr );
        }

        //*** Log in
        return( logIn( epfFile, ualFile, caHost, caPort, directoryHost,
                       directoryPort ) );
    }


	/**
     * Logs in to a profile.  The login is online, not offline.
     *
     * @param epfFile The EPF file name
     * @param ualFile The UAL file name
     * @param caHost The CA host name or IP address
     * @param caPort The CA port
     * @param directoryHost The directory host name or IP address
     * @param directoryPort The directory port
     *
     * @return A logged in User instance.
     *
     * @throws IOException
     * @throws UserFatalException
     * @throws UserBadPasswordException
     * @throws CertificateException
     */
    public static User
    logIn( String epfFile, String ualFile, String caHost, int caPort,
           String directoryHost, int directoryPort )
            throws IOException, UserFatalException,
                   UserBadPasswordException,
                   CertificateException
    {
        User user = new User();

        CredentialReader ualReader = new UALCredentialReader(
                new FilenameProfileReader( epfFile ),
                new FileInputStream( ualFile ) );

        user.setCredentialWriter(
                new FilenameProfileWriter( epfFile, 0 ) );

        user.setConnections(
                new JNDIDirectory( directoryHost, directoryPort ),
                new ManagerTransport( caHost, caPort ) );

        user.login( ualReader, null );

        return( user );
    }


    /**
     * Sets the setting name corresponding to the EPF file name.
     *
     * @param settingName The new setting name
     */
    public void
    setEPFFileSetting( String settingName )
    {
        m_epfFileSetting = settingName;
    }


    /**
     * Sets the setting name corresponding to the UAL file name.
     *
     * @param settingName The new setting name
     */
    public void
    setUALFileSetting( String settingName )
    {
        m_ualFileSetting = settingName;
    }


    /**
     * Sets the setting name corresponding to the CA host.
     *
     * @param settingName The new setting name
     */
    public void
    setCAHostSetting( String settingName )
    {
        m_caHostSetting = settingName;
    }


    /**
     * Sets the setting name corresponding to the CA port.
     *
     * @param settingName The new setting name
     */
    public void
    setCAPortSetting( String settingName )
    {
        m_caPortSetting = settingName;
    }


    /**
     * Sets the setting name corresponding to the directory host.
     *
     * @param settingName The new setting name
     */
    public void
    setDirectoryHostSetting( String settingName )
    {
        m_directoryHostSetting = settingName;
    }


    /**
     * Sets the setting name corresponding to the directory port.
     *
     * @param settingName The new setting name
     */
    public void
    setDirectoryPortSetting( String settingName )
    {
        m_directoryPortSetting = settingName;
    }


    /**
     * Constructs an error message for missing required settings.
     */
    private static String
    missing( String settingName )
    {
        return( "Missing required setting: " + settingName );
    }


	/**
     * Assume an extension for a file name.  If <code>fileName</code> already
     * has an extension, it will be replaced with <code>extension</code>;
     * otherwise, <code>extension</code> will be appended to
     * <code>fileName</code>.
     *
     * @param fileName The input file name (may be absolute or relative).
     * @param extension The extension to assume, including the leading ".".
     *
     * @return The constructed file name.
     */
    private static String
    assumeExtension( String fileName, String extension )
    {
        //*** Parse the provided path
        File file = new File( fileName );
        File parent = file.getParentFile();
        if( parent == null  &&  fileName.endsWith( File.separator ) ) {
            parent = file;      // handle cases like "c:\" and "/"
        }

        //*** Assume the provided extension
        String oldName = file.getName();
        String newName = null;
        int index = oldName.lastIndexOf( '.' );
        if( index != -1 ) {
            newName = oldName.substring( 0, index ) + extension;
        } else {
            newName = oldName + extension;
        }

        //*** Construct the new path string
        return( new File( parent, newName ).toString() );
    }

}
