//===========================================================================
//
// Copyright (c)  1998-2010 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.servlet;

import java.io.IOException;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;

import com.entrust.toolkit.User;
import com.entrust.toolkit.credentials.CredentialReader;
import com.entrust.toolkit.credentials.FilenameProfileReader;
import com.entrust.toolkit.util.SecureStringBuffer;
import com.entrust.toolkit.x509.directory.JNDIDirectory;

/**
 * A sample client application that is designed to communicate securely with
 * a corresponding Servlet using PKCS #7 to secure communication.
 */
public class PKCS7SampleClient {

    /**
     * Sample client application.
     * <p></p>
     */
    public static void main(String[] args) {
        // Check the command-line arguments.  If they're not there, then
        // print out a usage message.
        if (args.length < 4) {
            System.out.println("\nDemonstrates exchange of PKCS#7 messages with a servlet");
            System.out.println("\nUsage: Client <profile> <password> <message> <servlet URL> [-dir <ip address> ]\n");
            return;
        }

        // Parse the command-line arguments.
        int index = 0;
        String profile = args[index++];
        String password = args[index++];
        String message = args[index++];
        URL servletUrl = null;
        try {
            servletUrl = new URL(args[index++]);
        } catch (MalformedURLException e) {
            throw new IllegalArgumentException("<servlet URL> contains an URL: " + e.toString());
        }
        if (!servletUrl.getProtocol().equalsIgnoreCase("http"))
            throw new IllegalArgumentException("<servlet URL> must contain an http URL");
        String directoryIP = null;
        if (args.length > index && args[index].equals("-dir"))
            directoryIP = args[++index];

        System.out.print("Logging in User... ");
        User user = new User();
        try {
            // Set a connection to the certificate repository (LDAP Directory)
            if (directoryIP != null)
                user.setConnections(new JNDIDirectory(directoryIP, 389), null);

            // Login User
            CredentialReader cr = new FilenameProfileReader(profile);
            user.login(cr, new SecureStringBuffer(new StringBuffer(password)));
            System.out.println("DONE");
            System.out.println("User online: " + user.isOnline());
        } catch (Exception e) {
            System.out.println("FAILED");
            e.printStackTrace();
            return;
        }

        // Connect to servlet URL
        HttpURLConnection httpURLConnection = null;
        try {
            httpURLConnection = (HttpURLConnection) servletUrl.openConnection();
        } catch (Exception e) {
            System.out.println("Unable to connect to <serverURL> using http protocol");
            e.printStackTrace();
            return;
        }

        // Send a signed/encrypted PKCS #7 format message to the servlet and
        // received a signed/encrypted PKCS #7 message as a reply
        System.out.println("\nSTART: Communication with servlet");
        try {
            // Connect to the servlet, and use a POST to send the encrypted/signed
            // message
            System.out.println("\nSending PKCS #7 message to servlet...");
            httpURLConnection.setUseCaches(false);
            httpURLConnection.setRequestMethod("POST");
            httpURLConnection.setDoOutput(true);
            httpURLConnection.connect();
            PKCS7Util.writeEncryptSignedMessage(user, message, httpURLConnection.getOutputStream());
            System.out.println("Message sent");
            System.out.println("\n- Start of sent message -\n" + message);
            System.out.println("- End of sent message -");

            // Read the reply from the servlet, and decrypt/verify the response
            System.out.println("\nReceiving PKCS #7 message from servlet...");
            int responseCode = httpURLConnection.getResponseCode();
            if (httpURLConnection.getResponseCode() != HttpURLConnection.HTTP_OK)
                throw new IOException("Unexpected response code received from servlet: " + responseCode);
            String response = PKCS7Util.readEncryptedSignedMessage(user, httpURLConnection.getInputStream());
            System.out.println("Message received");
            System.out.println("\n- Start of received message -\n" + response);
            System.out.println("- End of received message -\n");
        } catch (Exception e) {
            httpURLConnection.disconnect();
            e.printStackTrace();
        }
        System.out.println("END: Communication with servlet");
    }
}