//===========================================================================
//
// Copyright (c)  1998-2010 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.servlet;

import java.io.IOException;

import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.entrust.toolkit.User;
import com.entrust.toolkit.credentials.CredentialReader;
import com.entrust.toolkit.credentials.FilenameProfileReader;
import com.entrust.toolkit.util.SecureStringBuffer;
import com.entrust.toolkit.x509.directory.JNDIDirectory;

/**
 * A sample servlet, designed to securely communicate with a corresponding
 * client using HTTP POST and PKCS #7 messages.
 */
public class PKCS7SampleServlet extends HttpServlet {

    /** The users whose credentials will be sued to secure communications. */
    private static User m_user;

    /**
     * <p>
     * Logs into a User that represents the server, so the servlet can perform 
     * cryptographic operations.
     * </p>
     * 
     * @param config
     *      configuration information for this servlet
     * 
     * @throws ServletException
     *      if an error occurs during initialization
     */
    public void init(ServletConfig config) throws ServletException {
        super.init(config);
        // Only initialize once
        if (m_user != null)
            return;

        // In this simple demo, the profile name and password are hard-coded.
        String profileName = config.getInitParameter("EPF");
        SecureStringBuffer password = new SecureStringBuffer(new StringBuffer(config.getInitParameter("Password")));
        String directoryIP = config.getInitParameter("DirectoryIP");

        try {
            // Create new user
            m_user = new User();

            //  Connect to the infrastructure.
            if (directoryIP != null) {
                JNDIDirectory dir = new JNDIDirectory(directoryIP, 389);
                m_user.setConnections(dir, null);
            }

            CredentialReader cr = new FilenameProfileReader(getServletContext().getRealPath(profileName));
            m_user.login(cr, new SecureStringBuffer(password));
        } catch (Exception e) {
            System.out.println("PKCS7SampleServlet initialization failed");
            e.printStackTrace();
            throw new ServletException("PKCS7SampleServlet initialization failed: " + e.toString());
        }
        System.out.println("PKCS7SampleServlet initialized");
    }

    /**
     * Returns a short description of this servlet.
     * 
     * @return
     *      a description of the servlet
     */
    public String getServletInfo() {
        return "Entrust Authority(TM) Security Toolkit for the Java(R) Platform Basic PKCS#7 Servlet Example";
    }

    /**
     * Handles communication with a client.  
     * 
     * <p>
     * Receives an encrypted/signed PKCS #7 formatted message from the client, 
     * decrypts it, and validates the signature.  It then appends creates a 
     * confirmation message, encrypts and signs it, and returns it as a reply
     * to the client.
     * </p>
     *
     * @param req
     *      the request the client has made of the servlet
     * 
     * @param res
     *      the response the servlet sends to the client
     * 
     * @throws ServletException
     *      if the request for the POST could not be handled
     * 
     * @exception IOException
     *      if an error occurs while communicating with the client
     */
    public synchronized void doPost(HttpServletRequest req, HttpServletResponse res) throws ServletException,
        IOException {

        // Prepare to acknowledge the POST
        res.setStatus(HttpServletResponse.SC_OK);
        System.out.println();
        System.out.println("START: Communication with client");
        System.out.println();
        try {
            // Receive and decrypt/verify PKCS #7 message
            System.out.println("Receiving message from client");
            String message = PKCS7Util.readEncryptedSignedMessage(m_user, req.getInputStream());
            System.out.println("Message received");
            System.out.println();
            System.out.println("- Start of received message -");
            System.out.println(message);
            System.out.println("- End of received message -");
            System.out.println();

            // Encrypt/sign PKCS #7 response message
            System.out.println("Sending reply to client");
            String response = "I have received the following secret message from you: " + message;
            PKCS7Util.writeEncryptSignedMessage(m_user, response, res.getOutputStream());
            System.out.println("Message sent");
            System.out.println();
            System.out.println("- Start of sent message -");
            System.out.println(response);
            System.out.println("- End of sent message -");            
            System.out.println();            
            System.out.println("END: Communication with client");
        } catch (Exception e) {
            System.out.println("Failed while communicating with client");
            e.printStackTrace();
            throw new IOException("Failed while communicating with client: " + e.toString());
        }
    }
}