// Copyright (C) 2018 Entrust Datacard
// All rights reserved.

package com.entrust.toolkit.examples.smimev3;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateExpiredException;
import java.security.cert.CertificateNotYetValidException;
import java.util.Enumeration;

import javax.mail.MessagingException;
import javax.mail.internet.MimeMessage;

import com.entrust.toolkit.PKCS7DecodeStream;
import com.entrust.toolkit.PKCS7EncodeStream;
import com.entrust.toolkit.User;
import com.entrust.toolkit.credentials.CredentialReader;
import com.entrust.toolkit.credentials.FilenameProfileReader;
import com.entrust.toolkit.exceptions.InvalidUserException;
import com.entrust.toolkit.exceptions.UserBadPasswordException;
import com.entrust.toolkit.exceptions.UserFatalException;
import com.entrust.toolkit.exceptions.UserNotLoggedInException;
import com.entrust.toolkit.exceptions.UserRecoverException;
import com.entrust.toolkit.security.provider.Initializer;
import com.entrust.toolkit.util.SecureStringBuffer;

import iaik.x509.X509Certificate;

/**
 * This sample demonstrates how to decode S/MIME protected files (files with a
 * .p7m extension) that were written by Entrust Entelligence. It demonstrates
 * how to decode signed, encrypted, and 'signed & encrypted' p7m files.
 * 
 * <p>
 * This sample does not show how to encode the files (which are input parameters
 * here), as they are expected to come from Entelligence. Since the protected
 * file can be setup in layers ie (signed and then encrypted), as each layer is
 * decoded, it's content will be written to disc allowing you do inspect it.
 * </p>
 *
 * <p>
 * To run the sample, from the <code>etjava\examples</code> directory, run the
 * following:
 * </p>
 * 
 * <pre>
 * java -classpath classes;../etjava/lib/enttoolkit.jar com.entrust.toolkit.examples.smime.JTKInterop
 *       "epf file" "epf password" "inputFile" "outputDir"
 * </pre>
 * <p>
 *
 */

public class JTKInterop {

    private static int LAYER = 1;
    private static String outputDir;
    private static String fileNamePrefix = "unwrapped_layer_";

    public JTKInterop() {
        Initializer.getInstance().setProviders(Initializer.MODE_NORMAL);
    }

    /**
     * The main driver for the sample.
     * 
     * @param args
     *            The program arguments. See the class description for details.
     */
    public static void main(String[] args) {

        if (args.length < 4) {
            System.err.println("Usage");
            System.out.println("EPF");
            System.out.println("EPF password");
            System.out.println(
                    "inputFile:  p7m file produced by Entelligence actions of  'Encrypt and Digitally Sign File' | 'Encrypt File' | 'Digitally Sign File'");
            System.out.println("outputDir");
            return;
        }

        String epf = args[0];
        String password = args[1];
        String inputFile = args[2];
        outputDir = args[3];

        System.out.println("EPF:          " + epf);
        System.out.println("EPF password: " + password);
        System.out.println("InputFile:    " + inputFile);
        System.out.println("OutputDir:    " + outputDir);
        System.out.println();

        JTKInterop decoder = new JTKInterop();

        User user = null;
        try {

            user = decoder.login(epf, password);
            decoder.unWrapNextLayer(user, inputFile, deriveFile(outputDir, fileNamePrefix, "" + LAYER++));

        } catch (Throwable t) {
            t.printStackTrace();
        } finally {
            if (user != null && user.isLoggedIn()) {
                try {
                    user.logout();
                } catch (UserNotLoggedInException e) {
                    e.printStackTrace();
                }

            }
        }

    }

    /**
     * // Log in to the supplied epf file
     * 
     * @param epf
     *            Path to EPF file
     * @param password
     *            Clear password of EPF file
     * @return EPF user object
     * @throws UserRecoverException
     * @throws UserBadPasswordException
     * @throws UserFatalException
     * @throws CertificateException
     * @throws FileNotFoundException
     */
    private User login(String epf, String password) throws UserRecoverException, UserBadPasswordException,
            UserFatalException, CertificateException, FileNotFoundException {

        User user = new User();
        CredentialReader cr = new FilenameProfileReader(epf);
        user.login(cr, new SecureStringBuffer(new StringBuffer(password)));
        return user;

    }

    /**
     * Read the input file to get the contentType to determine next stage of
     * processing
     * 
     * @param inputFile
     *            Input file to decode which may contain a mime header
     * @return mime ContentType
     * @throws FileNotFoundException
     * @throws IOException
     * @throws MessagingException
     */
    private String readMimeContentType(String inputFile) throws FileNotFoundException, IOException, MessagingException {

        try (InputStream is = new FileInputStream(inputFile)) {

            MimeMessage mimeLayer = new MimeMessage(null, is);
            String contentType = mimeLayer.getHeader("content-type", null);

            return contentType;
        }

    }

    /**
     * Recursively process the layered P7 input file decoding each layer till we
     * get to the text/plain layer
     * 
     * @param user
     *            EPF user object
     * @param inputFile
     *            P7 file to decode
     * @param outputFile
     *            OutputFile to write decoded data to
     * @throws FileNotFoundException
     * @throws IOException
     * @throws MessagingException
     * @throws NullPointerException
     * @throws NoSuchAlgorithmException
     */
    private void unWrapNextLayer(User user, String inputFile, String outputFile) throws FileNotFoundException,
            IOException, MessagingException, NullPointerException, NoSuchAlgorithmException {

        // check if file has a mime header
        String contentTypePreCheck = readMimeContentType(inputFile);

        try (InputStream is = new FileInputStream(inputFile)) {

            if (contentTypePreCheck == null) {
                // no mime header found, so decode the input stream directly

                decodeP7Stream(user, is, outputFile);
                unWrapNextLayer(user, outputFile, deriveFile(outputDir, fileNamePrefix, "" + LAYER++));
            } else {
                // we have a mime header, consume the mime header to get the
                // contentType to tell us how to handle the content

                MimeMessage mimeLayer = new MimeMessage(null, is);
                String contentType = mimeLayer.getHeader("content-type", null);
                Object content = mimeLayer.getContent();

                System.out.println("contentType: " + contentType);

                if (contentType.startsWith("application/x-pkcs7-mime")
                        || contentType.startsWith("application/pkcs7-mime")) {
                    // the content is P7 stuff so decode it

                    InputStream data = (InputStream) content;
                    decodeP7Stream(user, data, outputFile);
                    unWrapNextLayer(user, outputFile, deriveFile(outputDir, fileNamePrefix, "" + LAYER++));
                } else if (contentType.startsWith("text/plain")) {
                    // the content is text so extract it, this should be the
                    // final layer

                    String data = (String) content;
                    try (OutputStream fos = new FileOutputStream(outputFile)) {
                        System.out.println("Writing to file: " + outputFile);
                        fos.write(data.getBytes());
                    }

                } else {
                    System.err.println("Un-Implemented ContentType handler:" + contentType);
                }

            }
        }

        System.out.println();
    }

    /**
     * Given a P7 input stream, decode its content, write the decoded stream to
     * file
     * 
     * @param user
     *            EPF user object
     * @param is
     *            P7 InputStream
     * @param outputFile
     *            OutputFile to write decoded data to
     * @throws InvalidUserException
     * @throws NullPointerException
     * @throws NoSuchAlgorithmException
     * @throws IOException
     */
    private void decodeP7Stream(User user, InputStream is, String outputFile)
            throws InvalidUserException, NullPointerException, NoSuchAlgorithmException, IOException {

        try (PKCS7DecodeStream decodeStream = new PKCS7DecodeStream(user, is)) {
            displayOperation(decodeStream.getOperation());

            try (OutputStream fos = new FileOutputStream(outputFile)) {
                System.out.println("Writing to file: " + outputFile);
                copy(decodeStream, fos);
            }

            displaySigner(decodeStream);
            displayCerts(decodeStream);
        }
    }

    private static void displayCerts(PKCS7DecodeStream decodeStream) {

        if (decodeStream.getIncludedCertificates() == null) {
            return;
        }

        System.out.println("Included Certs:");
        for (Enumeration enumeration = decodeStream.getIncludedCertificates().elements(); enumeration
                .hasMoreElements();) {
            X509Certificate cert = (X509Certificate) enumeration.nextElement();
            System.out.println("   Subject: " + cert.getSubjectDN().getName());
            System.out.println("   Serial Number: " + cert.getSerialNumber());
            System.out.println("   Issuer : " + cert.getIssuerDN().getName());
            System.out.println();

            try {
                cert.checkValidity();
            } catch (CertificateExpiredException | CertificateNotYetValidException e) {
                // TODO Auto-generated catch block
                e.printStackTrace();
            }
        }
    }

    private static void displaySigner(PKCS7DecodeStream decodeStream) {

        for (int index = 0; index < decodeStream.getNumberOfSignatures(); index++) {
            X509Certificate signingCert = decodeStream.getSignerCertificate(index);
            System.out.println("   Signer " + (index + 1) + ": " + signingCert.getSubjectDN());
            System.out.println("   Serial Number: " + signingCert.getSerialNumber());
            System.out.println("   Issuer: " + signingCert.getIssuerDN());
            System.out.println();

            try {
                signingCert.checkValidity();
            } catch (CertificateExpiredException | CertificateNotYetValidException e) {

                e.printStackTrace();
            }

        }

    }

    /**
     * Generate a file name indicating the layer, so we can see the layers as we
     * unwrap the P7 input file
     * 
     * @param dir
     *            Directory
     * @param prefix
     *            Filename prefix
     * @param suffix
     *            Filename suffix (layer)
     * @return derivedFile name
     */

    private static String deriveFile(String dir, String prefix, String suffix) {

        File file = new File(dir);
        if (!file.isDirectory()) {
            throw new IllegalArgumentException(dir + "  is not a dir");
        }

        String derivedFile = dir + File.separator + prefix + suffix + ".txt";
        File f = new File(derivedFile);

        return f.getPath();
    }

    private static void displayOperation(int operation) {

        String oper = "Unknown";

        switch (operation) {
        case PKCS7EncodeStream.SIGN_AND_ENCRYPT:
            oper = "SIGN_AND_ENCRYPT";
            break;

        case PKCS7EncodeStream.ENCRYPT_ONLY:
            oper = "ENCRYPT_ONLY";
            break;

        case PKCS7EncodeStream.SIGN_ONLY:
            oper = "SIGN_ONLY";
            break;

        case PKCS7EncodeStream.CLEAR_SIGN:
            oper = "CLEAR_SIGN";
            break;

        case PKCS7EncodeStream.EXPORT_CERTIFICATES:
            oper = "EXPORT_CERTIFICATES";
            break;

        default:

        }

        System.out.println("\nOperation: " + oper);

    }

    /**
     * Used to copy data from an input stream to an output stream
     * 
     * @param is
     *            InputStream
     * @param os
     *            OutputStream
     * @throws IOException
     */
    private static void copy(InputStream is, OutputStream os) throws IOException {

        byte[] b = new byte[128];
        int i = is.read(b);
        while (i >= 0) {
            os.write(b, 0, i);
            i = is.read(b);
        }

    }

}
