//===========================================================================
//
// Copyright (c)  1997-2010 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.smimev3;

import iaik.asn1.structures.Attribute;
import iaik.cms.SignerInfo;
import iaik.pkcs.pkcs10.CertificateRequest;
import iaik.smime.CompressedContent;
import iaik.smime.EncryptedContent;
import iaik.smime.PKCS10Content;
import iaik.smime.SignedContent;
import iaik.x509.X509Certificate;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.security.PrivateKey;
import java.security.SignatureException;
import java.security.cert.Certificate;
import java.util.Date;
import java.util.Properties;

import javax.activation.CommandMap;
import javax.activation.MailcapCommandMap;
import javax.mail.Address;
import javax.mail.FetchProfile;
import javax.mail.Flags;
import javax.mail.Folder;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.Multipart;
import javax.mail.Part;
import javax.mail.Session;
import javax.mail.Store;
import javax.mail.URLName;

import com.entrust.toolkit.User;
import com.entrust.toolkit.credentials.StreamProfileReader;
import com.entrust.toolkit.util.IniFile;
import com.entrust.toolkit.util.SecureStringBuffer;

//import demo.keystore.CMSKeyStore;

/**
 * This class demonstrates the usage of the IAIK S/MIME implementation. To run this demo
 * you also need the packages:
 * <ul>
 * <li>mail.jar: Get it from <a href="http://www.javasoft.com/products/javamail/">Sun</a>.
 * <li>activation.jar: Get it from <a href="http://java.sun.com/beans/glasgow/jaf.html">Sun</a>.
 * <li>OPTIONAL: pop3.jar: If you need POP3 access.
 *     Get it from <a href="http://www.javasoft.com/products/javamail/">here</a>.
 * </ul>
 *
 */
public class SMimeShow {

	String section = null;
	IniFile properties = null;
	String Profile = null;
	String from = null;
	String protocol = null;
	String host = null;
	String user = null;
	String password = null;
	String mbox = null;
	String url = null;
	boolean verbose = false;
	PrivateKey privateKey = null;

	User recipient = null;
	static int c;
	
  
  public static final String mMailCapString =
	  "#\r\n" +
	  "# IAIK 'mailcap' file\r\n" +
	  "#\r\n" +
	  "multipart/signed;;               x-java-content-handler=iaik.smime.signed_content\r\n"    +
	  "application/x-pkcs7-signature;;  x-java-content-handler=iaik.smime.signed_content\r\n"    +
	  "application/x-pkcs7-mime;;       x-java-content-handler=iaik.smime.encrypted_content\r\n" +
	  "application/x-pkcs10;;           x-java-content-handler=iaik.smime.pkcs10_content\r\n"    +
	  "application/pkcs7-signature;;    x-java-content-handler=iaik.smime.signed_content\r\n"    +
	  "application/pkcs7-mime;;         x-java-content-handler=iaik.smime.encrypted_content\r\n" +
	  "application/pkcs10;;             x-java-content-handler=iaik.smime.pkcs10_content\r\n";  

  public static void main(String argv[]) throws Exception {
	(new SMimeShow()).show(argv);
	System.out.println("Ready!");
  }

  static public void setCommandMap() {
          CommandMap mimeMap = CommandMap.getDefaultCommandMap();
          MailcapCommandMap mc = new MailcapCommandMap(); 
          mc.addMailcap(mMailCapString);
          CommandMap.setDefaultCommandMap(mc);
  }

  public SMimeShow() throws Exception {
    
    //get the properties from example.ini
  	setCommandMap();
    try{
    properties = new IniFile("data/smime/smime_example.ini");
    }
    catch(FileNotFoundException ex){
        System.out.println("Could not found the file data/smime/smime_example.ini: " + ex.getMessage());
        throw ex;
    }
    
    //section  of IniFile for SMimeShow 
    section = new String("SMimeShow");
    
    //get the profile .epf of the recipient 
    FileInputStream fis = null;
    try {
        fis = new FileInputStream(properties.getString(section,"Profile"));
    }
    catch(FileNotFoundException e) {
        System.out.println("Profile not found!");
    }
    StreamProfileReader cr = null;
    cr = new StreamProfileReader(fis);

    //PKI login for recipient
    recipient = new User();
    try {
        recipient.login(cr, new SecureStringBuffer((properties.getString(section,"recipient_password"))));
    }
    catch(Exception e) {
        System.out.print("Login failed: " + e.getMessage());
        throw e;
    }
    
    // get the private key for decrypting the messages from the User
    try {
        privateKey = recipient.getDecryptionKey();
    }
    catch(Exception e) {
        System.out.println(e.getMessage());
    }
  }

  public void show(String[] argv) throws IOException {

	int msgnum = -1;
	int optind = 0;

	if (argv.length > 0) {
	  for (optind = 0; optind < argv.length; optind++) {
		  if (argv[optind].equals("-T")) {
			  protocol = argv[++optind];
		  } else if (argv[optind].equals("-H")) {
			  host = argv[++optind];
		  } else if (argv[optind].equals("-U")) {
			  user = argv[++optind];
		  } else if (argv[optind].equals("-P")) {
			  password = argv[++optind];
		  } else if (argv[optind].equals("-v")) {
			  verbose = true;
		  } else if (argv[optind].equals("-f")) {
			  mbox = argv[++optind];
		  } else if (argv[optind].equals("-L")) {
			  url = argv[++optind];
		  } else if (argv[optind].equals("--")) {
			  optind++;
			  break;
		  } else if (argv[optind].startsWith("-")) {
			  System.out.println("Usage: SMimeShow [-L url] [-T protocol] [-H host] [-U user] [-P password] [-f mailbox] [msgnum] [-v]");
			  System.exit(1);
		  } else {
			  break;
		  }
		}
	}
	else {
		//default properties for Mailserver -> input console: java SMimeShow
		//protocol of getting messages from the mailserver
		protocol = properties.getString(section,"protocol"); 
		//host of mailserver
		host = properties.getString(section,"host");
		//mailserver user
		user = properties.getString(section,"user");
		//get password of mailserver user 
		password = properties.getString(section,"user_password");
		mbox = properties.getString(section,"mbox");

	  verbose = true;
	}

    try {
        // Get a Properties object
        Properties props = System.getProperties();

        // Get a Session object
        Session session = Session.getDefaultInstance(props, null);

        // Get a Store object
        Store store = null;
        if (url != null) {
            URLName urln = new URLName(url);
            store = session.getStore(urln);
            store.connect();
        } else {
            if (protocol != null)
            store = session.getStore(protocol);
        else
            store = session.getStore();

            // Connect
            if (host != null || user != null || password != null)
            store.connect(host, user, password);
            else
            store.connect();
        }

        // Open the Folder
        Folder folder = store.getDefaultFolder();
        if (folder == null) {
        System.out.println("No default folder");
        System.exit(1);
        }

        folder = folder.getFolder(mbox);
        if (folder == null) {
        System.out.println("Invalid folder");
        System.exit(1);
        }

        folder.open(Folder.READ_WRITE);
        //folder.open(Folder.READ_ONLY);            // only READ for POP3
        int totalMessages = folder.getMessageCount();

        if (totalMessages == 0) {
            System.out.println("Empty folder");
            folder.close(false);
            store.close();
            System.exit(1);
        }

        if (verbose) {
            int newMessages = folder.getNewMessageCount();
            System.out.println("Total messages = " + totalMessages);
            System.out.println("New messages = " + newMessages);
            System.out.println("-------------------------------");
        }

        if (msgnum == -1) {
            // Attributes & Flags for all messages ..
            Message[] msgs = folder.getMessages();

            // Use a suitable FetchProfile
            FetchProfile fp = new FetchProfile();
            fp.add(FetchProfile.Item.ENVELOPE);
            fp.add(FetchProfile.Item.FLAGS);
            fp.add("X-Mailer");
            folder.fetch(msgs, fp);

            for (int i = 0; i < msgs.length; i++) {

                System.out.println("--------------------------");
                System.out.println("MESSAGE #" + (i + 1) + ":");
                Address[] address = msgs[i].getFrom();
                if (address != null) {
                   //If there is no address, just skip printing out address
                   if (address[0] != null) {
                       from = address[0].toString();
                   }
                   dump(msgs[i]);
                   //delete the message from the folder
                   msgs[i].setFlag(Flags.Flag.DELETED, true);
                }
            }
        } else {
            System.out.println("Getting message number: " + msgnum);
            Message m = folder.getMessage(msgnum);
        from = m.getFrom()[0].toString();
            dump(m);
        }

        folder.close(true);
        store.close();

    } catch (Exception ex) {
        System.out.println("Oops, got exception! " + ex.getMessage());
        ex.printStackTrace();
    }
    System.out.println("Ready! Press the space bar...");
    System.in.read();
    System.exit(1);
  }

  public void dump(Object o) throws Exception {

    if (o instanceof Message)
        dumpEnvelope((Message)o);

    if (o instanceof Part) {
       System.out.println("CONTENT-TYPE: "+((Part)o).getContentType());
      o = ((Part)o).getContent();
    }
    
    if (o instanceof CompressedContent) {
    	System.out.println("This message is a compressed content!");
    	CompressedContent cc = (CompressedContent)o;
    	//If content to be decompressed is compressed with a different
    	//compression algorithm, make sure the CompressionProvider you
    	//are using supports the compression.  Without this setting, the
    	//default CompressionProvider will be used which supports ZLIB 
    	//compression
    	iaik.cms.CompressionProvider nullprovider = new NullCompression();
		cc.setCompressionProvider(nullprovider);
    	System.out.println("Algorithm: " + cc.getCompressionAlgorithm());
    	dump(cc.getContent());
    }

    else 
    if (o instanceof EncryptedContent) {
      // encrypted
        System.out.println("This message is encrypted!");

      EncryptedContent ec = (EncryptedContent)o;
      ec.decryptSymmetricKey(privateKey, 0);
      dump(ec.getContent());
    }
    else if (o instanceof SignedContent) {
      // signed
      System.out.println("This message is signed!");

      SignedContent sc = (SignedContent)o;

      if (sc.getSMimeType().equals("certs-only")) {
        System.out.println("This message contains only certificates!");
        Certificate[] certs = sc.getCertificates();
        for (int i = 0; i < certs.length; ++i) {
          //System.out.println(certs[i].toString());
        }
      }
      else {
        X509Certificate signer = null;
        try {
            signer = sc.verify();
            System.out.println("Verification with verify() successful!");
            System.out.println("This message is signed from: "+signer.getSubjectDN());
            try {
                sc.verify(signer.getPublicKey());
                System.out.println("Verification with verify(cert) successful!");
            }
            catch(Exception e) {
                System.out.println("Verification with verify(cert) failed!");
            }
            //System.out.println(signer.toString(true));


            X509Certificate[] certs = iaik.utils.Util.convertCertificateChain(sc.getCertificates());


            // arrange chain to get user cert at index 0
            certs = iaik.utils.Util.arrangeCertificateChain(certs,false);
            if (certs != null) {
               // verify certs and search for trusted cert in chain
                for(int i = 0; i < certs.length; i++) {
                    try {
                        recipient.validate(certs[i]);
                        System.out.println("Certificate trusted!");
                    }
                    catch(Exception e) {
                        System.out.println("Certificate not trusted!");
                    }
                }
            } // else: there may be more certs in the cert set: do some more sophisticated verification
        } catch (SignatureException ex) {
            System.out.println("Exception : " + ex.getMessage());
            System.out.println("Signature verification error!");
        }

        SignerInfo[] signer_infos = sc.getSignerInfos();
        for (int i=0; i<signer_infos.length; i++) {
           Attribute[] signedAttributes = signer_infos[i].getSignedAttributes();
           if (signedAttributes != null) {
             System.out.println("SignerInfo " + i + " contains the following signed attributes:");
             for (int j = 0; j < signedAttributes.length; j++) {
               System.out.println(signedAttributes[j].getType());
               System.out.println(signedAttributes[j]);
               System.out.println();
             }
           }
           Attribute[] unsignedAttributes = signer_infos[i].getUnsignedAttributes();
           if (unsignedAttributes != null) {
            System.out.println("SignerInfo " + i + " contains the following unsigned attributes:");
            for (int j = 0; j < unsignedAttributes.length; j++) {
              System.out.println(unsignedAttributes[j].getType());
            }
           }
        }

        dump(sc.getContent());
      }
    }
    
    else if (o instanceof PKCS10Content) {
      System.out.println("This message contains a certificate request:");
      PKCS10Content pkcs10 = (PKCS10Content)o;
      CertificateRequest request = pkcs10.getCertRequest();
      System.out.println(request.toString());
      try {
         if (request.verify())
           System.out.println("Request verification ok for " + request.getSubject());
         else
           System.out.println("Request verification error for " + request.getSubject());
      } catch (SignatureException ex) {
         System.out.println("Request verification error for " + request.getSubject());
      }
    }
    else if (o instanceof String) {
        System.out.println("Content is a String");
        System.out.println("---------------------------");
        System.out.println((String)o);
      } else if (o instanceof Multipart) {
        System.out.println("----------------> Content is a Multipart");
        Multipart mp = (Multipart)o;
        int count = mp.getCount();
        for (int i = 0; i < count; i++) {
        System.out.println("----------------> Multipart: "+(i+1));
            dump(mp.getBodyPart(i));
        }
        System.out.println("----------------> End of Multipart");
    } else if (o instanceof Message) {
        System.out.println("Content is a Nested Message");
        System.out.println("---------------------------");
        dump(o);
      } else if (o instanceof InputStream) {
        System.out.println("Content is just an input stream: "+o);
        System.out.println("---------------------------");
        InputStream is = (InputStream)o;
        int a;
        int sum = 0;
        byte[] buf = new byte[1024];
        while ((a = is.read(buf)) > 0) {
            System.out.println("Bytes available: "+a);
        sum += a;
        }
        System.out.println("Length of data: "+sum+" bytes");        
    }
  }

    /**
     * Prints the envelope of a message.
     */
  public static void dumpEnvelope(Message m) throws MessagingException {
    System.out.println("This is the message envelope");
    System.out.println("---------------------------");
    Address[] a;
      // FROM
    if ((a = m.getFrom()) != null) {
        for (int j = 0; j < a.length; j++)
        System.out.println("FROM: " + a[j].toString());
    }

    // TO
    if ((a = m.getRecipients(Message.RecipientType.TO)) != null) {
        for (int j = 0; j < a.length; j++)
            System.out.println("TO: " + a[j].toString());
    }

    // SUBJECT
    System.out.println("SUBJECT: " + m.getSubject());

    // DATE
    Date d = m.getSentDate();
    System.out.println("SendDate: "+(d != null ? d.toString() : "UNKNOWN"));

    // SIZE
    System.out.println("Size: " + m.getSize());

    // FLAGS:
    Flags flags = m.getFlags();
    StringBuffer sb = new StringBuffer();
    Flags.Flag[] sf = flags.getSystemFlags(); // get the system flags

    boolean first = true;
      for (int i = 0; i < sf.length; i++) {
        String s;
        Flags.Flag f = sf[i];
        if (f == Flags.Flag.ANSWERED)
            s = "\\Answered";
        else if (f == Flags.Flag.DELETED)
            s = "\\Deleted";
        else if (f == Flags.Flag.DRAFT)
            s = "\\Draft";
        else if (f == Flags.Flag.FLAGGED)
            s = "\\Flagged";
        else if (f == Flags.Flag.RECENT)
            s = "\\Recent";
        else if (f == Flags.Flag.SEEN)
            s = "\\Seen";
        else
            continue;   // skip it
        if (first)
            first = false;
        else
            sb.append(' ');
        sb.append(s);
    }

    String[] uf = flags.getUserFlags(); // get the user flag strings
    for (int i = 0; i < uf.length; i++) {
        if (first)
            first = false;
        else
            sb.append(' ');
        sb.append(uf[i]);
    }
    System.out.println("FLAGS = " + sb.toString());

    // X-MAILER
    String[] hdrs = m.getHeader("X-Mailer");
    if (hdrs != null) {
        System.out.println("X-Mailer: " + hdrs[0]);
    }
    else {
        System.out.println("X-Mailer NOT available");
    }
  }
}
