//===========================================================================
//
// Copyright (c)  2018 Entrust Datacard Corporation.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.smproxy;

import com.entrust.toolkit.User;
import com.entrust.toolkit.credentials.CredentialWriter;
import com.entrust.toolkit.credentials.RoamingCredentialReader;
import com.entrust.toolkit.roaming.RoamingConfiguration;
import com.entrust.toolkit.roaming.RoamingConfigurationWithSMProxy;
import com.entrust.toolkit.smproxy.ProxyOrder;
import com.entrust.toolkit.util.SecureStringBuffer;

/**
 * A sample application to demonstrate how to log in to a Roaming Digital
 * Identity via SMProxy. It accepts a roaming user id, a password, and a path to an
 * entrust.ini file, then logs in to that identity.
 * 
 * <dl>
 *     <dt>Note:</dt>
 *     <dd>This program requires an Entrust Authority Roaming Server 6.0 or higher
 *         to run.</dd>
 * </dl>
 */
public class Roaming
{

    /**
     * The main program.
     * 
     * @param argv
     *     The program arguments. Three program arguments are required:
     * <ol>
     * <li>roaming user id - the roaming user id to log in to</li>
     * <li>password - the user's password</li>
     * <li>entrust.ini - the path to the entrust.ini file containing the roaming configuration settings.</li>
     * </ol>
     */
    public static void main(String argv[])
    {
        if (argv.length != 3)
        {
            System.out.println("Usage: java com.entrust.toolkit.examples.roaming.Login <roaming user id> <password> <entrust.ini>");
            return;
        }

        // Extract the command line arguments and display them.
        String roamingUserId = argv[0];
        String password = argv[1];
        String iniFile = argv[2];

        System.out.println();
        System.out.println(
            "Roaming Sample application to demonstrate logging in to a Roaming Digital Identity via SMProxy");
        System.out.println("user id  = " + roamingUserId);
        System.out.println("password = " + password);
        System.out.println("entrust.ini = " + iniFile);

        // Start by creating a basic User object.
        User roamingUser = new User();
        try
        {
        	boolean smProxyEnabled = true;
        	
        	// There are three possible methods below to roam over an SMProxy Connection.  Uncomment
        	// the method you would like to use.
        	
        	// First option - using a roaming configuration object with SMProxy enabled.
        	RoamingConfiguration roamingConfiguration = new RoamingConfiguration(iniFile, smProxyEnabled);

            // Create a credential reader that will read the roaming credentials
            // for user roamingUserId. Roaming Configuration must contain all of the required
            // roaming and SMProxy configuration information.
//            RoamingCredentialReader reader = new RoamingCredentialReader(roamingUserId, roamingConfiguration);
            

            // Second option - using a configuration object that supports proxy order.
        	// Set a roaming configuration object that will try SMProxy first.
        	RoamingConfigurationWithSMProxy roamingConfigurationWithProxyOrder = 
        			new RoamingConfigurationWithSMProxy(iniFile, ProxyOrder.PROXYFIRST);

            // Create a credential reader that will read the roaming credentials
            // for user roamingUserId. Roaming Configuration must contain all of the required
            // roaming and SMProxy configuration information.
//            RoamingCredentialReader reader = new RoamingCredentialReader(roamingUserId, roamingConfigurationWithProxyOrder);
            
            
            // Third option - just specify the desire to use SMProxy in the constructor.            
        	// Create a credential reader that will read the roaming credentials
            // for user roamingUserId. iniFile must contain all of the required
            // roaming and SMProxy configuration information.
            RoamingCredentialReader reader = new RoamingCredentialReader(roamingUserId, iniFile, smProxyEnabled);
            

            // Set a credential writer so that the digital identity can be
            // updated if required. The method reader.getCredentialWriter()
            // returns a credential writer that will write back to the same
            // roaming user id and has the same roaming configuration information
            // as the reader. The following code would accomplish the same thing:
            //   CredentialWriter writer = new RoamingCredentialWriter(roamingUserId, iniFile);
            CredentialWriter writer = reader.getRoamingCredentialWriter();
            roamingUser.setCredentialWriter(writer);

            // Set connections so that the User object can contact the Directory and PKI.
            roamingUser.setConnections(iniFile);

            // now can do roaming login
            System.out.println("Roaming Login...");
            roamingUser.login(reader, new SecureStringBuffer(new StringBuffer(password)));
            System.out.println("Login complete");
            System.out.println();

            System.out.println("Getting the verification certificate...");
            System.out.println(roamingUser.getVerificationCertificate());

            System.out.println("Getting the encryption certificate...");
            System.out.println(roamingUser.getEncryptionCertificate());

            // Now do crypto operations.  The procedure is exactly the same as
            // for any other type of credentials.
            
            roamingUser.logout();
        }
        catch (Exception e)
        {
            // No error handling is performed to keep the code simpler.
            e.printStackTrace();
        }
    }
}
