//===========================================================================
//
// Copyright (c)  2001-2010 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.ssl.multithread;

import iaik.x509.X509Certificate;

import java.io.FileInputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import javax.net.ssl.SSLSocket;
import javax.net.ssl.X509KeyManager;
import javax.net.ssl.X509TrustManager;

import com.entrust.toolkit.User;
import com.entrust.toolkit.credentials.StreamProfileReader;
import com.entrust.toolkit.security.provider.Entrust;
import com.entrust.toolkit.security.provider.Initializer;
import com.entrust.toolkit.util.IniFile;
import com.entrust.toolkit.util.SecureStringBuffer;
import com.entrust.toolkit.x509.CertVerifier;
import com.entrust.toolkit.x509.jsse.JSSEUtils;
import com.entrust.toolkit.x509.jsse.JSSEX509KeyManager;
import com.entrust.toolkit.x509.jsse.JSSEX509TrustManager;

/**
 * This class implements the client part of the multithreading example. To try
 * it run both the server and the client. For all other documentation see the
 * {@link Server} class.
 * 
 * @see Server
 */
public class Client {

    private final static boolean DEBUG = true;

    // private final static int PORT = 4434;

    private final static long TIME_RENEGOTIATE = 1 * 1000;

    static IniFile properties = null;
    static String section = null;
    static String profileName = null;
    static String pwd = null;
    static String port = null;

    public static void main0(String args[]) throws Throwable {

        try {

            // Initialize the security providers
            Initializer.getInstance().setProviders(Initializer.MODE_NORMAL);

            // get the properties from https.ini
            properties = new IniFile("data/ssl/multithread.ini");

            // get Section
            section = new String("multithread");

            // get all the properties in order to run this Example
            profileName = properties.getString(section, "epfClient");
            pwd = properties.getString(section, "pwd");
            port = properties.getString(section, "port");

            // create a SecureStringBuffer Object for the password
            SecureStringBuffer password = new SecureStringBuffer(new StringBuffer(pwd), 0);
            User user = new User();

            // Log into an Entrust user, whose credentials we will use
            // for client authentication.
            System.out.println("log in");
            StreamProfileReader cr = new StreamProfileReader(new FileInputStream(profileName));
            user.login(cr, password);

            String serverHost;
            if (args.length == 0) {
                serverHost = "localhost";
            } else {
                serverHost = args[0];
            }

            javax.net.ssl.SSLContext context = JSSEUtils.getSSLContext();

            X509Certificate[] certChain = new X509Certificate[2];
            certChain[0] = user.getVerificationCertificate();
            certChain[1] = user.getCaCertificate();

            // Create a KeyManager
            X509KeyManager kmgr = new JSSEX509KeyManager(certChain, user.getSigningKey());

            // Create a TrustManager
            X509TrustManager tmgr;
            if (user.isLoggedIn()) {
                tmgr = new JSSEX509TrustManager(user);
            } else {
                X509Certificate rootCert = certChain[certChain.length - 1];
                CertVerifier certVerifier = new CertVerifier(rootCert, user.getDirectory(), user.getClientSettings());
                tmgr = new JSSEX509TrustManager(certVerifier);
            }

            // Initialize the SSLContext
            context.init(new X509KeyManager[] { kmgr }, new X509TrustManager[] { tmgr },
                    Entrust.getDefaultSecureRandomInstance());

            SSLSocket socket = (SSLSocket) context.getSocketFactory().createSocket(serverHost,
                    Integer.valueOf(port).intValue());

            String[] suites = new String[] { "SSL_DHE_RSA_WITH_DES_CBC_SHA", "SSL_DHE_RSA_WITH_3DES_EDE_CBC_SHA",
                    "TLS_DHE_RSA_WITH_AES_128_CBC_SHA", "TLS_DHE_RSA_WITH_AES_256_CBC_SHA" };

            socket.setEnabledCipherSuites(suites);

            System.out.println("Connected to " + socket.getInetAddress() + ":" + socket.getPort());
            socket.setSoTimeout(20 * 1000);

            socket.startHandshake();
            System.out.println("Initial handshake ok.");

            Feeder feeder = new Feeder(socket.getOutputStream(), "CFeeder");
            Reader reader = new Reader(socket.getInputStream(), "CReader");

            feeder.start();
            reader.start();

            int n = 0;

            String[] enabled = socket.getEnabledCipherSuites();
            List<String> cipherList = new ArrayList<String>();
            Collections.addAll(cipherList, enabled);

            try {

                while (n < 20) {
                    feeder.checkError();
                    reader.checkError();

                    String activeCipher = socket.getSession().getCipherSuite();
                    System.out.println("activeCipher:" + activeCipher);

                    for (int i = 0; i < cipherList.size(); i++) {
                        String item = cipherList.get(i);
                        if (item.equals(activeCipher)) {
                            cipherList.remove(i);
                        }
                    }
                    cipherList.add(activeCipher); // reordering, putting to
                                                  // bottom

                    socket.setEnabledCipherSuites((String[]) cipherList.toArray(new String[0]));

                    if (socket.isConnected()) {
                        System.out.println("client socket open");
                    } else {
                        System.out.println("client socket closed");
                    }

                    socket.startHandshake(); // force re-negotiation
                    System.out.println("client " + n + " handshake ok.");

                    n++;
                    System.out.print("client Renegotiates: " + n + ", Reads: " + reader.getCount() + ", Writes: "
                            + feeder.getCount() + "\r");
                    System.out.flush();

                    Thread.sleep(TIME_RENEGOTIATE);
                }

                // socket.close();

            } finally {
                feeder.setActive(false);
                reader.setActive(false);
                socket.close();
                System.out.println();
            }
        } catch (Exception e) {
            System.out.println("client exception: " + e.getMessage());
            System.out.println();
            e.printStackTrace();
        }
    }

    public static void main(String args[]) {
        try {
            main0(args);
            System.out.println("client Finished.");
        } catch (Throwable e) {
            System.err.println("Unexpected client exception:");
            e.printStackTrace(System.err);
        }
    }
}
