//===========================================================================
//
// Copyright (c)  2003-2010 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.threads;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.security.NoSuchAlgorithmException;

import com.entrust.toolkit.PKCS7DecodeStream;
import com.entrust.toolkit.PKCS7EncodeStream;
import com.entrust.toolkit.User;
import com.entrust.toolkit.credentials.CredentialReader;
import com.entrust.toolkit.credentials.CredentialWriter;
import com.entrust.toolkit.credentials.FilenameProfileReader;
import com.entrust.toolkit.credentials.FilenameProfileWriter;
import com.entrust.toolkit.credentials.UserStatus;
import com.entrust.toolkit.exceptions.PKCS7Exception;
import com.entrust.toolkit.exceptions.UserFatalException;
import com.entrust.toolkit.exceptions.UserKeyManagementException;
import com.entrust.toolkit.exceptions.UserNotLoggedInException;
import com.entrust.toolkit.util.SecureStringBuffer;

/**
 * Sample to show use of key management while multiple threads are active.
 * <p>
 * Applications that remain logged in for a long time generally need to
 * perform some key management, to check for certificate revocation,
 * pending certificate expiry, or forced key updates. Key management
 * should only be performed while no other threads have access to the User
 * object being used. Unexpected results may occur if one thread is updating
 * the keys while another is trying sign or verify data.
 * </p>
 * <p>While this sample uses PKCS7 for creating and reading messages, the
 * general structure applies to any multi-threaded application that uses a
 * <code>User</code> object.</p>
 * <p>This sample performs the following actions:</p>
 * <ul>
 *   <li>In the main thread, log in to the User object.</li>
 *   <li>Start a thread for key management, passing the logged in User.</li>
 *   <li>Start several worker threads that use the logged in User to create and verify PKCS7.</li>
 *   <li>Wait for a while, letting the worker and key management threads run.</li>
 *   <li>Interrupt the key management and worker threads.</li>
 *   <li>Log out.</li>
 * </ul>
 * <p>The key management thread does the following:</p>
 * <ul>
 *   <li>Obtain exclusive access to the User object.</li>
 *   <li>Perform any required key updates.</li>
 *   <li>Release exclusive access to the User object.</li>
 *   <li>Wait for a while, letting the worker threads run.</li>
 *   <li>Repeat until interrupted.</li>
 * </ul>
 * <p>The worker threads do the following:</p>
 * <ul>
 *   <li>Obtain shared read access to the User object.</li>
 *   <li>Create a PKCS7 message, then verify it.</li>
 *   <li>Release shared read access to the User object.</li>
 *   <li>Wait for a while, letting other threads run.</li>
 *   <li>Repeat until interrupted.</li>
 * </ul>
 * <p>Coordination of thread access to the User is done via the Coordinator class,
 * which allows any number of threads to read from the User at once, but only allows
 * one thread to write (while at the same time denying read access).</p>
 * 
 * <p>Usage:</p>
 * <pre>Pkcs7ThreadSample path_to_epf password time_to_run [inifile]</pre>
 * <dl>
 * <dt>path_to_epf</dt><dd>Path to EPF file. For example, /epffiles/MyEpfFile.epf</dd>
 * <dt>password</dt><dd>The password for the EPF. For example, MyPassword</dd>
 * <dt>time_to_run</dt><dd>The number of seconds for which to run. For example, 60</dd>
 * <dt>inifile</dt><dd>Optional entrust.ini file. For example, /winnt/entrust.ini</dd>
 * </dl>
 * <p>If no entrust.ini file is specified, the sample will still run, but it will not
 * be able to perform key management tasks.</p> 
 * <p>For example, from the examples directory</p> 
 * <pre>javac -sourcepath source -d classes -classpath ../lib/enttoolkit.jar source/com/entrust/toolkit/examples/threads/Pkcs7ThreadSample.java
 * java -classpath classes;../lib/enttoolkit.jar com.entrust.toolkit.examples.threads.Pkcs7ThreadSample
 *   data/userdata/RSAUser1.epf ~Sample7~ 30</pre>
 */
public class Pkcs7ThreadSample
{
    /**
     * The main program, which parses the command line and runs the threads.
     * 
     * @param args
     *     Program arguments, see class documentation for details on running this sample.
     */
    public static void main(String[] args) throws Exception
    {
        // First, parse the command line options.
        System.out.println("Sample to show usage of PKCS7 in multiple threads.");
        if (args.length != 3 && args.length != 4)
        {
            System.out.println(
                "Usage: ThreadSampleP7 <epf file> <password> <number of seconds to run> [<entrust.ini file>]");
            return;
        }

        String epfFile = args[0];
        String password = args[1];
        int secondsToRun = Integer.parseInt(args[2]);
        String iniFile = null;
        if (args.length == 4)
        {
            iniFile = args[3];
        }

        // Initialization of the User object should be done before passing it
        // off to any other threads.
        User user = new User();
        CredentialReader reader = new FilenameProfileReader(epfFile);

        System.out.println("Using Entrust identity: " + epfFile);
        System.out.println("Will run threads for approximately " + secondsToRun + " seconds");
        if (iniFile == null)
        {
            // No ini file supplied. The program will still run, but it won't
            // actually be able to do any key management.
            System.out.println("Not using an entrust.ini file, no key management possible");
        }
        else
        {
            // Ini file supplied, set connections to the PKI, and set a
            // credential writer so the new keys can be written back
            // to the epf file.
            System.out.println("Using entrust.ini file: " + iniFile);
            user.setConnections(iniFile);
            CredentialWriter writer = new FilenameProfileWriter(epfFile);
            user.setCredentialWriter(writer);
        }

        // Log in the user with the supplied password
        System.out.println("Logging in");
        user.login(reader, new SecureStringBuffer(new StringBuffer(password)));
        System.out.println("Logged in");

        // Create a new object that will be used to coordinate
        // the thread activity.
        Coordinator coordinator = new Coordinator(user);

        // Create and start the thread that will perform user key management
        // activities.
        UserManagementThread managementThread =
            new UserManagementThread(coordinator, "User management thread");
        managementThread.start();

        // Start some worker threads that will use the User to sign
        // and verify PKCS7 data.
        int numThreads = 5;
        WorkerThread[] workerThreads = new WorkerThread[numThreads];
        for (int i = 0; i < numThreads; ++i)
        {
            workerThreads[i] = new WorkerThread(coordinator, "Worker thread " + i);
            workerThreads[i].start();
        }

        // Have the main thread sleep for a while
        Thread.sleep(secondsToRun * 1000);

        // Main thread is awake again, it's time to shut the program down.
        // interrupt all of the worker threads, then join to them to ensure
        // they are correctly shut down.
        for (int i = 0; i < numThreads; ++i)
        {
            System.out.println("Stopping worker thread " + i);
            workerThreads[i].requestStop();
        }
        for (int i = 0; i < numThreads; ++i)
        {
            workerThreads[i].join();
        }

        // Stop the key management thread. As it may be sleeping for a long
        // time, interrupt it.
        System.out.println("Stopping key management thread");
        managementThread.requestStop();
        managementThread.interrupt();
        managementThread.join();

        // Finally, log out.
        if (user.isLoggedIn())
        {
            try
            {
                user.logout();
            }
            catch (UserNotLoggedInException e)
            {
            }
        }
    }
}

/**
 * This class acts as a way to coordinate activity between multiple threads
 * that want to use a shared resource - in this case a User object - for either
 * read-only or read/write access. It is simplistic in that it does not guarantee
 * that a thread requesting write access will ever get it, they may be
 * blocked indefinitely by reader threads. A more robust
 * mechanism should be used in practice.
 */
class Coordinator
{
    private User m_user;
    private int m_reading = 0;
    private boolean m_isLocked = false;

    /**
     * Constructor.
     * 
     * @param user
     *     the User object that will be shared between threads.
     */
    public Coordinator(User user)
    {
        m_user = user;
    }

    /**
     * Return the shared User object.
     * @return the shared User object.
     */
    public User getUser()
    {
        return m_user;
    }

    /**
     * Obtain exclusive write access to the shared User object.
     * May cause blocking. Every call to this method should be
     * matched by a call to releaseWriteAccess.
     * 
     * @throws InterruptedException
     *     if the thread is interrupted while waiting.
     */
    public synchronized void getWriteAccess() throws InterruptedException
    {
        if (m_isLocked)
        {
            wait();
        }
        m_isLocked = true;
    }

    /**
     * Release exclusive access to the shared User object.
     * Should only be called after getWriteAccess returns.
     *
     */
    public synchronized void releaseWriteAccess()
    {
        if (m_isLocked)
        {
            m_isLocked = false;
            notify();
        }
    }

    /**
     * Obtain read access to the shared User object.
     * May cause blocking. Every call to this method should be
     * matched by a call to releaseReadAccess.
     * 
     * @throws InterruptedException
     *     if the thread is interrupted while waiting.
     */
    public synchronized void getReadAccess() throws InterruptedException
    {
        if (m_reading == 0)
        {
            // Prevent any other thread obtaining write access.
            getWriteAccess();
        }
        ++m_reading;
    }

    /**
     * Release read access to the shared User object.
     * Should only be called after getReadAccess returns.
     *
     */
    public synchronized void releaseReadAccess()
    {
        --m_reading;
        if (m_reading == 0)
        {
            // No threads are reading, allow write access.
            releaseWriteAccess();
        }
    }
}

/**
 * Thread for user key management. Only one of these threads
 * should be created for each User object.
 */
class UserManagementThread extends Thread
{
    private Coordinator m_coordinator;
    private boolean m_isStopRequested;

    /**
     * Constructor.
     * 
     * @param coordinator
     *     the object that will coordinate access to the shared User object.
     * @param threadName
     *     the name of the thread.
     */
    public UserManagementThread(Coordinator coordinator, String threadName)
    {
        super(threadName);
        m_coordinator = coordinator;
        m_isStopRequested = false;
    }

    /**
     * Request that the thread stop running.
     */
    public void requestStop()
    {
        m_isStopRequested = true;
    }

    /**
     * The main thread procedure. 
     */
    public void run()
    {
        // Run until the thread is asked to stop.
        while (m_isStopRequested == false)
        {
            // Check for the key update
            try
            {
                if (!checkForKeyUpdate())
                {
                    System.out.println("Problem with logged in User!");
                    break;
                }
            }
            catch (InterruptedException e)
            {
                break;
            }

            // Sleep for a while. The number here is only one second, in practice
            // this check would only need to be done occasionally, perhaps once per
            // hour or once per day. The frequency will depend upon the application,
            // and how critical it is to have the latest keys.
            try
            {
                Thread.sleep(1000);
            }
            catch (InterruptedException e)
            {
                break;
            }
        }
        System.out.println("User management thread stopped.");
    }

    /**
     * Check to see if key updates are required, and if so,
     * perform them.
     * 
     * @return
     *     true if everything runs successfully, false if there is
     * some failure.
     * 
     * @throws InterruptedException
     *     if the thread is interrupted while waiting for write access to
     * the user.
     */
    private boolean checkForKeyUpdate() throws InterruptedException
    {
        // Get exclusive write access to the user.
        m_coordinator.getWriteAccess();

        try
        {
            // Do the required key management operations. This contacts the
            // Security Manager, so it can be considered an expensive check
            // to perform. A less expensive check can be done with 
            // User.keyUpdatesRequired(), which will only check for key updates
            // due to pending key and certificate expiry. It returns an array of
            // certificates which can be updated with User.updateUserKeys().
            m_coordinator.getUser().doRequiredKeyManagement();
            UserStatus status = m_coordinator.getUser().getStatus();

            // It may be important to the application to examine which
            // keys, if any, were updated. For this example, we'll
            // merely display the status.
            System.out.println("Status after key management:");
            System.out.println(status);
        }
        catch (UserNotLoggedInException e)
        {
            // Should never get here, if we do it likely indicates a programming
            // error.
            System.out.println("User is not logged in!");
            return false;
        }
        catch (UserKeyManagementException e)
        {
            // Whether this exception is fatal or not will depend on the application.
            // it may simply mean that the CA was not reachable. If this happens
            // repeatedly, it may indicate a problem with the CA that needs to be
            // investigated.
            System.out.println("Non-fatal error during key management: " + e.getMessage());
        }
        catch (UserFatalException e)
        {
            // A serious problem occurred during key management.
            // The User running the application should probably no longer be allowed
            // to perform crypto operations.
            System.out.println("Fatal error occurred during key management: " + e.getMessage());
            return false;
        }
        finally
        {
            // release write access
            m_coordinator.releaseWriteAccess();
        }

        return true;
    }
}

/**
 * Thread that uses keys from the User object to do PKCS7 work.
 * The work done is trivial, a simple message is signed and then verified.
 * This is for illustrative purposes of how a server with multiple threads
 * for processing PKCS7 might work.
 */
class WorkerThread extends Thread
{
    private Coordinator m_coordinator;
    private boolean m_isStopRequested;

    /**
     * Constructor.
     * 
     * @param coordinator
     *     the object that will coordinate access to the shared User object.
     * @param threadName
     *     the name of the thread.
     */
    public WorkerThread(Coordinator coordinator, String threadName)
    {
        super(threadName);
        m_coordinator = coordinator;
        m_isStopRequested = false;
    }

    /**
     * Request that the thread stop running.
     */
    public void requestStop()
    {
        m_isStopRequested = true;
    }

    /**
     * The main thread procedure. Runs until interrupted.
     * Obtains read access to the User, does some PKCS7 work
     */
    public void run()
    {
        while (m_isStopRequested == false)
        {
            // Get read access to the User object.
            try
            {
                m_coordinator.getReadAccess();
            }
            catch (InterruptedException e)
            {
                break;
            }

            boolean interrupted = doWork();

            // Make sure to release read access before doing anything else.
            m_coordinator.releaseReadAccess();

            if (interrupted)
            {
                break;
            }

            // Simulate waiting for another request by sleeping.
            try
            {
                Thread.sleep(500);
            }
            catch (InterruptedException e)
            {
                break;
            }
        }
        System.out.println(getName() + " stopped.");
    }

    /**
     * Do some work. While the work done is not actually useful, this
     * indicates how User objects would be shared between multiple
     * worker threads using PKCS7.
     * 
     * @return true if the work was interrupted, false otherwise.
     */
    private boolean doWork()
    {
        System.out.println(getName() + " running");
        ByteArrayOutputStream os = new ByteArrayOutputStream();
        try
        {
            PKCS7EncodeStream p7e =
                new PKCS7EncodeStream(m_coordinator.getUser(), os, PKCS7EncodeStream.SIGN_AND_ENCRYPT);

            p7e.write("Message to protect".getBytes());
            p7e.close();
        }
        catch (PKCS7Exception e)
        {
            // Take action appropriate to the application
            System.out.println(getName() + ": Could not create P7: " + e.getMessage());
            if (e.getInnerException() instanceof InterruptedException)
            {
                return true;
            }
        }
        catch (IOException e)
        {
            // Some other IO Exception happened, take appropriate action.
            System.out.println(getName() + ": Could not write P7 due to an IO Exception: " + e.getMessage());
        }

        try
        {
            InputStream is = new ByteArrayInputStream(os.toByteArray());
            PKCS7DecodeStream p7d = new PKCS7DecodeStream(m_coordinator.getUser(), is);
            byte[] unprotectedData = new byte[1024];
            int bytesRead = p7d.read(unprotectedData);
            while (bytesRead != -1)
            {
                // Do something with unprotected data
                // Read more data
                bytesRead = p7d.read(unprotectedData);
            }
        }
        catch (NoSuchAlgorithmException e)
        {
            // Take action appropriate to the application
            System.out.println(getName() + ": Could not read P7, unknown algorithm: " + e.getMessage());
        }
        catch (PKCS7Exception e)
        {
            // Take action appropriate to the application
            System.out.println(getName() + ": Could not read P7: " + e.getMessage());
            if (e.getInnerException() instanceof InterruptedException)
            {
                return true;
            }
        }
        catch (IOException e)
        {
            // Some other IO Exception happened, take appropriate action.
            System.out.println(getName() + ": Could not read P7 due to an IO Exception: " + e.getMessage());
        }
        return false;
    }
}
