//===========================================================================
//
// Copyright (c)  2003-2010 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.timestamping;

import iaik.asn1.ASN1Object;
import iaik.asn1.DerCoder;
import iaik.asn1.ObjectID;
import iaik.asn1.structures.Attribute;
import iaik.asn1.structures.ChoiceOfTime;
import iaik.utils.Util;
import iaik.x509.X509Certificate;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.util.Date;
import java.util.Properties;

import javax.activation.MailcapCommandMap;
import javax.mail.Message;
import javax.mail.Session;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeMessage;

import com.entrust.toolkit.User;
import com.entrust.toolkit.credentials.FilenameProfileReader;
import com.entrust.toolkit.security.provider.Initializer;
import com.entrust.toolkit.timestamp.HttpTimeStampTransport;
import com.entrust.toolkit.timestamp.TimeStampClient;
import com.entrust.toolkit.timestamp.TimeStampVerifier;
import com.entrust.toolkit.util.SecureStringBuffer;
import com.entrust.toolkit.x509.CertVerifier;

/**
 * This class contains sample code for how time-stamped PKCS7, CMS, and S/MIME 
 * signed data can be created and then verified and validated.
 * 
 * <P>
 * When verifying signed data, a check is done to ensure that the data has not
 * been modified since it was signed.  When validating the signed data, the 
 * time-stamp is verified and validated, and the signer certificate is then
 * validated at the time indicated in the time-stamp.
 * </P>
 */
public class TimeStampedSignedData {

    /******************************************************************************/
    /*                               SAMPLE PROGRAM                               */
    /******************************************************************************/

    public static void main(String[] args) {
        // Check arguments
        int size = args.length;
        if (size < 3 || size > 4) {
            System.out.println("Usage: TimeStampedSignedData <user's epf> <user's password> <tsa's url>");
            System.out.println(
                "Usage: TimeStampedSignedData <user's epf> <user's password> <tsa's url> <tsa's base64 encoded cert>");
            System.exit(1);
        }

        // Parse the TSA certificate if it was provided
        X509Certificate tsaCert = null;
        if (size == 4) {
            Initializer.getInstance().setProviders(Initializer.MODE_NORMAL);
            try {
                tsaCert = new X509Certificate(Util.Base64Decode(Util.toASCIIBytes(args[3])));
            } catch (Exception e) {
                System.out.println("Invalid TSA certificate");
                e.printStackTrace();
            }
        }

        try {
            // Create and login the user (offline)
            System.out.print("Logging in user... ");
            m_user = new User();
            FilenameProfileReader reader = new FilenameProfileReader(args[0]);
            SecureStringBuffer password = new SecureStringBuffer(new StringBuffer(args[1]));
            m_user.login(reader, password);
            System.out.println("DONE\n");

            // Add the TSA certificate as a trusted certificate
            // This must be done if the user does not share a trust relationship
            // with the TSA by some other means such as cross-certification. 
            CertVerifier certVerifier = (CertVerifier) m_user.getCertVerifier();
            if (tsaCert != null) {
                System.out.print("Adding TSA certificate as a trusted certificate... ");
                m_user.getClientSettings().setAllowCAPAB(true);
                certVerifier.getCertificateStore().addTrustedCertificate(tsaCert);
                System.out.println("DONE\n");
            }

            // Create the time-stamp client            
            m_timeStampClient =
                new TimeStampClient(new HttpTimeStampTransport(args[2]), new TimeStampVerifier(certVerifier));

            // Create different types of signed data
            iaik.pkcs.pkcs7.SignedDataStream pkcs7SignedData = createPKCS7SignedData();
            iaik.cms.SignedDataStream cmsSignedData = createCMSSignedData();
            iaik.security.smime.SignedContent smimeV2SignedContent = createSMIMEV2SignedContent();
            iaik.smime.SignedContent smimeV3SignedContent = createSMIMEV3SignedContent();

            // Verify PKCS7 signed data
            System.out.println("\nVerifying and validating the PKCS7 signed data");
            size = pkcs7SignedData.getSignerInfos().length;
            System.out.println("  signed data contains " + size + " signatures");
            for (int i = 0; i < size; i++) {
                System.out.print("  verifying and validating signature at index '" + i + "'... ");
                pkcs7SignedData.verifyAndValidate(i, certVerifier);
                System.out.println("SUCCESS");
            }

            // Verify CMS signed data
            System.out.println("\nVerifying and validating the CMS signed data");
            size = cmsSignedData.getSignerInfos().length;
            System.out.println("  signed data contains " + size + " signatures");
            for (int i = 0; i < size; i++) {
                System.out.print("  verifying and validating signature at index '" + i + "'... ");
                cmsSignedData.verifyAndValidate(i, certVerifier);
                System.out.println("SUCCESS");
            }

            // Verify S/MIME v2 signed data
            System.out.println("\nVerifying and validating the S/MIME v2 signed data");
            smimeV2SignedContent.verifyAndValidate(certVerifier);
            size = smimeV3SignedContent.getSignerInfos().length;
            System.out.println("  signed data contains " + size + " signatures");
            System.out.print("  verifying and validating signature at index '0'... ");
            System.out.println("SUCCESS");

            // Verify S/MIME v3 signed data
            System.out.println("\nVerifying and validating the S/MIME v3 signed data");
            size = smimeV3SignedContent.getSignerInfos().length;
            System.out.println("  signed data contains " + size + " signatures");
            for (int i = 0; i < size; i++) {
                System.out.print("  verifying and validating signature at index '" + i + "'... ");
                smimeV3SignedContent.verifyAndValidate(i, certVerifier);
                System.out.println("SUCCESS");
            }
        } catch (Exception e) {
            System.out.println("FAILED");
            e.printStackTrace();
        }
    }

    /******************************************************************************/
    /*                                 STATIC DATA                                */
    /******************************************************************************/

    /** The user that will be creating and verifying the signed data. */
    private static User m_user;

    /** Time-stamp client; used to request time-stamps from the TimeStamp 
     *  Authority. */
    private static TimeStampClient m_timeStampClient;

    /******************************************************************************/
    /*                                CONSTRUCTORS                                */
    /******************************************************************************/

    /**
     * The default constructor.
     * <P></P>
     */
    private TimeStampedSignedData() {
    }

    /******************************************************************************/
    /*                           INTERNAL HELPER METHODS                          */
    /******************************************************************************/

    /**
     * Create some PKCS7 signed data that contains a time-stamp
     * <P></P>
     * 
     * @return
     *      the PKCS7 signed data 
     * 
     * @exception Exception
     *      thrown if an error occurs
     */
    private static iaik.pkcs.pkcs7.SignedDataStream createPKCS7SignedData() throws Exception {
        System.out.print("Creating PKCS7 signed data... ");

        // Create the PKCS7 SignedData
        iaik.pkcs.pkcs7.SignedDataStream signedData =
            new iaik.pkcs.pkcs7.SignedDataStream(
                new ByteArrayInputStream("test".getBytes()),
                iaik.pkcs.pkcs7.SignedDataStream.IMPLICIT);

        // Include relevant certificates in the SignedContent
        X509Certificate[] caCerts = m_user.getCaCertificateChain();
        int size = caCerts.length + 1;
        X509Certificate[] certificates = new X509Certificate[size];
        certificates[0] = m_user.getVerificationCertificate();
        System.arraycopy(caCerts, 0, certificates, 1, size - 1);
        signedData.setCertificates(certificates);

        // Set the signer and request that the signature be time-stamped        
        iaik.pkcs.pkcs7.IssuerAndSerialNumber issuerAndSerialNumber =
            new iaik.pkcs.pkcs7.IssuerAndSerialNumber(m_user.getVerificationCertificate());
        iaik.pkcs.pkcs7.SignerInfo signerInfo =
            new iaik.pkcs.pkcs7.SignerInfo(
                issuerAndSerialNumber,
                iaik.asn1.structures.AlgorithmID.sha,
                m_user.getSigningKey());
        Attribute[] attributes = new Attribute[2];
        attributes[0] = new Attribute(ObjectID.contentType, new ASN1Object[] { ObjectID.cms_data });
        attributes[1] = new Attribute(ObjectID.signingTime, new ASN1Object[] { new ChoiceOfTime().toASN1Object()});
        signerInfo.setAuthenticatedAttributes(attributes);
        signerInfo.requestTimeStamp(m_timeStampClient);
        signedData.addSignerInfo(signerInfo);

        // Encode the PKCS7 data (send)
        ByteArrayOutputStream encodedPkcs7DataStream = new ByteArrayOutputStream();
        DerCoder.encodeTo(signedData.toASN1Object(), encodedPkcs7DataStream);

        // Decode the PKCS7 data (receive)
        signedData = new iaik.pkcs.pkcs7.SignedDataStream(new ByteArrayInputStream(encodedPkcs7DataStream.toByteArray()));
        InputStream dataIs = signedData.getInputStream();
        byte[] buf = new byte[1024];
        int r;
        while ((r = dataIs.read(buf)) > 0) {
            // do something useful
        }
        System.out.println("DONE");
        return signedData;
    }

    /**
     * Create some CMS signed data that contains a time-stamp
     * <P></P>
     * 
     * @return
     *      the CMS signed data 
     * 
     * @exception Exception
     *      thrown if an error occurs
     */
    private static iaik.cms.SignedDataStream createCMSSignedData() throws Exception {
        System.out.print("Creating CMS signed data... ");

        // Create the CMS SignedData
        iaik.cms.SignedDataStream signedData =
            new iaik.cms.SignedDataStream(
                new ByteArrayInputStream("test".getBytes()),
                iaik.cms.SignedDataStream.IMPLICIT);

        // Include relevant certificates in the SignedContent
        X509Certificate[] caCerts = m_user.getCaCertificateChain();
        int size = caCerts.length + 1;
        X509Certificate[] certificates = new X509Certificate[size];
        certificates[0] = m_user.getVerificationCertificate();
        System.arraycopy(caCerts, 0, certificates, 1, size - 1);
        signedData.setCertificates(certificates);

        // Set the signer and request that the signature be time-stamped        
        iaik.cms.CertificateIdentifier issuerAndSerialNumber =
            new iaik.cms.IssuerAndSerialNumber(m_user.getVerificationCertificate());
        iaik.cms.SignerInfo signerInfo =
            new iaik.cms.SignerInfo(
                issuerAndSerialNumber,
                iaik.asn1.structures.AlgorithmID.sha,
                m_user.getSigningKey());
        Attribute[] attributes = new Attribute[2];
        attributes[0] = new Attribute(ObjectID.contentType, new ASN1Object[] { ObjectID.cms_data });
        attributes[1] = new Attribute(ObjectID.signingTime, new ASN1Object[] { new ChoiceOfTime().toASN1Object()});
        signerInfo.setSignedAttributes(attributes);
        signerInfo.requestTimeStamp(m_timeStampClient);
        signedData.addSignerInfo(signerInfo);

        // Encode the CMS data (send)
        ByteArrayOutputStream encodedCmsDataStream = new ByteArrayOutputStream();
        DerCoder.encodeTo(signedData.toASN1Object(), encodedCmsDataStream);

        // Decode the CMS data (receive)
        signedData = new iaik.cms.SignedDataStream(new ByteArrayInputStream(encodedCmsDataStream.toByteArray()));
        InputStream dataIs = signedData.getInputStream();
        byte[] buf = new byte[1024];
        int r;
        while ((r = dataIs.read(buf)) > 0) {
            // do something useful
        }
        System.out.println("DONE");
        return signedData;
    }

    /**
     * Create some S/MIME V2 signed data that contains a time-stamp
     * <P></P>
     * 
     * @return
     *      the S/MIME V2 signed data 
     * 
     * @exception Exception
     *      thrown if an error occurs
     */
    private static iaik.security.smime.SignedContent createSMIMEV2SignedContent() throws Exception {
        System.out.print("Creating S/MIME V2 signed content... ");

        MailcapCommandMap mccm = (MailcapCommandMap) MailcapCommandMap.getDefaultCommandMap();
        mccm.addMailcap("application/x-pkcs7-mime; ; x-java-content-handler=iaik.security.smime.encrypted_content");
        MailcapCommandMap.setDefaultCommandMap(mccm);

        // Create the SMIMEv2 SignedContent
        iaik.security.smime.SignedContent signedContent =
            new iaik.security.smime.SignedContent(true, iaik.security.smime.SignedContent.SIGNED_DATA);
        signedContent.setText("test");

        // Include relevant certificates in the SignedContent
        X509Certificate[] caCerts = m_user.getCaCertificateChain();
        int size = caCerts.length + 1;
        X509Certificate[] certificates = new X509Certificate[size];
        certificates[0] = m_user.getVerificationCertificate();
        System.arraycopy(caCerts, 0, certificates, 1, size - 1);
        signedContent.setCertificates(certificates);

        // Set the signer and request that the signature be time-stamped
        signedContent.setSigner(m_user.getSigningKey(), m_user.getVerificationCertificate());
        signedContent.requestTimeStamp(m_timeStampClient);

        // Create a MIME message that contains the SignedContent
        Properties props = System.getProperties();
        Session session = Session.getDefaultInstance(props, null);
        MimeMessage mimeMsg = new MimeMessage(session);
        mimeMsg.setFrom(new javax.mail.internet.InternetAddress("Java.Toolkit@entrust.com"));
        mimeMsg.setRecipients(Message.RecipientType.TO, InternetAddress.parse("Java.Toolkit@entrust.com", false));
        mimeMsg.setSentDate(new Date());
        mimeMsg.setContent(signedContent, signedContent.getContentType());
        signedContent.setHeaders(mimeMsg);
        mimeMsg.saveChanges();

        // Write the MIME message (encode/send) 
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        mimeMsg.writeTo(baos);
        baos.flush();
        baos.close();
        byte[] encoded = baos.toByteArray();

        // Read the MIME message (receive/decode)
        ByteArrayInputStream bais = new ByteArrayInputStream(encoded);
        mimeMsg = new MimeMessage(session, bais);
        signedContent = (iaik.security.smime.SignedContent) mimeMsg.getContent();
        System.out.println("DONE");
        return signedContent;
    }

    /**
     * Create some S/MIME V3 signed data that contains a time-stamp
     * <P></P>
     * 
     * @return
     *      the S/MIME V3 signed data 
     * 
     * @exception Exception
     *      thrown if an error occurs
     */
    private static iaik.smime.SignedContent createSMIMEV3SignedContent() throws Exception {
        System.out.print("Creating S/MIME V3 signed content... ");

        MailcapCommandMap mccm = (MailcapCommandMap) MailcapCommandMap.getDefaultCommandMap();
        mccm.addMailcap("application/pkcs7-mime; ; x-java-content-handler=iaik.smime.encrypted_content");
        MailcapCommandMap.setDefaultCommandMap(mccm);

        // Create the SMIMEv3 SignedContent
        iaik.smime.SignedContent signedContent =
            new iaik.smime.SignedContent(true, iaik.smime.SignedContent.SIGNED_DATA);
        signedContent.setText("test");

        // Include relevant certificates in the SignedContent
        X509Certificate[] caCerts = m_user.getCaCertificateChain();
        int size = caCerts.length + 1;
        X509Certificate[] certificates = new X509Certificate[size];
        certificates[0] = m_user.getVerificationCertificate();
        System.arraycopy(caCerts, 0, certificates, 1, size - 1);
        signedContent.setCertificates(certificates);

        // Add the signer and request that the signature be time-stamped
        signedContent.addSigner(m_user.getSigningKey(), m_user.getVerificationCertificate());
        signedContent.requestTimeStamp(m_timeStampClient);

        // Create a MIME message that contains the SignedContent
        Properties props = System.getProperties();
        Session session = Session.getDefaultInstance(props, null);
        MimeMessage mimeMsg = new MimeMessage(session);
        mimeMsg.setFrom(new javax.mail.internet.InternetAddress("Java.Toolkit@entrust.com"));
        mimeMsg.setRecipients(Message.RecipientType.TO, InternetAddress.parse("Java.Toolkit@entrust.com", false));
        mimeMsg.setSentDate(new Date());
        mimeMsg.setContent(signedContent, signedContent.getContentType());
        signedContent.setHeaders(mimeMsg);
        mimeMsg.saveChanges();

        // Write the MIME message (encode/send) 
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        mimeMsg.writeTo(baos);
        baos.flush();
        baos.close();
        byte[] encoded = baos.toByteArray();

        // Read the MIME message (receive/decode)
        ByteArrayInputStream bais = new ByteArrayInputStream(encoded);
        mimeMsg = new MimeMessage(session, bais);
        signedContent = (iaik.smime.SignedContent) mimeMsg.getContent();
        System.out.println("DONE");
        return signedContent;
    }
}
