//===========================================================================
//
// Copyright (c)  2000-2010 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.xml.sign;

import java.security.PrivateKey;

import org.w3c.dom.Document;

import iaik.ixsil.algorithms.CanonicalizationAlgorithmImplCanonicalXML;
import iaik.ixsil.algorithms.DigestAlgorithmImplSHA1;
import iaik.ixsil.core.Signer;
import iaik.ixsil.core.SignerReference;
import iaik.ixsil.core.SignerSignedInfo;
import iaik.ixsil.init.IXSILInit;
import iaik.ixsil.keyinfo.SignerKeyManager;
import iaik.ixsil.util.URI;
import iaik.x509.X509Certificate;

import com.entrust.toolkit.KeyAndCertificateSource;
import com.entrust.toolkit.User;
import com.entrust.toolkit.examples.xml.utils.Utils;
import com.entrust.toolkit.util.SecureStringBuffer;

/**
 * This class shows how to create a detached XML signature. A detached
 * signature is performed over content external to the Signature element,
 * so the signature is "detached" from the content it signs. This is a basic
 * example, so the signature contains a single Reference to one external
 * resource.  The resource may contain arbitrary binary data, not necessarily
 * XML content.
 * <p>
 * The XML signature contains public key information that includes the
 * signer's public key certificate and X509 subject name.
 */
public class Sign
{
  /**
   * A source of keys and certificates
   */
  private KeyAndCertificateSource m_source ;

  /**
   * The main method.
   */
  public static void main(String[] args) throws Exception
  {
    Sign signatureExample = new Sign(args);

    // Create the detached signature
    Document signatureDOMDoc = signatureExample.createSignature(args[3]);

    // Write the detached signature to a file
    Utils.writeDocumentToFile(args[4], signatureDOMDoc);

    System.out.println("Done");
  }

  /**
   * Constructor for this example.
   *
   * Logs into an Entrust user and initializes the IXSIL library.
   */
  public Sign(String[] args) throws Exception
  {
    if (args.length != 5)
    {
      System.out.println("\nPlease specify five command line parameters:");
      System.out.println("  (1) Path and filename of an Entrust user profile");
      System.out.println("  (2) Profile password");
      System.out.println("  (3) The URL of an IXSIL initialization properties file\n\t" +
                                "e.g. \"file:/C:/etjava/examples/xml/init/properties/init.properties\"\n\t" +
                                "e.g. \"init.properties\"\n\t" +
                                "e.g. \"http://hostname/init.properties\"");
      System.out.println("  (4) URI of the data to be signed, e.g. \"file:/c:/test/binarydata.dat\"");
      System.out.println("  (5) Path and filename for the resulting XML signature file");
      System.exit(0);
    }

    // Need the signing key and certificate to be in a KeyAndCertificateSource wrapper.
    m_source = new KeyAndCertificateSource();
    
    // If you using a keystore such as P12, PFX, Entrust EPF, or java.security.KeyStore,
    // then log in here, to get access to its signing key and certificate.
    User user = Utils.login(args[0], new SecureStringBuffer(new StringBuffer(args[1])));
    PrivateKey pk = user.getSigningKey();
    X509Certificate cert = user.getVerificationCertificate();
    
    // In any case, provide a signing key and verification certificate from whatever source you use.
    m_source.setSigningInfo(pk, cert);

    // The properties file is specified by a URI.  You can provide the name of the
    // file, e.g 'init.properties', to the URI constructor.  In that case, the folder
    // that contains init.properties must be in the classpath.
    // Please refer to the comments in the 'init.properties' sample.
    URI initProps = new URI(args[2]);

    // Initialize IXSIL
    System.out.println("Initializing IXSIL properties from \"" + initProps + "\"...");
    IXSILInit.init(initProps);
  }

  /**
   * Creates a detached signature.
   *
   * @param uriTobeSigned A URI that references data to be signed, not necessarily XML content
   * @return A DOM document containing the detached XML Signature
   */
  public Document createSignature(String uriTobeSigned)
  throws Exception
    {
        // Create a Signer
        Signer signer = new Signer(new URI(uriTobeSigned));

        // Set its ID attribute
        signer.getSignature().setId("Signature001");

        // Get an interface for dsig SignedInfo
        SignerSignedInfo signedInfo = signer.getSignature().getSignerSignedInfo();

        // Configure a canonicalization algorithm
        CanonicalizationAlgorithmImplCanonicalXML c14nAlg = new CanonicalizationAlgorithmImplCanonicalXML();

        // Set canonicalization algorithm
        signedInfo.setCanonicalizationAlgorithm(c14nAlg);

        // Configure a signature algorithm
        Utils.configureAlgorithm(m_source, signedInfo);

        // Create a Reference
        SignerReference reference = signedInfo.createReference();

        // Configure the Reference for an detached signature.  Its URI
        // is an absolute URI which references the document to be signed.
        reference.setURI(new URI(uriTobeSigned));

        // Configure a digest algorithm
        DigestAlgorithmImplSHA1 digestAlg = new DigestAlgorithmImplSHA1();

        // Set digest algorithm
        reference.setDigestAlgorithm(digestAlg);

        // Add Reference to signature
        signedInfo.addReference(reference);

        // Recommended -- Attach the user's X.509 verification certificate to the Signature
        int x509Data = Utils.X509DATA_CERTIFICATE;

        // Optional -- Attach the X.509 certificate's subject name and/or issuer and serial number
        // x509Data |= Utils.X509DATA_SUBJECT_NAME | Utils.X509DATA_ISSUER_SERIAL;

        // Set the X509Data
        SignerKeyManager keyManager = Utils.setX509KeyInfo(x509Data,
                                                           signer,
                                                           m_source.getVerificationCertificate(),
                                                           "KeyInfo001");

        // For MS .NET.
        //
        // The XML dsig verifier in .NET (the .NET Framework SignedXml class) does not
        // recognize X.509 certificates, so the signer must attach raw <KeyValue> in
        // place of <X509Data>.  However, a signature can be validated on the basis
        // of <KeyValue> only if the verifier already possesses a trusted copy of the
        // signer's key.  Transporting a public key via a <KeyValue> element is not
        // secure (even if the <KeyInfo> is signed), so your verifier must take
        // precautions -- please read the javadoc for the methods 'setKeyValue(...)'
        // and especially 'validatePublicKey(...)' in 'examples/xml/utils/Utils.java'.
        //
        // If you need to support MS .NET, use the following in place of setX509KeyInfo(...) above.
        // SignerKeyManager keyManager = Utils.setKeyValue(signer,
        //                                        m_user.getVerificationCertificate().getPublicKey(),
        //                                        "KeyInfo001");

        // Optional -- Sign the KeyInfo element
        Utils.signKeyInfo(signedInfo, keyManager);

        // Apply the transforms in each Reference and ensure the resulting XML content is not empty.
        Utils.processTransforms(signer.getSignature(),
                                false /* in general cannot display because it might be binary data */);

        // Compute the signature value
        signer.getSignature().sign();

        // Get resulting XML document
        return signer.toDocument();
    }
}