//===========================================================================
//
// Copyright (c)  2000-2010 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.xml.sign;

import java.io.InputStream;

import org.w3c.dom.Document;

import iaik.ixsil.core.Verifier;
import iaik.ixsil.core.VerifierReference;
import iaik.ixsil.core.VerifierSignature;
import iaik.ixsil.exceptions.URIException;
import iaik.ixsil.init.IXSILConstants;
import iaik.ixsil.init.IXSILInit;
import iaik.ixsil.util.ExternalReferenceResolverImpl;
import iaik.ixsil.util.URI;

import com.entrust.toolkit.KeyAndCertificateSource;
import com.entrust.toolkit.User;
import com.entrust.toolkit.examples.xml.utils.Utils;
import com.entrust.toolkit.util.SecureStringBuffer;
import com.entrust.toolkit.x509.CertVerifier;

/**
 * This class shows how to verify an XML signature.
 */
public class Verify
{
    
  /**
   * The verification procedure that this sample will demonstrate
   */    
  static public final int DEMO_METHOD_1 = 1;
  static public final int DEMO_METHOD_2 = 2;
  static public final int DEMO_METHOD_3 = 3;
    
  /**
   * Provides the CertVerifier to the IXSIL Trustmanager interface
   */
  private KeyAndCertificateSource m_source ;
  
  /**
   * Saves the schemaLocation argument to the Verifier constructor
   */
  String m_schemaLocations = null ;

  /**
   * The main method.
   */
  public static void main(String[] args) throws Exception
  {
    // Debugging
    Utils.printSystemProperties();
    System.out.println();
    
    // Instantiate this example
    Verify verifyExample = new Verify(args);

    // Optional -- display the XML
    Utils.displayURL(args[3]);

    // Verify an XML Signature using one of the three methods
    verifyExample.validateSignature(args[3], DEMO_METHOD_1);
    System.out.println("Done");
  }

  /**
   * Constructor for this example. Initializes IXSIL with a Trustmanager.
   */
  public Verify(String[] args) throws Exception
  {
    if (args.length != 4)
    {
      System.out.println("\nPlease specify four command line parameters:");
      System.out.println("  (1) Path and filename of an Entrust user profile");
      System.out.println("  (2) Profile password");
      System.out.println("  (3) The URL of an IXSIL initialization properties file\n\t" +
                                "e.g. \"file:/C:/etjava/examples/xml/init/properties/init.properties\"\n\t" +
                                "e.g. \"init.properties\"\n\t" +
                                "e.g. \"http://hostname/init.properties\"");
      System.out.println("  (4) URI of the XML signature to be validated\n\t" +
                                "e.g. \"file:/C:/etjava/examples/xml/signature.xml\"\n\t" +
                                "e.g. \"file:signature.xml\"\n\t" +
                                "e.g. \"http://hostname/signature.xml\"");
      System.exit(0);
    }

    // X509 certificates are validated by a CertVerifier instance.  If you already
    // have a root of trust certificate, you can create a CertVerifier.
    //
    // CertVerifier certverifier = new CertVerifier(rootOfTrustX509Certificate, jndidirectory, null));
    //
    // Otherwise, if you are using a keystore to protect the root-of-trust certificate,
    // log into it and get the ROT from the User object.  The keystore can be a P12, PFX, 
    // Entrust EPF, or java.security.KeyStore.
    //
    User user = Utils.login(args[0], new SecureStringBuffer(new StringBuffer(args[1])));
    CertVerifier certverifier = (CertVerifier) user.getCertVerifier();    
    
    // Either way, initialize the the XML Trustmanager through a KeyAndCertificateSource wrapper.
    m_source = new KeyAndCertificateSource(certverifier);

    // Initialize the IXSIL library
    String init = args[2];
    System.out.println("Initializing IXSIL properties from \"" + init + "\"...");

    // The properties file is specified by a URI.  You can provide the name of the
    // file, e.g 'init.properties', to the URI constructor.  In that case, the folder
    // that contains init.properties must be in the classpath.
    // Please refer to the comments in the 'init.properties' sample.
    IXSILInit.init(new URI(init));
    System.out.println("IXSIL has been initialized.");
    
    // Initialize IXSIL's schema location parameter, required when
    // constructing the Verifier from a DOM document instance,
    // as in Method (1), where your Document instance must be 
    // created with a validating parse.  See also Utils.createDOMDocumentJAXP.
    initSchemaLocations();
  }

  /**
   * Validates the XML signatures created by the signing samples.
   * If the validation succeeds, no exception will be thrown.
   *
   * @param url an URL that points to a resource that contains an XML Signature <p>
   *            e.g. "file:/C:/etjava/examples/xml/signedData.xml"
   *            e.g. "file:signedData.xml"
   *            e.g. "http://host/signedData.xml"
   */
  public void validateSignature(String url, int method)
  throws Exception
  {
    URI baseURI = new URI(url);

    // Open an input stream to the XML document containing the Signature
    System.out.println("Opening \"" + url + "\"");
    
    // Get an InputStream to the document containing the XML Signature
    ExternalReferenceResolverImpl res = new ExternalReferenceResolverImpl(baseURI);
    InputStream istream = res.resolve(baseURI);
    
    // Specify an XPath that selects the Signature element.  The Toolkit's
    // signing examples set the Signature ID to 'Signature001'.
    String signatureSelector = "//*[@Id=\"Signature001\"]";
    
    System.out.println("Creating a Verifier...");

    /* Create a Verifier using Method (1), (2), or (3)
     * 
     * Note: With any Method, if your application reads the XML document
     *       directly from an InputStream, so there is no base URI, set the
     *       'baseURI' parameter to 'null'.
     */
     
    // An URL to the noNamespaceSchemaLocation is required if the XML source does 
    // not declare it in a schema instance declaration, 'xsi:noNamespaceSchemaLocation=...'.
    String noNamespaceSchemaLocation = null ; // e.g. file:/C:/myschemas/foo.xsd
    
    // Optional parameter to Verifier constructor;  refer to its javadoc.
    String additionalNSPrefixes = null ;

    Verifier verifier = null;
    Document doc = null;
     
    System.out.println("Verify sample is demonstrating Method " + method);

    switch (method) {

        case DEMO_METHOD_1 :

            // Method (1) -- Your application might already have an instance of a DOM Document instance that 
            // it created earlier using any JAXP implementation.  As an example, we'll create 
            // a Document instance here though the JAXP API, then pass that Document to the Verifier.
            // This Verifier constructor assumes that your Document instance was created by a validating parser,
            // so any ID attributes outside the <Signature> will resolve when the signature is verified.
            // In any case, you must at least provide the dsig schema as one of the schema locations
            // in the last argument.  See also Utils.createDOMDocumentJAXP.
            doc = Utils.createDOMDocumentJAXP( istream, noNamespaceSchemaLocation, m_schemaLocations);
            verifier =  new Verifier(doc, baseURI, signatureSelector, additionalNSPrefixes);
            break;

        case DEMO_METHOD_2 :

            // Method (2) -- Create the Verifier from an InputStream.
            verifier =  new Verifier(istream,
                                     baseURI,
                                     signatureSelector,
                                     additionalNSPrefixes,
                                     noNamespaceSchemaLocation,
                                     m_schemaLocations);
              break;

        case DEMO_METHOD_3 :

            // Method (3) -- Create the Verifier from a DOM Document that is parsed from an InputStream
            doc =  Utils.createDOMDocument(istream, baseURI, noNamespaceSchemaLocation, m_schemaLocations);
            verifier =  new Verifier(doc, baseURI, signatureSelector, additionalNSPrefixes);
            break;
            
        default :
            // should not get here
            throw new Exception("No demonstration method was selected");

    }  
    
    
    // Set a trust manager to validate the certificates
    Utils.setTrustmanager(verifier, m_source) ;
    

    // Validate the XML signature
    System.out.println("Verifying...");
    VerifierSignature signature = verifier.getSignature();
    signature.verify();

    System.out.println("Verified !");

    // The signature has been verified, but the public key might have come merely
    // from a KeyValue in the XML Signature, not a valid certificate.
    // 
    // NOTE: To disable IXSIL's use of KeyValue information, remove that subElement
    //       setting from the 'keymanager.properties' properties file.   This will
    //       ensure that the public verification key comes from a valid certificate.
    //       If you do that, you can remove the following line;
    Utils.validatePublicKey(signature);

    // Show what this Signature signed.
    VerifierReference[] references = signature.getVerifierSignedInfo().getVerifierReferences();
    for(int i=0; i < references.length; i++) {

        String signedContentURI = references[i].getURI();

        // If the URI is empty, the entire document was signed (enveloped signature).
        if(signedContentURI.length() == 0) signedContentURI = url;

        System.out.println("URI of signed content: \"" + signedContentURI + "\"");
    }
  }
  

  /**
   * A helper method that constructs a schemaLocations parameter in the format
   * required by the IXSIL's Verifier class constructor.   The schemaLocations parameter
   * is a space separated sequence of namespace URI's and schema URL's, e.g. 
   *
   * "http://www.w3.org/2000/09/xmldsig# file:/C:/test/init/schemas/Signature.xsd"
   *
   * <p>
   * This method finds the signature schema URL in the property called
   * 'DOMUtils.SignatureSchema' in the init properties file. </p>
   */
  private String initSchemaLocations()
  throws URIException
  {
    m_schemaLocations = Utils.getDSIGschemaLocations() + " " + Utils.getWSschemaLocations();
    System.out.println("Schema locations: \""  + m_schemaLocations + "\"");
    return m_schemaLocations;
  }
  

}
