//===========================================================================
//
// Copyright (c)  2000-2010 Entrust.  All rights reserved.
// 
//===========================================================================
package com.entrust.toolkit.examples.xml.soap;

import java.io.IOException;
import java.io.InputStream;
import java.security.PrivateKey;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import iaik.ixsil.algorithms.CanonicalizationAlgorithm;
import iaik.ixsil.algorithms.CanonicalizationAlgorithmImplCanonicalXML;
import iaik.ixsil.algorithms.CanonicalizationAlgorithmImplExclusiveCanonicalXML;
import iaik.ixsil.algorithms.DigestAlgorithmImplSHA1;
import iaik.ixsil.algorithms.SignatureAlgorithm;
import iaik.ixsil.algorithms.SignatureAlgorithmImplDSA;
import iaik.ixsil.algorithms.SignatureAlgorithmImplRSA;
import iaik.ixsil.algorithms.TransformImplEnvelopedSignature;
import iaik.ixsil.core.Position;
import iaik.ixsil.core.Signer;
import iaik.ixsil.core.SignerReference;
import iaik.ixsil.core.SignerSignature;
import iaik.ixsil.core.SignerSignedInfo;
import iaik.ixsil.exceptions.IXSILException;
import iaik.ixsil.init.IXSILConstants;
import iaik.ixsil.init.IXSILInit;
import iaik.ixsil.keyinfo.SignerKeyManager;
import iaik.ixsil.util.DOMUtils;
import iaik.ixsil.util.ExternalReferenceResolverImpl;
import iaik.ixsil.util.URI;
import iaik.utils.RFC2253NameParserException;
import iaik.x509.X509Certificate;

import com.entrust.toolkit.KeyAndCertificateSource;
import com.entrust.toolkit.examples.xml.utils.Utils;
import com.entrust.toolkit.xencrypt.core.TransformImplDecryption;


/**
 * This class has methods that sign a SOAP message or selected fragments of an XML document or a whole XML document.
 */
public class SigningWorker
{
    //~ Static fields/initializers /////////////////////////////////////////////////////////////////////////////////////////////////

    // Arbitrary Id attribute values that will be used when creating Signatures
    private static final String KEYINFO_ID_001 = "KeyInfo001";
    private static final String SIG_ID_001 = "Signature001";

    //~ Instance fields ////////////////////////////////////////////////////////////////////////////////////////////////////////////

    /** Signer's verification certificate */
    private X509Certificate m_x509SignCert = null;

    /** Signing key */
    private PrivateKey m_signingKey = null;

    //~ Constructors ///////////////////////////////////////////////////////////////////////////////////////////////////////////////

    /**
     * Constructor.  Sets a private signing key and an X.509 verification certificate.
     */
    public SigningWorker(KeyAndCertificateSource source)
    {
        m_x509SignCert = source.getVerificationCertificate();
        m_signingKey = source.getSigningKey();
    }

    //~ Methods ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    /**
     * Returns a string representation of an XML signature over a fragment of an XML document.
     * The fragment is selected by an ID attribute and the Signature contains an internal
     * Reference to that content.
     *
     * @param toBeSignedXmlById the Id of the XML element to be signed
     * @param xmlResourceUri the XML document that contains the content to sign
     * @param xpathSignaturePosition an XPath expression that selects the element where the &lt;Signature&gt; will be placed
     * @param noNamespaceSchemaLocation a location of a schema for elements without a namespace prefix
     * @param schemaLocations a schemaLocations for all namespaces
     *
     * @return a String that provides the signature
     *
     * @throws IXSILException if IXSIL throws an exception when creating the signature
     * @throws IOException if there is an exception when serializing the signature to a string
     */
    public String xmlFragmentSign(String toBeSignedXmlById, String xmlResourceUri, String xpathSignaturePosition,
        String noNamespaceSchemaLocation, String schemaLocations)
    throws IXSILException, IOException, RFC2253NameParserException
    {
        Signer signer = null;

        // check that absolute URI can be created from xmlResourceUri
        checkForAbsoluteURI(xmlResourceUri);

        // figure out where signature is going
        if (xpathSignaturePosition == null) {
            // insert signature as first child of root element
            xpathSignaturePosition = "/*[1]";
        }

        Position insertHere = new Position(xpathSignaturePosition, "", 0);
        String fragment = null;

        // set up Signer
        URI baseURI = new URI(xmlResourceUri);
        ExternalReferenceResolverImpl res = new ExternalReferenceResolverImpl(baseURI);
        InputStream isURI = res.resolve(baseURI);

        if (toBeSignedXmlById == null) {
            // parse without validation: sign whole document
            signer = new Signer(isURI, baseURI, insertHere);
        } else {
            // parse with validation: sign internal reference (takes significantly longer)
            Document doc = DOMUtils.createDocumentFromXMLInstance(isURI, baseURI, DOMUtils.VALIDATION_YES_,
                    noNamespaceSchemaLocation, schemaLocations);
            signer = new Signer(doc, baseURI, insertHere);
        }

        Document signedDoc = envelopedSignToDocument(signer, toBeSignedXmlById);
        fragment = Utils.documentToString(signedDoc);

        return fragment;
    }

    /**
     * Returns a string representation of an XML signature over a fragment of an XML document.
     * The fragment is selected by an ID attribute and the Signature contains an internal
     * Reference to that content.
     *
     * @param toBeSignedXmlDataById the Id of the XML element to be signed
     * @param xmlInputStream an InputStream to the XML document that contains the content to sign
     * @param xpathSignaturePosition an XPath expression that selects the element where the &lt;Signature&gt; will be placed
     * @param noNamespaceSchemaLocation a location of a schema for elements without a namespace prefix
     * @param schemaLocations a schemaLocations for all namespaces
     *
     * @return a String that provides the signature
     *
     * @throws IXSILException if IXSIL throws an exception when creating the signature
     * @throws IOException if there is an exception when serializing the signature to a string
     */
    public String xmlFragmentSignData(String toBeSignedXmlDataById, InputStream xmlInputStream, String xpathSignaturePosition,
        String noNamespaceSchemaLocation, String schemaLocations)
    throws IXSILException, IOException, RFC2253NameParserException
    {
        Signer signer = null;

        // figure out where signature is going
        if (xpathSignaturePosition == null) {
            // insert signature as first child of root element
            xpathSignaturePosition = "/*[1]";
        }

        Position insertHere = new Position(xpathSignaturePosition, "", 0);

        if (toBeSignedXmlDataById == null) {
            // parse without validation: sign whole document
            signer = new Signer(xmlInputStream, null /* no base URI */, insertHere);
        } else {
            // parse with validation: sign internal reference (takes significantly longer)
            Document doc = DOMUtils.createDocumentFromXMLInstance(xmlInputStream, null /* no base URI */, DOMUtils.VALIDATION_YES_,
                    noNamespaceSchemaLocation, schemaLocations);
            signer = new Signer(doc, null /* no base URI */, insertHere);
        }

        Document signedDoc = envelopedSignToDocument(signer, toBeSignedXmlDataById);
        String fragment = Utils.documentToString(signedDoc);


        return fragment;
    }

    /**
     * Returns string representation of XML signature.
     *
     * @param toBeSignedSoapBodyById the Id of the XML element to be signed
     * @param soapEnvelopeUri an InputStream to the XML document that contains the content to sign
     * @param noNamespaceSchemaLocation a location of a schema for elements without a namespace prefix
     * @param schemaLocations a schemaLocations for all namespaces
     *
     * @return a String that provides the signature
     *
     * @throws IXSILException if IXSIL throws an exception when creating the signature
     * @throws IOException if there is an exception when serializing the signature to a string
     */
    public String soapSign(String toBeSignedSoapBodyById, String soapEnvelopeUri, String noNamespaceSchemaLocation,
        String schemaLocations) throws IXSILException, IOException
    {
        Signer signer = null;
        Position insertHere = null;

        // check that absolute URI can be created from soapEnvelopeUri
        checkForAbsoluteURI(soapEnvelopeUri);

        // set up Signer
        URI baseURI = new URI(soapEnvelopeUri);
        ExternalReferenceResolverImpl res = new ExternalReferenceResolverImpl(baseURI);
        InputStream isURI = res.resolve(baseURI);

        Document doc = null;

        // parse with validation: sign internal reference (takes significantly longer
        // than parsing without validation)
        doc = DOMUtils.createDocumentFromXMLInstance(isURI, baseURI, DOMUtils.VALIDATION_YES_, noNamespaceSchemaLocation,
                schemaLocations);

        // get WS-Security namespace and prefix
        String wsSecNamespaceURI = IXSILInit.getInitProperty(IXSILConstants.INITPROP_URI_WS_SECURITY_);
        String wsSecPrefix = IXSILInit.getInitProperty(IXSILConstants.INITPROP_NS_PREFIX_WS_SECURITY_);

        // check for existence of SOAP Header, and build it if it does not already exist.
        buildSOAPHeader(doc, wsSecNamespaceURI, wsSecPrefix);

        // Signature will always be inserted in the same position
        insertHere = new Position("(/descendant::" + wsSecPrefix + ":Security[position()=1])[1]", "", 0);

        signer = new Signer(doc, baseURI, insertHere);

        Document signedDoc = soapSignToDocument(signer, toBeSignedSoapBodyById);

        // must let Signature know about WS-Security namespace, since it will make use of it in KeyInfo.
        signedDoc.getElementById(SIG_ID_001).setAttributeNS(IXSILConstants.NAMESPACE_URI_NAMESPACES_, "xmlns:" + wsSecPrefix,
            wsSecNamespaceURI);

        String signedSOAP = Utils.documentToString(signedDoc);

        return signedSOAP;
    }

    /**
     * Returns string representation of XML signature.
     *
     * @param toBeSignedSoapBodyById the Id of the XML element to be signed
     * @param xmlInputStream an InputStream to the XML document that contains the content to sign
     * @param noNamespaceSchemaLocation a location of a schema for elements without a namespace prefix
     * @param schemaLocations a schemaLocations for all namespaces
     *
     * @return a String that provides the signature
     *
     * @throws IXSILException if IXSIL throws an exception when creating the signature
     * @throws IOException if there is an exception when serializing the signature to a string
     */
    public String soapSignData(String toBeSignedSoapBodyById, InputStream xmlInputStream, String noNamespaceSchemaLocation,
        String schemaLocations) throws IXSILException, IOException
    {
        Signer signer = null;
        Position insertHere = null;

        // parse with validation: sign internal reference (takes significantly longer
        // than parsing without validation)
        Document doc = DOMUtils.createDocumentFromXMLInstance(xmlInputStream, null, DOMUtils.VALIDATION_YES_,
                noNamespaceSchemaLocation, schemaLocations);

        // get WS-Security namespace and prefix
        String wsSecNamespaceURI = IXSILInit.getInitProperty(IXSILConstants.INITPROP_URI_WS_SECURITY_);
        String wsSecPrefix = IXSILInit.getInitProperty(IXSILConstants.INITPROP_NS_PREFIX_WS_SECURITY_);

        // check for existence of SOAP Header, and build it if it does not already exist
        buildSOAPHeader(doc, wsSecNamespaceURI, wsSecPrefix);

        // Signature will always be inserted in the same position
        insertHere = new Position("(/descendant::" + wsSecPrefix + ":Security[position()=1])[1]", "", 0);

        signer = new Signer(doc, null, insertHere);

        Document signedDoc = soapSignToDocument(signer, toBeSignedSoapBodyById);

        // must let Signature know about WS-Security namespace
        // (since it will make use of it in KeyInfo)
        signedDoc.getElementById(SIG_ID_001).setAttributeNS(IXSILConstants.NAMESPACE_URI_NAMESPACES_, "xmlns:" + wsSecPrefix,
            wsSecNamespaceURI);

        String signedSOAP = Utils.documentToString(signedDoc);

        return signedSOAP;
    }

    /**
     * Creates DOM document representation of an XML signature over a SOAP Body.
     *
     * @param signer a Signer
     * @param id is the ID of an element to sign
     *
     * @return a DOM document the is the signed SOAP envelope
     *
     * @throws IXSILException if IXSIL throws an exception when creating the signature
     * @throws IOException if there is an exception when serializing the signature to a string
     * @throws RuntimeException if no Id to be signed was provided
     */
    private Document soapSignToDocument(Signer signer, String id)
    throws IXSILException, IOException
    {
        if (id == null) {
            // means that no element exists with ID "id"
            throw new RuntimeException("No Id was provided to identify an element to sign.");
        }

        // get interface for signature
        SignerSignature signature = signer.getSignature();

        // set signature ID attribute
        signature.setId(SIG_ID_001);

        // get interface for signedInfo object
        SignerSignedInfo signedInfo = signature.getSignerSignedInfo();

        // specify Exclusive XML Cannonicalization algorithm - WS Security spec recommends its use
        // use default URI "http://www.w3.org/2001/10/xml-exc-c14n#"
        CanonicalizationAlgorithm c14nAlg = new CanonicalizationAlgorithmImplExclusiveCanonicalXML();

        // set canonicalization algorithm
        signedInfo.setCanonicalizationAlgorithm(c14nAlg);

        // specify signature algorithm
        setSignatureAlgorithm(signedInfo);

        // create reference to resource we want to sign
        SignerReference reference = signedInfo.createReference();

        URI uri = new URI(null, null, "", null, id);

        if (signer.toDocument().getElementById(id) == null) {
            // means that no element exists with ID "id"
            throw new RuntimeException("Failed to resolve the ID to a DOM element!");
        }

        reference.setURI(uri);

        // create decryption transform - WS Security spec recommends its use
        // default URI: "http://www.w3.org/2001/04/decrypt#"
        TransformImplDecryption transform = new TransformImplDecryption();

        // set decryption transform in the reference: include in case so that
        // XML content is to be encrypted after being signed
        reference.insertTransformAt(transform, reference.getTransformsNumber());

        // create new instance of digest algorithm : we will always use SHA1
        DigestAlgorithmImplSHA1 digestAlg = new DigestAlgorithmImplSHA1();

        // set digest algorithm
        reference.setDigestAlgorithm(digestAlg);

        // add resource reference to signature
        signedInfo.addReference(reference);

        // add keyInfo : SecurityTokenReference
        SignerKeyManager signerKeyManager = Utils.setSecurityTokenReferenceKeyInfo(signer, m_x509SignCert, KEYINFO_ID_001);

        // compute signature value
        signer.getSignature().sign();

        // return resulting XML document
        return signer.toDocument();
    }

 
    /**
     * Creates DOM document representation of enveloped XML signature.  In an enveloped signature, the signature is inserted into
     * the XML data to be signed.  The DOM document returned is identical to the resource to be signed, but contains a signature
     * element. <p>
     * 
     * Note: This method is also used when you want to sign a subset of an XML document. In this case, the id argument is used.
     *
     * @param signer a Signer, must already have been configured with a <code>Position</code> where the &lt;Signature&gt; will be placed.
     * @param id an ID in the XML document that identifies the element to sign
     *
     * @return a String that provides the signature
     *
     * @throws IXSILException if IXSIL throws an exception when creating the signature
     * @throws IOException if there is an exception when serializing the signature to a string
     */
    private Document envelopedSignToDocument(Signer signer, String id)
    throws IXSILException, IOException, RFC2253NameParserException
    {
        // get interface for signature
        SignerSignature signature = signer.getSignature();

        // set signature ID attribute
        signature.setId(SIG_ID_001);

        // get interface for signedInfo object
        SignerSignedInfo signedInfo = signature.getSignerSignedInfo();

        // specify canonicalization algorithm
        // use default URI "http://www.w3.org/TR/2001/REC-xml-c14n-20010315"
        CanonicalizationAlgorithm c14nAlg = new CanonicalizationAlgorithmImplCanonicalXML();

        // set canonicalization algorithm
        signedInfo.setCanonicalizationAlgorithm(c14nAlg);

        // specify signature algorithm
        setSignatureAlgorithm(signedInfo);

        // create reference to resource we want to sign
        SignerReference reference = signedInfo.createReference();

        if (id == null) {
            // if signing entire XML resource ...
            // set URI attribute value of reference to null: this ensures that
            // signature will be over whole document
            reference.setURI(new URI(""));
        } else {
            // if signing subset of XML resource ...
            URI uri = new URI(null, null, "", null, id);
            Document doc = signer.toDocument();
            Element element = doc.getElementById(id);

            if (element == null) {
                // means that no element exists with ID "id"
                throw new RuntimeException("Failed to resolve the ID to a DOM element.");
            }
            reference.setURI(uri);
        }

        // create enveloped signature transform
        // use default URI "http://www.w3.org/2000/09/xmldsig#enveloped-signature"
        TransformImplEnvelopedSignature transform = new TransformImplEnvelopedSignature();

        // set enveloped signature transform in the reference: this prevents
        // signing of signature itself
        reference.insertTransformAt(transform, reference.getTransformsNumber());

        // create new instance of digest algorithm : we will always use SHA1
        DigestAlgorithmImplSHA1 digestAlg = new DigestAlgorithmImplSHA1();

        // set digest algorithm
        reference.setDigestAlgorithm(digestAlg);

        // add resource reference to signature
        signedInfo.addReference(reference);

        // Recommended -- Attach the user's X.509 verification certificate to the Signature
        int x509Data = Utils.X509DATA_CERTIFICATE;

        // Optional -- Attach the X.509 certificate's subject name and/or issuer and serial number
        // x509Data |= Utils.X509DATA_SUBJECT_NAME | Utils.X509DATA_ISSUER_SERIAL;
    
        // Set the X509Data
        SignerKeyManager signerKeyManager = Utils.setX509KeyInfo(x509Data,
                                                            signer,
                                                            m_x509SignCert,
                                                            KEYINFO_ID_001);        

        // for non-repudiation of XML signature, sign keyInfo
        Utils.signKeyInfo(signedInfo, signerKeyManager);

        // compute signature value
        signer.getSignature().sign();

        // return resulting XML document
        return signer.toDocument();
    }
    
    
    /**
     * Utility method for SOAP signing use case.  Checks for the existence of the SOAP Header inside the SOAP envelope.  If Header
     * not present, builds it.  The assumption is that if the SOAP Header element exists, then the WS-Security Security element
     * also exists and it is in the right spot (i.e. as child of the Header) in the document.
     *
     * @param doc DOM document
     * @param wsSecNamespaceURI WS-Security namespace URI
     * @param wsSecPrefix wsSecNamespaceURI namespace prefix
     */
    private void buildSOAPHeader(Document doc, String wsSecNamespaceURI, String wsSecPrefix)
    {
        // get Document root
        Element root = doc.getDocumentElement();

        // get the SOAP namespace URI
        String soapNsUri = root.getNamespaceURI();
        
        // Get the Header element, if any.
        NodeList nl = doc.getElementsByTagNameNS(soapNsUri, IXSILConstants.SOAP_HEADER_TAG_);
        Element security = null;

        // If Header element exists, there's nothing to do.
        if (nl.getLength() != 0) {
            return ;
        }
        
        // if SOAP Header does not exist, create it
        Element header = doc.createElementNS(soapNsUri, root.getPrefix() + ":" + IXSILConstants.SOAP_HEADER_TAG_);

        // insert it as first child of root
        Node oldFirst = root.getFirstChild();
        root.insertBefore(header, oldFirst);

        // now add Security element
        security = doc.createElementNS(wsSecNamespaceURI, wsSecPrefix + ":" + IXSILConstants.WS_SECURITY_TAG_);

        // add WS-Security namespace as an attribute of the SOAP Envelope
        header.insertBefore(security, null);
        root.setAttributeNS(IXSILConstants.NAMESPACE_URI_NAMESPACES_,
                            IXSILConstants.NAMESPACE_PREFIX_NAMESPACES_ + wsSecPrefix,
                            wsSecNamespaceURI);
    }    

    /**
     * Returns Position object from an XPath expression.  The Position can be used for placement of 
     * to Signature element in an enveloped signature.
     *
     * @param signaturePosition Signature element is inserted as first child of signaturePosition element.  If signaturePosition is
     *        null, Signature element is inserted as first child of root element.
     *
     * @return Position object that shows where to put the Signature element
     */
    private Position insertSignatureHere(String signaturePosition)
    {
        // create xpath expression to specify where you want to insert the signature
        String xpath = null;

        if (signaturePosition != null) {
            // insert signature as first child of signaturePosition element
            xpath = "(//" + signaturePosition + ")[1]";
        } else {
            //insert signature as first child of root element
            xpath = "/*[1]";
        }
        return new Position(xpath, "", 0);
    }

    /**
     * Sets a SignatureAlgorithm of the correct kind, depending on whether this user has an RSA or DSA private signing key.
     *
     * @param signedInfo The SignerSignedInfo for the XML signature
     *
     * @throws IXSILException if requesting representation from the signature algorithm fails.
     */
    private void setSignatureAlgorithm(SignerSignedInfo signedInfo)
    throws IXSILException
    {
        // Create a signature algorithm of the appropriate kind, which
        // depends on the kind of private key this user possesses.
        SignatureAlgorithm alg = (m_signingKey.getAlgorithm().equals("DSA")) ? (SignatureAlgorithm)new SignatureAlgorithmImplDSA()
                                                                             : (SignatureAlgorithm)new SignatureAlgorithmImplRSA();

        // Set private key
        alg.setSignerKey(m_signingKey);

        // Set signature algorithm
        signedInfo.setSignatureAlgorithm(alg);
    }

    /**
     * Throws an <code>IXSILIllegalArgumentException</code> if an absolute URI cannot be
     * created from the String provided.
     *
     * @param uri a URI
     *
     * @throws IXSILException if uri string does not have a valid form or is not an absolute URI
     */
    private static void checkForAbsoluteURI(String uri)
    throws IXSILException
    {
        iaik.ixsil.util.Utils.checkArgForCondition("uri", !new URI(uri).isAbsolute());
    }
}
