//===========================================================================
//
// Copyright (c)  2003-2010 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.xml.utils;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.FactoryConfigurationError;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.Document;
import org.xml.sax.ErrorHandler;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.SAXNotRecognizedException;
import org.xml.sax.SAXNotSupportedException;
import org.xml.sax.SAXParseException;

import com.entrust.toolkit.xml.util.JAXParserConstants;

/**
 * <code>JAXParse</code> is a simple wrapper around a JAXP implementation.  It provides methods
 * for getting a parser, setting schema URLs, and parsing XML source to a DOM <code>Document</code>.
 * 
 * The Toolkit's XML example code uses this class to demonstrate how your application should 
 * sign or verify a <code>Document</code> instance that it has parsed using a JAXP library 
 * installed on the system, such as Apache Xerces.  The application can pass its <code>Document</code>
 * object directly to the constructor of the Toolkit's {@link iaik.ixsil.core.Signer} or 
 * {@link iaik.ixsil.core.Verifier}, and the Toolkit will use the installed JAXP implementation
 * as it creates or verifies XML signatures on it.
 * <p/> 
 * To ensure the Toolkit uses the library in the Java runtime, instead of its internal parser,
 * <i>you must set the following property in its XML initialization file
 * (usually named init.properties</i>);
 * 
 * <DL>
 *  <DD><B>DOMUtils.ImplementingClass = com.entrust.toolkit.xml.util.DOMUtilsImplApache</B></DD>
 * </DL>
 * 
 * In this example, the parser is an Apache implementation (Xerces).
 * <p/> 
 * For details on how JAXP determines which implementation to use, refer to {@link #getDocumentBuilder}.
 * <p/>   
 * For details on how to use the Toolkit's internal parser, see {@link com.entrust.toolkit.xml.util.DOMUtilsImplApache}.
 * <p/>   
 * @see com.entrust.toolkit.xml.util.DOMUtilsImplBase
 * @see iaik.ixsil.core.Signer#Signer(Document signatureDocOM, URI signatureDocBaseURI, Position signaturePosition)
 * @see iaik.ixsil.core.Verifier#Verifier(Document signatureDocOM, URI signatureDocBaseURI, String signatureSelectorXPath, String additionalNSPrefixes)
 */

public class JAXParse implements JAXParserConstants
{
    // The JAXP implementations we support
    private static final int JAXP_IMPLEMENTATION_UNKNOWN = 0;
    private static final int JAXP_IMPLEMENTATION_APACHE = 1;
    
    // Identifies the particular kind of JAXP implementation        
    private int m_jaxpImpl = JAXP_IMPLEMENTATION_UNKNOWN;
    
    // DocumentBuilderFactory
    private DocumentBuilderFactory m_dbFactory = null ;
    
    // DocumentBuilder
    private DocumentBuilder m_parser = null ;
    
    // An ErrorHandler ; 
    private ErrorHandlerUtil m_handler = null ; 
    
    // URL to the schema for elements in the default namespace for this document
    private String m_noNamespaceSchemaLocation = null;
    
    // schema locations string
    private String m_schemaLocation = null;
    
    // default is a validating parse, if 'parse(InputStream is)'
    // is invoked before 'setDocumentBuilder(boolean validating)'
    private boolean m_validating = true ;   
    
    /**
     * Constructor for JAXParse.
     */
    public JAXParse()
    {
        init();
    }
    
    /**
     * Constructor that sets schemas.
     */
    public JAXParse(String noNamespaceSchemaLocation, String schemaLocations)
    {
        this();
        setNoNamespaceSchemaLocation(noNamespaceSchemaLocation);
        setSchemaLocations(schemaLocations);
    }
    
    
    /**
     * At run-time, determine which JAXP implementation is installed and perform 
     * initialization specific to that particular implementation;  load
     * resource strings and set the Attribute and Feature interfaces.
     * But note that JAXParse source will compile against any JAXP implementation!
     */
    private void init()
    {
        System.out.println(
            "JAXP DOM implementation system property setting: "
                + System.getProperty(JAXP_DOM));
        try
        {
            // Get name of installed DocumentBuilderFactory
            m_dbFactory =  DocumentBuilderFactory.newInstance();
            String dbFactoryClassName = m_dbFactory.getClass().getName();

            // Optional: While debugging, report which JAR file has the JAXP implementation.
            String library = getUrlToResource(m_dbFactory.getClass()).toExternalForm();
            System.out.println("JAXP implementation: \"" + library + "\"");

            if (JAXP_DOM_APACHE.equals(dbFactoryClassName))
            {
                System.out.println("It's an Apache parser.");
                m_jaxpImpl = JAXP_IMPLEMENTATION_APACHE ;
            }
            else
            {
                // It a JAXP parser that had better not require any additional
                // configuration, apart from setting standard JAXP features.
                System.out.println("It's an unrecognized JAXP parser.");
            }
        }
        catch (FactoryConfigurationError e)
        {
            throw new RuntimeException(
                "No JAXP parser was found. "
                    + " Please refer to the javadoc for JAXParse.getDocumentBuilder for information on how you configure a parser: "
                    + e.toString());
        }
    }
    
    /**
     * Specifies that a particular JAXP implementation will be used, by setting system properties.
     * See the JAXP javadoc for
     * <a href="http://java.sun.com/webservices/docs/1.0/api/javax/xml/parsers/DocumentBuilderFactory.html#newInstance()">
     * DocumentBuilderFactory</a>.
     * 
     * @param dom identifier of the provider of a JAXP DOM implementation,
     *      e.g. {@link com.entrust.toolkit.xml.util.JAXParserConstants#JAXP_DOM_APACHE}
     * @param sax identifier of the provider of a JAXP SAX implementation,
     *      e.g. {@link com.entrust.toolkit.xml.util.JAXParserConstants#JAXP_SAX_APACHE}
     * @param transform identifier of the provider of a JAXP XSLT implementation,
     *      e.g. {@link com.entrust.toolkit.xml.util.JAXParserConstants#JAXP_TRANSFORM_APACHE}
     */
    static synchronized public void setJAXPImplementation(
        String dom,
        String sax,
        String transform)
    {
        System.setProperty(JAXP_SAX, (sax == null) ? JAXP_SAX_APACHE : sax);
        System.setProperty(JAXP_DOM, (dom == null) ? JAXP_DOM_APACHE : dom);
        System.setProperty(JAXP_TRANSFORM,
                        (transform == null) ? JAXP_TRANSFORM_APACHE : transform);
    }

    /**
     * Parses XML source to a DOM Document instance.
     */
    public synchronized Document parse(InputStream istream)
        throws
            ParserConfigurationException,
            SAXException,
            IOException
    {
        if(m_parser == null) {
            setDocumentBuilder(m_validating);
        }
        
        // reset the error count, in case we're re-using a DocumentBuilder
        if (m_validating) {
            m_handler.resetErrorCount();
        }
 
        Document document = m_parser.parse(new InputSource(istream));

        // Report whether there were errors in a validating parse
        if (m_validating) {
            int count = m_handler.getErrorCount();
            if ( count > 0) {
                System.out.println(
                    "\nThis document did not validate against the schema(s) provided.  Found " 
                        + count + " error(s) !");
            }
            else {
                System.out.println(
                    "\nThis document validates correctly against the schema(s) provided.");
            }
        }

        return document;
    }

    /**
     * Sets only those parser features that can be set through the JAXP API,
     * so are common to all JAXP implementations.  Typically, your application will
     * have to set additional implementation-specific features, in order to get 
     * the desired behavior.
     */
    private void setJAXPparserFeatures(DocumentBuilderFactory factory)
        throws SAXNotRecognizedException, SAXNotSupportedException
    {
        // set JAXP Features.
        System.out.println("Setting parser feature: namespace awareness");
        factory.setNamespaceAware(true);

        System.out.println(
            "Setting parser feature: including ignorable whitespace in element content");
        factory.setIgnoringElementContentWhitespace(false);

        System.out.println("Setting parser feature: not expanding entity references");
        factory.setExpandEntityReferences(false);
        
        //Set the Secure Processing feature
        System.out.println("Setting parser feature: secure processing");
        iaik.ixsil.util.Utils.setSecureProcessing(factory);
        
    }

    /**
     * Enable the use of XML schema(s) and their locations.  
     * Invoke the JAXP API to do this, but use the particular parser attribute 
     * strings that Apache parsers recognize.
     * <p/>
     * @param factory the DocumentBuilderFactory in wihch the attributes will be set
     */
    private void setSchemasJAXPImplApache(DocumentBuilderFactory factory)
    {
        // Apache specific way to enable use of schemas.  Required for Xerces 2 to recognize schemas,
        // otherwise the exception message "Document root element "foo", must match DOCTYPE root "null"."
        factory.setAttribute(
            APACHE_PARSER_FEATURE_SCHEMA_VALIDATION,
            Boolean.TRUE);

        if (m_noNamespaceSchemaLocation != null)
        {
            factory.setAttribute(
                APACHE_PARSER_PROPERTY_EXTERNAL_NONAMESPACESCHEMALOCATION,
                m_noNamespaceSchemaLocation);
        }

        if (m_schemaLocation != null)
        {
            factory.setAttribute(
                APACHE_PARSER_PROPERTY_EXTERNAL_SCHEMALOCATIONS,
                m_schemaLocation);
        }
    }

    /**
     * setSchemasJAXPImplUnknown
     */
    private void setSchemasJAXPImplUnknown(DocumentBuilderFactory factory)
        throws SAXNotRecognizedException, SAXNotSupportedException
    {
        return;
    }
    
    /**
     * Set any additional features required by this particular parser implementation
     */
    private void setAdditionalFeatures(DocumentBuilderFactory factory)
        throws SAXNotRecognizedException, SAXNotSupportedException
    {
        // Each DOM provider uses implementation specific names for the features/attributes it supports.
        switch (m_jaxpImpl)
        {
            case JAXP_IMPLEMENTATION_APACHE :

                // More Apache specific features
                factory.setAttribute(
                    APACHE_PARSER_FEATURE_CREATE_ENTITY_REFERENCE_NODES,
                    Boolean.FALSE);

                factory.setAttribute(
                    APACHE_PARSER_FEATURE_CONTINUE_AFTER_FATAL_ERROR,
                    Boolean.FALSE);

                // Do not defer node expansion.
                factory.setAttribute(
                    APACHE_PARSER_FEATURE_DEFER_NODE_EXPANSION,
                    Boolean.FALSE);

                // Whitespace included.
                factory.setAttribute(
                    APACHE_PARSER_FEATURE_INCLUDE_IGNORABLE_WHITESPACE,
                    Boolean.TRUE);
                
                break;

            case JAXP_IMPLEMENTATION_UNKNOWN :
                // nothing to do
                break;

            default :
                // shouldn't get here
                break;
        }
    }

    /**
     * Sets the JAXP DocumentBuilder object for this instance.  After it has been set,
     * this instance can parse a document.  The required schemas, if any, must be set
     * before this method is invoked.
     * <p>
     * You can request a specific JAXP implementation by: (i) a system property
     * (see {@link #setJAXPImplementation}), or (ii) in the 'jaxp.properties' file 
     * in your JRE lib folder, or (iii) by the JAR Services API, or (iv) by a system default.
     * If those settings provide no implementation, this method returns
     * the Toolkit's internal implementation, which is based on Apache Xerces version 1.4.4.
     * </p>
     * <p>
     * The libraries for an external JAXP implementation should be on the class path, or be 
     * installed extensions to the JRE.
     * </p>
     * <p>
     * Please read the javadoc at 
     * <a href="http://java.sun.com/webservices/docs/1.0/api/javax/xml/parsers/DocumentBuilderFactory.html#newInstance()">
     * <code>javax.xml.parsers.DocumentBuilder.newInstance()</code></a>
     * for details on how one uses these settings to specify a particular JAXP implementation.
     * </p>
     * @param validating is a Boolean value used to determine if the <code>DocumentBuilder</code> will validate
     *                   documents as it parses them. If the value is true, the document will be validated when it
     *                   is parsed. If the value is false, the document will not be validated.
     */
    public synchronized void setDocumentBuilder(boolean validating)
        throws
            ParserConfigurationException,
            SAXNotRecognizedException,
            SAXNotSupportedException
    {
        m_validating = validating ;

        setJAXPparserFeatures(m_dbFactory);
        
        // If parsing with validation, set the schemas
        if(m_validating) {
            
            // Each DOM provider uses implementation specific names for the features/attributes it supports.
            switch (m_jaxpImpl)
            {
                case JAXP_IMPLEMENTATION_APACHE :
                    setSchemasJAXPImplApache(m_dbFactory);
                    break;
    
                case JAXP_IMPLEMENTATION_UNKNOWN :
                    setSchemasJAXPImplUnknown(m_dbFactory);
                    break;
                    
                default :
                    break;              
            }
        }
        
        // set any additional features specific to particular JAXP implementations
        setAdditionalFeatures(m_dbFactory);

        // If parsing with validation, must attach an ErrorHandler.
        m_dbFactory.setValidating(m_validating);
        
        DocumentBuilder db = m_dbFactory.newDocumentBuilder();
        db.setErrorHandler(new ErrorHandlerUtil());
        
        m_parser = db;
    }

    /**
     * Returns the noNamespaceSchemaLocation for this parser.
     * @return String
     */
    public String getNoNamespaceSchemaLocation()
    {
        return m_noNamespaceSchemaLocation;
    }

    /**
     * Sets a noNamespaceSchemaLocation for this parser.
     * @param noNamespaceSchemaLocation The noNamespaceSchemaLocation to set
     */
    public void setNoNamespaceSchemaLocation(String noNamespaceSchemaLocation)
    {
        m_noNamespaceSchemaLocation = noNamespaceSchemaLocation;
    }

    /**
     * Returns the schemaLocations string for this parser.
     * @return String
     */
    public String getSchemaLocations()
    {
        return m_schemaLocation;
    }

    /**
     * Sets a schemaLocations string for this parser.
     * @param schemaLocation The schemaLocation to set
     */
    public void setSchemaLocations(String schemaLocation)
    {
        m_schemaLocation = schemaLocation;
    }

    /** A private helper that returns the URL to a provided class.
    *
    * @param c is a <code>java.lang.Class</code>
    * @return the <code>URL</code> to the resource
    */
    private static URL getUrlToResource(Class c)
    {
        final String resource =
            "/" + c.getName().replace('.', '/').concat(".class");
        final Class cfinal = c;
        return (
            URL) java
                .security
                .AccessController
                .doPrivileged(new java.security.PrivilegedAction()
        {
            public java.lang.Object run()
            {
                return cfinal.getResource(resource);
            }
        });

    }

    /**
     * This inner class implements the <code>org.xml.sax.ErrorHandler</code> interface.
     * It comes from sample code that ships with Apache Xerces (modified slightly).
     */
    class ErrorHandlerUtil implements ErrorHandler
    {
        private int m_numberOfErrors = 0;
        
        ErrorHandlerUtil()
        {
            m_handler = this ;
        }
       

        // Warning
        public void warning(SAXParseException ex)
        {
            m_numberOfErrors++;
            System.err.println(
                "[Warning] " + getLocationString(ex) + ": " + ex.getMessage());
        }

        // Error
        public void error(SAXParseException ex)
        {
            m_numberOfErrors++;
            System.err.println(
                "[Non-Fatal Error] "
                    + getLocationString(ex)
                    + ": "
                    + ex.getMessage());
        }

        // Fatal error
        public void fatalError(SAXParseException ex) throws SAXException
        {
            m_numberOfErrors++;
            System.err.println(
                "[Fatal Error] "
                    + getLocationString(ex)
                    + ": "
                    + ex.getMessage());
            throw ex;
        }

        // error count
        public int getErrorCount()
        {
            return m_numberOfErrors;
        }
        
        // error count
        private void resetErrorCount()
        {
            m_numberOfErrors = 0 ;
        }


        /** Returns a string of the location. */
        private String getLocationString(SAXParseException ex)
        {
            StringBuffer str = new StringBuffer();

            String systemId = ex.getSystemId();
            if (systemId != null)
            {
                int index = systemId.lastIndexOf('/');
                if (index != -1)
                    systemId = systemId.substring(index + 1);
                str.append(systemId);
            }
            str.append(':');
            str.append(ex.getLineNumber());
            str.append(':');
            str.append(ex.getColumnNumber());

            return str.toString();

        } // getLocationString(SAXParseException):String

    }

}
