//===========================================================================
//
// Copyright (c)  2002-2024 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.activeDirectory;

import java.io.FileOutputStream;

import com.entrust.toolkit.User;
import com.entrust.toolkit.credentials.CredentialReader;
import com.entrust.toolkit.credentials.FilenameProfileReader;
import com.entrust.toolkit.credentials.FilenameProfileWriter;
import com.entrust.toolkit.util.HttpsDirectoryClient;
import com.entrust.toolkit.util.ManagerTransport;
import com.entrust.toolkit.util.SecureStringBuffer;
import com.entrust.toolkit.x509.LdapDirectory;

/**
 * LoginActiveDirectory.java demonstrates how to log in a user whose X.509 certificates
 * reside on a MS Active Directory.  The procedure is almost identical to the
 * standard login with other Directories that support LDAP, except that this class
 * uses an HttpsDirectoryClient object instead of a JNDIDirectory  object to communicate
 * with an SSL-enabled Active Directory proxy.
 * <p>
 * Because the client authenticates to the Active Directory proxy using SSL, the client
 * does not need to access its Windows login password.
 * <p>
 * Please refer to the ActiveDirectoryServer.java sample to see how to run
 * the SSL-enabled Active Directory proxy.
 */
public class LoginActiveDirectory
{
    public static void main(String args[])
    {
        // Check the command-line arguments.  If they're not there, then
        // print out a usage message.
        if (args.length < 2) {
            System.out.println();
            System.out.println("usage: LoginActiveDirectory <profile> <password> [-dir <AD proxy IP address> <AD proxy port> ] [-pki <Authority IP address> ]");
            System.out.println();
            System.exit(0);
        }
        try {
            //  Parse the command-line arguments.
            int index = 0;
            String profile = args[index++];
            String password = args[index++];
            String managerIPAddress = null;
            String directoryIPAddress = null ;
            String directoryProxyPort = null ;

            while (args.length > index) {
                if (args[index].equals("-dir")) {
                    directoryIPAddress = args[++index];
                    directoryProxyPort = args[++index];
                }

                if (args[index].equals("-pki")) {
                    managerIPAddress = args[++index];
                }
                ++index;
            }

            //  Display the parameters
            System.out.println();
            System.out.println("profile: " + profile);
            System.out.println("password: " + password);
            if (managerIPAddress != null) {
                System.out.println("IP address of Entrust Authority: " + managerIPAddress);
            }
            if (directoryIPAddress != null) {
                System.out.println("IP address of Active Directory SSL Proxy : " + directoryIPAddress + ":" + directoryProxyPort);
            }
            System.out.println();

            // Instantiate a user
            User user = new User();

            // Instantiate a credential reader
            CredentialReader cr = new FilenameProfileReader(profile);

			// Instantiate a profile writer.
            FilenameProfileWriter writer = new FilenameProfileWriter(profile, 0);
            user.setCredentialWriter(writer);

    	    // Create a transport to access Active Directory through an SSL-enabled proxy.
    	    // Note: Set the protocol here to 'http'.  HttpsDirectoryClient handles the SSL.
    	    String directoryServerURL = "http://" + directoryIPAddress + ":" + directoryProxyPort;
    	    LdapDirectory dir = new HttpsDirectoryClient(directoryServerURL, 0);

    	    // Create a transport directly to the Authority in the usual way, irrespective
    	    // of whether the X.509 Directory is a MS Active Directory or another kind.
            int authorityPortNumber = 829 ;  // the well known port for PKIX CMP
            ManagerTransport emt = new ManagerTransport(managerIPAddress, authorityPortNumber);

            // Set those connections
            user.setConnections(dir, emt);
            System.out.println("Directory connection set to: " + directoryServerURL);
            System.out.println("Authority connection set to: " + managerIPAddress + ":" + authorityPortNumber);

            // Log in
            System.out.println("login ONLINE");
            int status = user.login(cr, new SecureStringBuffer(new StringBuffer(password)));
            System.out.println(" *** LOGIN COMPLETE *** ");

            // Always check the password status after a login
	        if((status & User.WARNING_PW_EXPIRED) != 0) {
	            System.out.println("WARNING_PW_EXPIRED");
	            System.out.println("Password lifetime was " +
	                user.getClientSettings().getPasswordRuleTester().getExpirationTimeInWeeks() + " weeks");
	        }
	        if((status & User.WARNING_PW_NOT_VALID) != 0) {
	            System.out.println("WARNING_PW_NOT_VALID");
            }

            // Example: Now we can read Active Directory through the SSL proxy,
            // because the login gave us SSL client credentials
            String caName = user.getCaCertificate().getSubjectDN().toString();
            System.out.println("CA distinguished name: " + "\"" + caName + "\"");
            byte[][] attr = user.getDirectory().getAttr(caName, "caCertificate");
            if(attr.length > 0) {
	            FileOutputStream ostream = new FileOutputStream("caCertificate.cer");
	            ostream.write(attr[0]);
	            ostream.close();
	        }

            System.out.println("Done.");
        }
        catch (Exception e) {
            System.out.println();
            System.out.println("exception thrown:");
            e.printStackTrace();
        }
    }
}
