//===========================================================================
//
// Copyright (c)  2003-2024 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.capi;

import java.io.File;
import java.io.IOException;
import java.security.Signature;
import java.security.interfaces.DSAPublicKey;
import java.util.Arrays;
import java.util.Iterator;

import javax.crypto.Cipher;

import com.entrust.toolkit.JniInitializer;
import com.entrust.toolkit.User;
import com.entrust.toolkit.capi.CapiHandle;
import com.entrust.toolkit.credentials.CapiCredentialReader;
import com.entrust.toolkit.credentials.CapiIdentities;
import com.entrust.toolkit.credentials.CapiIdentity;
import com.entrust.toolkit.exceptions.CapiException;
import com.entrust.toolkit.exceptions.UserNotLoggedInException;
import com.entrust.toolkit.security.provider.Initializer;

/**
 * Creates and logs into a User object, using keys stored in CAPI. Performs the
 * following steps:
 * <nl>
 * <li>Finds and displays all of the identities in CAPI.</li>
 * <li>Prompts for an identity to log in to.</li>
 * <li>Logs in to the selected identity.</li>
 * <li>Signs and verifies a simple message using a key stored in CAPI.</li>
 * </nl>
 * <p>
 * To run, from the examples directory:
 * <pre>
 * java -classpath ..\lib\enttoolkit.jar;classes com.entrust.toolkit.examples.capi.CapiLogin
 * </pre>
 * </p>
 * <p>
 * Note that this sample requires the native library JniCapi.dll to be available.
 * </p>
 */
public class CapiLogin
{

    /**
     * The main driver for the CapiLogin example.
     * 
     * @param args
     *     ignored.
     */
    public static void main(String[] args)
    {
        // These lines load the native library that contains the native CAPI
        // calls relative to the current working directory. The path could be
        // changed to any absolute path, or the lines can be removed completely if
        // JniCapi.dll is installed in the system path.
        File nativeLibrary = new File("../lib/x64/JniCapi_64.dll");
        JniInitializer.initializeCapi(nativeLibrary.getAbsolutePath());

        // Insert the Entrust Provider in to the list of Providers so that
        // crypto operations will work.
        Initializer.getInstance().setProviders(Initializer.MODE_NORMAL);

        // Turn on extended debugging information to track CAPI
        // handle allocation.
        CapiHandle.setExtendedDebugging(true);

        User user = null;
        try
        {
            // List all of the users stored in CAPI, and choose one to log
            // in to.
            user = getCapiUser();

            if (user.getVerificationCertificate() != null)
            {
                // Sign and verify a simple message. 
                byte[] messageToProtect = "Message to sign".getBytes();

                // Must specify Entrust as the provider from which to get the
                // signature instance.  No other providers will be able to deal
                // with private keys stored in CAPI.
                Signature signature;
                if (user.getVerificationCertificate().getPublicKey() instanceof DSAPublicKey)
                {
                    signature = Signature.getInstance("DSA", "Entrust");
                }
                else
                {
                    // Currently only DSA and RSA work through CAPI, so if it isn't a DSA
                    // key it must be RSA.
                    signature = Signature.getInstance("SHA/RSA", "Entrust");
                }

                // Initialize the signature with the user's signing key.
                // getSigningKey() may in some cases return null, but the check
                // is not made here for the sake of simplicity.
                signature.initSign(user.getSigningKey());

                // Perform the actual signature operation.
                signature.update(messageToProtect);
                byte[] messageSignature = signature.sign();

                // Now begin the verification process. Similar to getSigningKey(),
                // getVerificationCertificate() could return null, but again the
                // check is left out for the sake of simplicity.
                signature.initVerify(user.getVerificationCertificate().getPublicKey());
                signature.update(messageToProtect);

                // Verify that the signature is correct.
                if (signature.verify(messageSignature))
                {
                    System.out.println("\r\nSuccessfully signed and verified message\r\n");
                }
                else
                {
                    System.out.println("\r\nFailed to verify message\r\n");
                }
            }

            // If the user has an encryption key pair, try to encrypt
            // and decrypt a message using RSA.
            if (user.getEncryptionCertificate() != null)
            {
                byte[] messageToProtect = "Message to encrypt".getBytes();

                // Must specify Entrust as the provider from which to get the
                // cipher instance.  No other providers will be able to deal
                // with private keys stored in CAPI.
                Cipher cipher = Cipher.getInstance("RSA", "Entrust");

                // Initialize the encryption with the user's public encryption key. 
                cipher.init(Cipher.ENCRYPT_MODE, user.getEncryptionCertificate().getPublicKey());

                // Perform the encryption
                byte[] encryptedMessage = cipher.doFinal(messageToProtect);

                // Now start the decryption process, specifying the user's private key.
                cipher.init(Cipher.DECRYPT_MODE, user.getDecryptionKey());

                // Decrypt the message.
                byte[] decryptedMessage = cipher.doFinal(encryptedMessage);

                // Make sure the decrypted message matches the original.
                if (Arrays.equals(decryptedMessage, messageToProtect))
                {
                    System.out.println("\r\nSuccessfully encrypted and decrypted message\r\n");
                }
                else
                {
                    System.out.println("\r\nFailed to encrypt and decrypt message\r\n");
                }
            }
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }
        finally
        {
            // Have to make sure the user is logged out cleanly, as the login
            // process will allocate native memory. 
            if (user != null && user.isLoggedIn())
            {
                try
                {
                    user.logout();
                }
                catch (UserNotLoggedInException e)
                {
                    // It's okay to ignore this exception, as a check was made
                    // to ensure the user was logged in.
                }
            }
        }

        // Display debugging information about the currently allocated
        // native CAPI objects.  It should display zeroes for all values.
        CapiHandle.debugInfo(true);
    }

    /**
     * Obtains a User from CAPI. Presents a list of identities found within
     * CAPI, prompts for a selection of one of them, and logs in to
     * the selected identity. 
     * 
     * @return
     *     a <code>User</code> object that has been successfully logged
     * in through CAPI.
     * 
     * @throws CapiException
     *     if there are no identities available, or if logging in to the
     * chosen identity fails.
     */
    public static User getCapiUser() throws CapiException
    {
        User user = new User();

        // Find all the recognized identities within CAPI.
        CapiIdentities identities = CapiIdentities.findIdentities();

        if (identities.size() == 0)
        {
            throw new CapiException("No CAPI identities found!");
        }

        // Loop through the found identities, displaying the name of each one.
        System.out.println("Number of identities found: " + identities.size());
        int index = 1;
        Iterator iterator = identities.iterator();
        while (iterator.hasNext())
        {
            CapiIdentity identity = (CapiIdentity) iterator.next();
            System.out.println(index + ". " + identity.getDistinguishedName());
            ++index;
        }

        // Now prompt for the identity that will be logged in.
        System.out.println("Enter index of identity to log in to: ");
        byte[] b = new byte[5];
        try
        {
            System.in.read(b);
            index = Integer.parseInt(new String(b).trim());
        }
        catch (IOException e)
        {
            System.out.println("Error reading, will use first identity");
            index = 1;
        }

        // If a valid identity was selected, log in to it.
        if (index > 0 && index <= identities.size())
        {
            iterator = identities.iterator();
            while (--index > 0)
            {
                iterator.next();
            }
            CapiIdentity identity = (CapiIdentity) iterator.next();

            // Create the object that actually reads the keys and certificates
            // from CAPI.
            CapiCredentialReader reader = new CapiCredentialReader(identity);

            try
            {
                // How the password is entered is up to the CAPI CSP, it cannot
                // be passed in from Java, so use null.
                user.login(reader, null);
            }
            catch (Exception e)
            {
                throw new CapiException("Unable to log in: " + e.getMessage());
            }
        }
        else
        {
            throw new CapiException("Invalid identity specified.");
        }
        return user;
    }
}
