//===========================================================================
//
// Copyright (c)  2003-2024 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.capi;

import java.io.File;
import java.security.cert.CertificateException;

import iaik.x509.X509Certificate;

import com.entrust.toolkit.JniInitializer;
import com.entrust.toolkit.capi.CapiConstants;
import com.entrust.toolkit.capi.CapiHandle;
import com.entrust.toolkit.capi.CertContext;
import com.entrust.toolkit.capi.CertPropertyId;
import com.entrust.toolkit.capi.CertStore;
import com.entrust.toolkit.capi.CertStoreProvType;
import com.entrust.toolkit.capi.CryptKeyProvInfo;
import com.entrust.toolkit.capi.KeySpec;
import com.entrust.toolkit.security.provider.Initializer;

/**
 * A sample program that lists the certificates stored in a CAPI certificate
 * store.
 * <p>
 * The name of the certificate store can be specified on the command line.
 * If no arguments are provided, then the &quot;My&quot; certificate store
 * is used.
 * </p>
 * <p>
 * To run, from the examples directory:
 * <pre>
 * java -classpath ..\lib\enttoolkit.jar;classes com.entrust.toolkit.examples.capi.EnumCertificates
 * java -classpath ..\lib\enttoolkit.jar;classes com.entrust.toolkit.examples.capi.EnumCertificates Root
 * </pre>
 * would list the certificates in the "My" and "Root" stores, respectively
 * </p>
 * <p>
 * Note that this sample requires the native library JniCapi.dll to be available.
 * </p>
 */
public class EnumCertificates
{
    /**
     * The main driver for the program.
     * 
     * @param args
     *     program arguments. If present, <code>args[0]</code> is used as
     * the name of the certificate store to open. Some common store names
     * are &quot;My&quot;, &quot;Root&quot;, and &quot;CA&quot;
     */
    public static void main(String[] args)
    {
        // These lines load the native library that contains the native CAPI
        // calls relative to the current working directory. The path could be
        // changed to any absolute path, or the lines can be removed completely if
        // JniCapi.dll is installed in the system path.
        File nativeLibrary = new File("../lib/x64/JniCapi_64.dll");
        JniInitializer.initializeCapi(nativeLibrary.getAbsolutePath());

        // Insert the Entrust Provider in to the list of Providers so that
        // crypto operations will work.
        Initializer.getInstance().setProviders(Initializer.MODE_NORMAL);

        // Name of the certificate store to open. If provided,
        // use the name from the command line.
        String certStoreName;
        if (args.length > 0)
        {
            certStoreName = args[0];
        }
        else
        {
            // Other common names are "Root" and "CA".
            certStoreName = "My";
        }

        // The certificate store itself.
        CertStore certStore = null;

        try
        {
            // Open the system store given in certStoreName. Open for
            // the current user, and do not create it if it is not
            // present. A null CAPI provider is specified, so the default
            // provider will be used.
            // Note that in this case, the simpler CertStore.openSystemStore
            // API could have been used.
            certStore =
                CertStore.openStore(
                    CertStoreProvType.CERT_STORE_PROV_SYSTEM,
                    0,
                    null,
                    CapiConstants.CERT_SYSTEM_STORE_CURRENT_USER
                        | CapiConstants.CERT_STORE_OPEN_EXISTING_FLAG,
                    certStoreName);

            if (certStore == null)
            {
                System.out.println("Unable to open certificate store \'" + certStoreName + "\'");
            }
            else
            {
                System.out.println("Listing certificates for certificate store \'" + certStoreName + "\'");

                // Enumerate all of the certificates in the store, and display the
                // details.  Note that each call to enumCertificatesInStore closes
                // the previous certificate.
                CertContext certContext = null;
                int index = 1;
                while (null != (certContext = certStore.enumCertificatesInStore(certContext)))
                {
                    System.out.println("Details of certificate " + index);

                    // Attempt to get the information about the provider and
                    // private key container name. Not all certificates will
                    // have this information available, for example, root CA
                    // certificates.
                    CryptKeyProvInfo ckpi =
                        (CryptKeyProvInfo) certContext.getCertificateProperty(
                            CertPropertyId.CERT_KEY_PROV_INFO_PROP_ID);
                    if (ckpi != null)
                    {
                        System.out.println("Provider name: " + ckpi.getProviderName());
                        System.out.println("Container name: " + ckpi.getContainerName());
                        if (ckpi.getKeySpec() == KeySpec.AT_KEYEXCHANGE)
                        {
                            System.out.println("Key type: Key Exchange");
                        }
                        else if (ckpi.getKeySpec() == KeySpec.AT_SIGNATURE)
                        {
                            System.out.println("Key type: Signature");
                        }
                        else
                        {
                            System.out.println("Key type: Unknown");
                        }
                    }

                    // Get the certificate bytes, and parse them in to a Java object.
                    byte[] certBytes = certContext.getCertBytes();
                    try
                    {
                        X509Certificate cert = new X509Certificate(certBytes);
                        System.out.println(cert.toString(true));
                    }
                    catch (CertificateException e)
                    {
                        System.out.println("Certificate " + index + " cannot be decoded: " + e.getMessage());
                    }
                    System.out.println("-------------------------------------------------------");
                    ++index;

                    // To save certContext for use later in the program, call
                    // certContext.duplicate();
                }
            }
        }
        catch (Exception e)
        {
            System.out.println("Caught exception: ");
            e.printStackTrace();
        }
        finally
        {
            // Make sure the certificate store is always closed, to avoid
            // memory leaks. 
            if (certStore != null)
            {
                certStore.close();
            }
        }

        // Display debugging information about the currently allocated
        // native CAPI objects.  It should display zeroes for all values.
        CapiHandle.debugInfo(true);
    }
}
