//===========================================================================
//
// Copyright (c) 2004-2024 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.cardms;

import java.io.File;
import java.security.GeneralSecurityException;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.PrivateKey;
import java.security.Signature;
import java.security.spec.AlgorithmParameterSpec;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.crypto.Cipher;

import iaik.asn1.DerCoder;
import iaik.security.cipher.SecretKey;
import iaik.security.dsa.DSAPrivateKey;
import iaik.security.rsa.RSAPrivateKey;
import iaik.utils.Util;
import iaik.x509.X509Certificate;

import com.entrust.toolkit.JniInitializer;
import com.entrust.toolkit.asn1.cmpv1.gkm.EntrustCertInfoId;
import com.entrust.toolkit.asn1.crmf.EncryptedValue;
import com.entrust.toolkit.asn1.crmf.POPOSigningKey;
import com.entrust.toolkit.asn1.crmf.POPOSigningKeyInput;
import com.entrust.toolkit.asn1.structures.AlgorithmID;
import com.entrust.toolkit.credentials.CMPForCardMS;
import com.entrust.toolkit.credentials.CardMSCertReqInfo;
import com.entrust.toolkit.credentials.P11DataObject;
import com.entrust.toolkit.credentials.P11PrivateKeyObject;
import com.entrust.toolkit.credentials.P11StorageObject;
import com.entrust.toolkit.credentials.P11X509CertificateObject;
import com.entrust.toolkit.exceptions.UserAlreadyLoggedInException;
import com.entrust.toolkit.exceptions.UserFatalException;
import com.entrust.toolkit.pkcs11.JNIPKCS11;
import com.entrust.toolkit.pkcs11.PKCS11LibraryConnection;
import com.entrust.toolkit.pkcs11.TokenInfo;
import com.entrust.toolkit.security.provider.Initializer;
import com.entrust.toolkit.util.AuthorizationCode;
import com.entrust.toolkit.util.CryptoUtils;
import com.entrust.toolkit.util.SecureStringBuffer;
import com.entrust.toolkit.x509.policies.ClientSettings;
import com.entrust.toolkit.x509.policies.EntrustCertDefnSettings;

/**
 * This sample application demonstrates how the Card Management System 
 * Integration feature in Toolkit (JTK) could be used by a Card Management 
 * System (CardMS).
 * 
 * <p>
 * This feature provides a CardMS with the ability to create/recover Entrust 
 * digital identities without the JTK having any access to the smart card the
 * digital identity will be stored on.  It requires that the CardMS handle the
 * generation of the user's signing key(s) and the actual writing of the 
 * digital identity to the smart card.
 * </p>
 * 
 * <p>
 * This code is designed to mimic the operation of a CardMS; it makes all the 
 * necessary calls into the JTK to provide user creation/recovery.  It also 
 * provides all the required services of a CardMS for integration with the JTK 
 * (signing key generation, smart card write operation).  In this example, the 
 * smart card is written via PKCS #11 calls, however, this could also by done 
 * through some other vendor specific mechanism (so long as the result is an
 * Entrust PKCS #11 format digital identity on the smart card)
 * </p> 
 *
 * <p>
 * Executing this sample application can be done as follows: 
 * </p>
 * 
 * <code>
 * SampleCardMS &lt;EASM IP&gt; &lt;EASM CMP Port&gt; 
 * &lt;User Reference Number&gt; &lt;User Authorization Code&gt; 
 * &lt;PKCS #11 Library&gt; &lt;Smart Card Slot ID&gt; 
 * &lt;Smart Card User PIN&gt; [-create | -recover]
 * </code>
 * 
 * <p>
 * It requires several command line arguments, each of which are described 
 * below:
 * </p>
 * 
 * <dl>
 *      <dt>EASM IP</dt>
 *      <dd>The IP address of the Entrust Authority Security Manager (EASM)
 *          that the user exists on</dd>
 *      <dt>EASM CMP Port</dt>
 *      <dd>The port on the EASM that is designated for CMP (typically 
 *          829)</dd>
 *      <dt>User Reference Number</dt>
 *      <dd>The user's reference number (issued by the EASM with the user is 
 *          setup for creation/recovery)</dd>
 *      <dt>User Authorization Code</dt>
 *      <dd>The user's authorization code (issued by the EASM with the user is 
 *          setup for creation/recovery)</dd>
 *      <dt>PKCS #11 Library</dt>
 *      <dd>The PKCS #11 library that will be used to access the smart card
 *          that the digital identity will be stored on</dd>
 *      <dt>Smart Card Slot ID</dt>
 *      <dd>The slot ID where the smart card is located</dd>
 *      <dt>Smart Card User PIN</dt>
 *      <dd>The normal User PIN required to log in to the smart card</dd>
 *      <dt>-create | -recover</dt>
 *      <dd>Indicates whether a create or recovery operation will be executed
 *          (default is a create operation)</dd>
 * </dl>    
 * 
 * @author Christopher D. Wood
 * @since 7.0sp1
 */
public class SampleCardMS {

    /*****************************************************************************/
    /*                                  CONSTANTS                                */
    /*****************************************************************************/

    /** The CardMS protocol encryption certificate; used by the EASM to protect all 
     *  private keys generated by the EASM and securely passed back to the CardMS. 
     *  This is RSA 2048 so it can protect RSA 2048 strength keys*/
    private static final String RAW_CMS_PROT_ENC_CERT_RSA_2048 = "MIICkjCCAXoCAQEwDQYJKoZIhvcNAQEFBQAwDjEMMAoGA1UEAxMDQ01TMCAXDTA0MDcyODE3MTY1MVoYDzIxMDQxMjMxMDUwMDAwWjAOMQwwCgYDVQQDEwNDTVMwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCg1nVRjuXO3TfckW42j/V5Vn/N6gQ4RgGVoFJe789WDmufU/9FMHMMSrBsW3GqDBz6Y4qW9IKWOeTt7MaeqJcYShaVFCLWagzXdYsEyULFn8IhKoL8nKq9BHLAzAxU40RA76tXqJqJe9cTGXV91sUpeiIKdjV5g5dFd+E52wVBh9dQvDS4m93/gnSePDZrM8eRnx1O3JZSAekAqjqtRgLXzAzPzKIXBx+76Bj7nVNhNSk9CnVj+R2MnclbXmLEb6aVpRnUDXfKEKshD1ecs/hDxigmT2iEvT0cZ55CGe1tEuUw9C59mlFg+f0SeMeVk3C2ucE57uSQbWLJeCpDIDDLAgMBAAEwDQYJKoZIhvcNAQEFBQADggEBAHrdPeb5TM7TraYIm0yCrmkAU51VqeibkUNqk7WuEhgjv9CUvaD7fPhaNj1GF32KDygVWZhoI3r+C0ww25SxVfBuWZiaAZKtYUctZZTEQ9+Xms/QHawoivh2XbQAd+cHZfVU9IJTtKqpw+HzCzjRw8wlo2TSIsFBuM8HZbqN9uOqEr8wicvqs4NbBUv4aWwGCbQqBpeIUxrrW2U0D3Jez4xJyZ2N2UMmC6aHuTIcH2rt2h6eBnirttcesWMpOFz5E/xKAW9lrYqsU+IEh0ZUM4Tp9nnMwdX3LhGtb8rh9fs6hHAyMzLF1BUCo8C6zm4w1X0EtozHYcjdcNWPFaMkLoI=";

    /** The CardMS protocol decryption key; used by the CMS to access all private 
     *  keys generated by the EASM and securely passed back to the CardMS. 
     *  This is RSA 2048 so it can protect RSA 2048 strength keys*/
    private static final String RAW_CMS_PROT_DEC_KEY_RSA_2048 = "MIIEvAIBADANBgkqhkiG9w0BAQEFAASCBKYwggSiAgEAAoIBAQCg1nVRjuXO3TfckW42j/V5Vn/N6gQ4RgGVoFJe789WDmufU/9FMHMMSrBsW3GqDBz6Y4qW9IKWOeTt7MaeqJcYShaVFCLWagzXdYsEyULFn8IhKoL8nKq9BHLAzAxU40RA76tXqJqJe9cTGXV91sUpeiIKdjV5g5dFd+E52wVBh9dQvDS4m93/gnSePDZrM8eRnx1O3JZSAekAqjqtRgLXzAzPzKIXBx+76Bj7nVNhNSk9CnVj+R2MnclbXmLEb6aVpRnUDXfKEKshD1ecs/hDxigmT2iEvT0cZ55CGe1tEuUw9C59mlFg+f0SeMeVk3C2ucE57uSQbWLJeCpDIDDLAgMBAAECggEAXN64a1mJ1MLjyJao/xRs9508utePjDbQH2etaGgsXlKBTkSgN4FcTRKrgEXlAf1wlhv4IGYbKUdQqxl3LpHGfR36oCY6X3ho5dhXZ3IW/inbXG9Bpz+HTxFMeYrY7AM2nBBH7ZvLb+t0KvEeaIUk32hIU6YiBoI3TMOF53RSOp2dHbjuvMzom1Vy6Abk+ZFfYgFk7mFJV/MpzRwS1xXcC4w4+emTSMn10r+FnYHcOT7tdDh+YLG4pDJtAvxIGPEMKtnBLzcVngnZVRbEaFAL3YD6LsTWUstcULwhucVrLEvzkuK93FErRBLp9hcviBlqOUQM8jCWsgqb/yrsYzZ3cQKBgQDL8GrGmxJaGnk5NNJcxQ4LqeEo2bLyaPohPauqhfBQ+2YVVfr+Y4+y/Q2bzEUa14RnlRfNa+kSNaP+TGGs/EwX7e8BVE/ZKWW0NkVUV2to8DPKwSun/XGC0V3VLRgnIrGcX6VkYbog/3G4YPSiBVFn5fObQZgl+uqLpI9SjTg7bwKBgQDJ5VSiNRvU/dKRyltaVEMIjnrQIZ0iOyBrhrgnVFErCgGEUlW17GRe2HFpvEapKsieWUISpQBva9hhLV3ce5CyHicm0hUEdbo3jo4Jt5TQO4zmnhEWPj/y+6DDUadm1rfuWcuxXnaQRdXV0oIJ4taawiVzqEkL0ntRR8wKVrQiZQKBgAaTRM4Md+YBzoDCrDmjfNVxb7oCctuP+OiTdXJ1CQ+DW4I6mHIVQa/SqtqoFl5MLo+sZe0Qmp4krpnnWoraQ5XN7RPM5RoqVIYf7dJPUz456GGUItUyvbMrDuxtnMa+Pp29E7cN05N2yPLZ5WlH5J0F8/nAfqhS07cW4iwWM4bvAoGAakqRRyCcuYK0BEs70w9/hxaz87Vuq6HjTJjNhO8wGG+ZItfSBgghQ4nCE7+ql89NdYhab8RsuTxW27ptqvFm2U9kn1yE8ick1bD4eIctEn/3i0Z2inZVZpAxMANiFZM/nH2r7oJj+0q47Gx6R8gj4bAK1ZwocM/sWntpIZxKE70CgYBOnw1Dsq/82VjqCR03iiARXwnBDIkmGcht4Ahw1fZw/ZOaxuAvNzJkyNyQK0/oJvVwGwRMlhIukTF9kHTlZ9OTylAPQogduwBVMGAOzhtnMyuoNi1cR4C6nSRLnhHrY/Ro7o2Ipj2FeAAPJ9Dt8ijZ2sMBqzf4LSjekSTNRdZR9Q==";

    /*****************************************************************************/
    /*                             SAMPLE APPLICATION                            */
    /*****************************************************************************/

    /** The sample application. */
    public static void main(String[] args) {
        // These lines load the native library that contains the native PKCS #11
        // calls relative to the current working directory. The path could be
        // changed to any absolute path, or the lines can be removed completely if
        // JNIPKCS11.dll is installed in the system path.
        File nativeLibrary = new File("../lib/win32/JNIPKCS11.dll");
        JniInitializer.initializeCapi(nativeLibrary.getAbsolutePath());

        // Parse arguments
        int size = args.length;
        if (size < 7 || size > 8) {
            System.out
                .println("SampleCardMS <EASM IP> <EASM CMP Port> <User Reference Number> <User Authorization Code> <PKCS #11 Library> <Smart Card Slot ID> <Smart Card User PIN> [-create | -recover]");
            return;
        }
        String easmIP = args[0];
        int easmCmpPort = 829;
        try {
            easmCmpPort = Integer.parseInt(args[1]);
        } catch (NumberFormatException e) {
            throw new IllegalArgumentException("<EASM Port> must represent an integer");
        }
        String refNum = args[2];
        String authCode = args[3];
        String p11Library = args[4];
        long slotID = 0;
        try {
            slotID = Integer.parseInt(args[5]);
        } catch (NumberFormatException e) {
            throw new IllegalArgumentException("<Smart Card Slot ID> must represent an integer");
        }
        String userPIN = args[6];
        boolean create = true;
        if (size == 8 && args[7].equalsIgnoreCase("-recover"))
            create = false;

        // Installs the Entrust cryptographic service provider and extracts entropy
        // from the system required for random number generation
        System.out.print("Initializing the Toolkit... ");
        Initializer.getInstance().setProviders(Initializer.MODE_NORMAL);
        System.out.println("DONE");

        // Instantiate the CardMS protocol encryption certificate and decryption key
        // (these were issued to the CardMS by some certification authority)
        X509Certificate cmsProtocolEncCert = null;
        PrivateKey cmsProtocolDecKey = null;
        try {
            cmsProtocolEncCert = new X509Certificate(Util.Base64Decode(RAW_CMS_PROT_ENC_CERT_RSA_2048.getBytes()));
            cmsProtocolDecKey = new RSAPrivateKey(Util.Base64Decode(RAW_CMS_PROT_DEC_KEY_RSA_2048.getBytes()));
        } catch (Exception e) {
            // This will never happen, the encoded key and cert are valid
        }

        // Begin user creation or recovery        
        CMPForCardMS cmp = new CMPForCardMS(easmIP, easmCmpPort);
        P11StorageObject[] digitalIdRepresentation = null;
        Map cmsGeneratedPrivateKeys = new HashMap();
        try {
            // Execute the general message transmission and, based on the policy 
            // information returned, create all the CardMS certificate request
            // information required during the following initialization or recovery 
            // transmission
            System.out
                .print("Executing General Message transmission and generating certificate request information... ");
            SecureStringBuffer secureRefNum = new SecureStringBuffer(new StringBuffer(refNum));
            AuthorizationCode secureAuthCode = new AuthorizationCode(new StringBuffer(authCode));
            CardMSCertReqInfo[] certReqInfo = doGeneralMessageTransmission(cmp, secureRefNum, secureAuthCode,
                cmsGeneratedPrivateKeys);
            System.out.println("DONE");

            // Execute the initialization or recovery transmission and extract the 
            // smart card based Entrust digital identity representation
            if (create) {
                System.out
                    .print("Executing Initialization transmission and retrieving digital identity representation... ");
                cmp.initializeAll(certReqInfo, cmsProtocolEncCert);
            } else {
                System.out.print("Executing Recovery transmission and retrieving digital identity representation... ");
                cmp.recoverAll(certReqInfo, cmsProtocolEncCert);
            }
            SecureStringBuffer secureUserPIN = new SecureStringBuffer(new StringBuffer(userPIN));
            digitalIdRepresentation = cmp.getDigitalIdRepresentation(secureUserPIN);
            System.out.println("DONE");
        } catch (Exception e) {
            System.out.println("FAILED");
            e.printStackTrace();
            return;
        }

        // Write the smart card based Entrust digital identity representation to
        // the smart card
        try {
            System.out.print("Writing the digital identity to the smart card... ");
            writeSmartCard(digitalIdRepresentation, cmsGeneratedPrivateKeys, p11Library, slotID, userPIN,
                cmsProtocolDecKey);
            System.out.println("DONE");
        } catch (Exception e) {
            // Execute the error transmission 
            cmp.error(e.toString());
            System.out.println("FAILED");
            e.printStackTrace();
            return;
        }

        // Complete user creation or recovery (confirmation transmission)
        try {
            System.out.print("Executing Confirmation transmission... ");
            cmp.confirmation();
            System.out.println("DONE");
        } catch (Exception e) {
            System.out.println("FAILED");
            e.printStackTrace();
        }
    }

    /*****************************************************************************/
    /*                          INTERNAL HELPER METHODS                          */
    /*****************************************************************************/

    /* 
     * (non-JavaDoc)
     * 
     * Executes a General Message transmission and creates all the necessary
     * CardMS certificate request information based on the policy settings 
     * received.
     * 
     * The General Message transmission provides the CardMS with the user's policy
     * settings, which include certificate definition policy settings.  Each
     * certificate definition corresponds to one of the user's private keys 
     * (certificate streams).  The policy settings indicates whether the key pair 
     * is to be generated by the CardMS or the EASM, the key algorithm, and the key 
     * algorithm (among other things).  Based on this information the CardMS can 
     * determine the key information it must inject into the following 
     * Initialization/Recovery transmission.
     * 
     * The API creates two structures: 
     *  
     * 1. a CardMSCertReqInfo[] containing all the necessary CardMS certificate 
     * request information (contains injected public key).  This information is 
     * required during the following Initialization/Recovery transmission.
     * 
     * 2. a Map containing all the private keys generated by the CardMS keyed by the 
     * certificate definition identifiers each CardMS injected key information 
     * corresponds to.  This information is required when writing the digital
     * identity to the smart card.
     * 
     * For signing keys (which are always generated by the CardMS), the EASM also 
     * requires proof-of-possession of these keys.  Proof-of-possession is provided 
     * by signing a representation of the public key with the private signing key.
     * Proof-of-possession for the private key is attached to the corresponding 
     * CardMS certificate request information.
     * 
     * @param cmp
     *      the CMP protocol to be used by the CardMS
     * 
     * @param refNum
     *      the user's reference number (issued by the EASM with the user is setup
     *      for creation/recovery)
     * 
     * @param authCode
     *      the user's authorization code (issued by the EASM with the user is setup
     *      for creation/recovery)
     * 
     * @param cmpGeneratedPrivateKeys
     *      a Map containing the private keys generated by the CardMS keyed by the 
     *      certificate definition identifiers each correspond to
     */
    private static CardMSCertReqInfo[] doGeneralMessageTransmission(CMPForCardMS cmp, SecureStringBuffer refNum,
        AuthorizationCode authCode, Map cmsGeneratedPrivateKeys) throws Exception {

        // Execute the General Message transmission
        cmp.generalMessage(refNum, authCode);

        // Generate all necessary injected key information based on the policy
        List certReqInfo = new ArrayList();
        try {
            // Examine the policy settings for each of the user's certificate 
            // definitions (each certificate definition corresponds to one of the
            // user's key pairs)
            Iterator iterator = cmp.getCertPolicySettings().getCertdefns();
            while (iterator.hasNext()) {
                EntrustCertDefnSettings certDefnSettings = (EntrustCertDefnSettings) iterator.next();

                // Only generate a key pair when the policy specifies client generation
                if (!certDefnSettings.isKeyClientGen())
                    continue;

                // Determine the key algorithm and strength (RSA-1024, RSA-2048, RSA-4096, RSA-6144, DSA-512, DSA-1024)
                String keyType = certDefnSettings.getKeyType();
                if (keyType == null) {
                    // Default falls back to pre-7.0 behaviour
                    ClientSettings clientSettings = cmp.getClientSettings();
                    String keyUsage = certDefnSettings.getKeyUsage();
                    if (keyUsage.equalsIgnoreCase("verification")) {
                        keyType = clientSettings.getSigningKeyType();
                    } else {
                        keyType = clientSettings.getEncryptionKeyType();
                    }
                }
                String algorithm = keyType.substring(0, keyType.indexOf('-'));
                int strength = Integer.parseInt(keyType.substring(keyType.indexOf('-') + 1));

                // Generate the key pair
                KeyPairGenerator keyGen = KeyPairGenerator.getInstance(algorithm, "Entrust");
                keyGen.initialize(strength);
                KeyPair keyPair = keyGen.generateKeyPair();

                // Create and add injected key information to the Map  
                EntrustCertInfoId certDefnId = certDefnSettings.getCertdefn();
                CardMSCertReqInfo certReqInfoValue = new CardMSCertReqInfo(certDefnId, keyPair.getPublic());

                // Attach proof-of-possession to the injected key information for
                // all signing keys; proof-of-possession consists of a signature 
                // over a representation of the the public key using the 
                // corresponding private key
                if (!certDefnSettings.getKeyUsage().equalsIgnoreCase("encryption")) {
                    // Generate the input to the signing operation 
                    POPOSigningKeyInput popoSigningKeyInput = cmp.generatePOPOSigningKeyInput(keyPair.getPublic(),
                        authCode);

                    // Determine the signing algorithm to be used (SHA1 is selected 
                    // as the digest algorithm)
                    AlgorithmID algID = null;
                    if (algorithm.equals("RSA")) {
                        algID = AlgorithmID.SignatureAlgs.sha1WithRSAEncryption;
                    } else if (algorithm.equals("DSA")) {
                        algID = AlgorithmID.SignatureAlgs.dsaWithSHA1;
                    } else {
                        throw new UserFatalException("Unsupported private key algorithm '" + algorithm + "'");
                    }

                    // Sign the input
                    Signature signer = Signature.getInstance("SHA-1/" + algorithm, "Entrust");
                    signer.initSign(keyPair.getPrivate());
                    signer.update(DerCoder.encode(popoSigningKeyInput.toASN1Object()));
                    POPOSigningKey popoSigningKey = new POPOSigningKey(algID, signer.sign());
                    popoSigningKey.setPOPOSKInput(popoSigningKeyInput);
                    certReqInfoValue.setPOPOSigningKey(popoSigningKey);
                }

                // Store the private key and injected key information 
                certReqInfo.add(certReqInfoValue);
                cmsGeneratedPrivateKeys.put(certDefnId, keyPair.getPrivate());
            }
        } catch (Exception e) {
            throw new UserFatalException("Failed while preparing the CardMS injected key information", e);
        }
        return (CardMSCertReqInfo[]) certReqInfo.toArray(new CardMSCertReqInfo[certReqInfo.size()]);
    }

    /* 
     * (non-JavaDoc)
     * 
     * Writes a smart card based Entrust digital identity representation to the
     * smart card in PCKS #11 format.
     * 
     * Each PKCS #11 object contained in the digital identity representation are 
     * written to the smart card will exactly the attributes specified.  In the
     * case of EASM generated private keys, the private key data will exist in 
     * encrypted format, so prior to writing to the token the private key is 
     * decrypted using the CardMS protocol decryption key.
     * 
     * @param digitalIdRepresentation
     *      a smart card based Entrust digital identity representation
     * 
     * @param cmsGeneratedPrivateKeys
     *      a Map containing all the private keys generated by the CardMS keyed by
     *      the certificate definition identifier each key corresponds too
     * 
     * @param p11Library
     *      the name of the vendor's PKCS #11 library that will be used to access
     *      the smart card
     * 
     * @param slotID
     *      the slot ID of the smart card
     * 
     * @param userPIN
     *      the user PIN of the smart card
     * 
     * @param cmsProtocolDecKey
     *      the CardMS protocol decryption key used to decryption private keys 
     *      generated by the EASM and securely passed back to the CardMS
     * 
     * @throws UserFatalException
     *      if an error occurred while attempting to write the smart card
     */
    private static void writeSmartCard(P11StorageObject[] digitalIdRepresentation, Map cmsGeneratedPrivateKeys,
        String p11Library, long slotID, String userPIN, PrivateKey cmsProtocolDecKey) throws UserFatalException {

        // Create connection to the smart card
        PKCS11LibraryConnection p11LibConn = null;
        JNIPKCS11 jnipkcs11 = null;
        try {
            p11LibConn = new PKCS11LibraryConnection(p11Library);
            jnipkcs11 = p11LibConn.getJNIPKCS11();
        } catch (Exception e) {
            throw new UserFatalException("Failed to connect to the smart card", e);
        }

        // Prepare the smart card for a new digital identity
        long rwSessionHandle = 0;
        try {
            // This should not actually be required, however, due to a problem with
            // ActivCard's PKCS #11 library (acpkcs201-en6.dll) it must be done.  
            // If this is not done, all the following calls will fail with 
            // CKR_SLOT_ID_INVALID
            //
            // TODO: remove once ActivCard resolves this problem
            jnipkcs11.getSlotList(true).getSlotIDs();

            // Determine if the token has a protected authentication path, if so,
            // passwords (PINs) are not passed via the P11 API, but instead entered
            // into the device by some external manner (PIN pad)
            if (jnipkcs11.getTokenInfo(slotID).isFlagSet(TokenInfo.CKF_PROTECTED_AUTHENTICATION_PATH))
                userPIN = null;

            // Open a read/write session with the token
            rwSessionHandle = jnipkcs11.openSession(slotID, true);

            // Login with the normal user
            try {
                jnipkcs11.login(rwSessionHandle, JNIPKCS11.CKU_USER, userPIN);
            } catch (UserAlreadyLoggedInException e) {
                // If the normal user was already logged in, it's okay
            }

            // Delete all PKCS #11 objects from the card
            try {
                // This should not actually be required, however, due to a problem
                // with ActivCard's PKCS #11 library acpkcs201-en6.dll) it must be 
                // done.  ActivCards' PKCS #11 library forces the automatic 
                // deletion of the corresponding certificate when a private key is
                // deleted.  If this is not done, the call to delete all objects 
                // MAY fail with CKR_OBJECT_HANDLE_INVALID
                //
                // TODO: remove once ActivCard resolves this problem
                long[] objectHandles = jnipkcs11.getPrivateKeyHandles(rwSessionHandle, null);
                if (objectHandles != null) {
                    for (int i = 0; i < objectHandles.length; i++) {
                        jnipkcs11.destroyObject(rwSessionHandle, objectHandles[i]);
                    }
                }

                // Delete all remaining objects from the card
                objectHandles = jnipkcs11.getAllObjectHandles(rwSessionHandle);
                if (objectHandles != null) {
                    for (int i = 0; i < objectHandles.length; i++) {
                        jnipkcs11.destroyObject(rwSessionHandle, objectHandles[i]);
                    }
                }
            } catch (Exception e) {
                throw new UserFatalException("Unable to delete all the objects from the smart card", e);
            }
        } catch (Exception e) {
            throw new UserFatalException("Failed while preparing the smart card for a digital identity", e);
        }

        // Write the digital identity to the smart card
        try {
            int size = digitalIdRepresentation.length;
            for (int i = 0; i < size; i++) {
                P11StorageObject obj = digitalIdRepresentation[i];

                // Write data object
                if (obj.isDataObject()) {
                    P11DataObject dataObj = (P11DataObject) obj;
                    jnipkcs11.setByteArrayObject(rwSessionHandle, obj.getCkaPrivate().booleanValue(),
                        obj.getCkaLabel(), dataObj.getCkaApplication(), dataObj.getCkaValue());
                    continue;
                }

                // Write X.509 certificate object
                if (obj.isX509CertificateObject()) {
                    P11X509CertificateObject certObj = (P11X509CertificateObject) obj;
                    jnipkcs11.setCertificate(rwSessionHandle, obj.getCkaPrivate().booleanValue(), obj.getCkaLabel(),
                        certObj.getCkaId(), new X509Certificate(certObj.getValue()));
                    continue;
                }

                // Write private key object
                P11PrivateKeyObject keyObj = (P11PrivateKeyObject) obj;
                EncryptedValue encryptedKey = keyObj.getEncryptedPrivateKey();
                PrivateKey key = null;
                if (encryptedKey == null) {
                    key = (PrivateKey) cmsGeneratedPrivateKeys.get(keyObj.getCertDefnId());
                } else {
                    // Decrypt an encrypted private key that was returned by the EASM
                    byte[] decryptedValue = decryptEncryptedValue(encryptedKey, cmsProtocolDecKey);
                    key = new RSAPrivateKey(decryptedValue);
                }
                String algorithm = key.getAlgorithm();
                if (algorithm.equalsIgnoreCase("RSA")) {
                    // RSA private key
                    RSAPrivateKey rsaKey = (RSAPrivateKey) key;
                    jnipkcs11.setRSAPrivateKey(rwSessionHandle, obj.getCkaLabel(), keyObj.getCkaId(), keyObj
                        .getCkaDecrypt().booleanValue(), keyObj.getCkaSign().booleanValue(), CryptoUtils
                        .getMagnitude(rsaKey.getModulus()), CryptoUtils.getMagnitude(rsaKey.getPublicExponent()),
                        CryptoUtils.getMagnitude(rsaKey.getPrivateExponent()), CryptoUtils.getMagnitude(rsaKey
                            .getPrimeP()), CryptoUtils.getMagnitude(rsaKey.getPrimeQ()), CryptoUtils
                            .getMagnitude(rsaKey.getPrimeExponentP()), CryptoUtils.getMagnitude(rsaKey
                            .getPrimeExponentQ()), CryptoUtils.getMagnitude(rsaKey.getCrtCoefficient()));
                } else if (algorithm.equalsIgnoreCase("DSA")) {
                    // DSA private key
                    DSAPrivateKey dsaKey = (DSAPrivateKey) key;
                    jnipkcs11.setDSAPrivateKey(rwSessionHandle, obj.getCkaLabel(), keyObj.getCkaId(), keyObj
                        .getCkaDecrypt().booleanValue(), keyObj.getCkaSign().booleanValue(), CryptoUtils
                        .getMagnitude(dsaKey.getParams().getP()), CryptoUtils.getMagnitude(dsaKey.getParams().getQ()),
                        CryptoUtils.getMagnitude(dsaKey.getParams().getG()), CryptoUtils.getMagnitude(dsaKey.getX()));
                } else {
                    throw new UserFatalException("Unsupported private key algorithm '" + algorithm + "'");
                }
            }

            // Terminate the connection with the smart card
            jnipkcs11.logout(rwSessionHandle);
            jnipkcs11.closeSession(rwSessionHandle);
            p11LibConn.closeConnection();
        } catch (Exception e) {
            throw new UserFatalException("Failed while writing the digital identity to the smart card", e);
        }
    }

    /* 
     * (non-JavaDoc)
     * 
     * Decrypts an encrypted value held in an EncryptedValue structure.
     *
     * For the decryption to be successful, the optional components 'symmAlg'
     * and 'encSymmKey' must be set.  The decryption process works as follows:
     *
     * 1. the 'encSymmKey' component is decrypted using the decryption key provided
     * 2. a symmetric key is created from the decrypted 'encSymmKey' that will 
     *    operate with the algorithm specified in the 'symmAlg' component
     * 3. the 'encValue' is decrypted using the symmetric key
     *
     * @param encryptedValue
     *      the EncryptedValue structure to be decrypted
     *
     * @param decryptionKey
     *       the key used for decrypting the 'encSymmKey' component
     *
     * @return byte[]
     *      the decrypted data
     *
     * @exception UserFatalException
     *      thrown if an error occurs while decrypting the EncryptedValue
     */
    private static byte[] decryptEncryptedValue(EncryptedValue encryptedValue, PrivateKey decryptionKey)
        throws UserFatalException {

        try {
            // Extract the necessary components from the EncryptedValue
            AlgorithmID symmAlg = encryptedValue.getSymmAlg();
            byte[] encSymmKey = encryptedValue.getEncSymmKey();
            if (symmAlg == null)
                throw new UserFatalException("'symmAlg' was not set, but required for decryption");
            if (encSymmKey == null)
                throw new UserFatalException("'encSymmKey' was not set, but required for decryption");

            // Decrypt encSymmKey
            byte[] symmKey = null;
            try {
                Cipher cipher;
                try {
                    cipher = Cipher.getInstance(decryptionKey.getAlgorithm(), "Entrust");
                } catch (GeneralSecurityException e) {
                    cipher = Cipher.getInstance(decryptionKey.getAlgorithm());
                }
                cipher.init(Cipher.DECRYPT_MODE, decryptionKey);
                symmKey = cipher.doFinal(encSymmKey);
            } catch (GeneralSecurityException e) {
                throw new UserFatalException("Unable to decrypt 'encSymmKey'", e);
            }

            // Decrypt encValue
            byte[] encValue = encryptedValue.getEncValue();
            try {
                SecretKey secretKey = new SecretKey(symmKey, symmAlg.getJCAKeyName());
                Cipher cipher = symmAlg.getCipherInstance();
                AlgorithmParameterSpec params = symmAlg.getAlgorithmParameterSpec();
                cipher.init(Cipher.DECRYPT_MODE, secretKey, params);
                return cipher.doFinal(encValue);
            } catch (GeneralSecurityException e) {
                throw new UserFatalException("Unable to decrypt 'encValue'", e);
            }

        } catch (UserFatalException e) {
            throw new UserFatalException("Unable to decrypt EncryptedValue", e);
        }
    }
}