//===========================================================================
//
// Copyright (c)  2019-2024 Entrust.  All rights reserved.
//
//===========================================================================

package com.entrust.toolkit.examples.cms;

import iaik.asn1.structures.AlgorithmID;
import iaik.asn1.structures.Attribute;
import iaik.cms.AuthenticatedDataStream;
import iaik.cms.CompressedDataStream;
import iaik.cms.OriginatorInfo;
import iaik.cms.PasswordRecipientInfo;
import iaik.cms.RecipientInfo;
import iaik.security.spec.PBKDF2KeyAndParameterSpec;
import iaik.smime.SMimeEncrypted;
import iaik.utils.Util;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.FileInputStream;
import java.io.InputStream;

import javax.crypto.SecretKey;
import jakarta.mail.Part;
import jakarta.mail.internet.MimeBodyPart;

import com.entrust.toolkit.security.provider.Initializer;
import com.entrust.toolkit.util.SecureStringBuffer;


/**
 * This sample shows how to use the Java Toolkit's Cryptographic Message Syntax (CMS) (RFC 2630) functionality
 * to process a password protected file that was generated by Entrust Entelligence Security Provider (ESP) 9.3 or higher.
 * The sample demonstrates the use of a PasswordRecipientInfo and an AuthenticatedDataStream, within the context of
 * processing an ESP generated file.
 * </p>
 * Note, a password protected file generated by ESP has a .pp7m file extension. (This sample does not support the
 * ESP self-decrypting .exe output file option).
 */
public class DecodeESPFromPassword {

    /**
     * The main program.
     *
     * @param args
     *        Program arguments. See the help below for the expected command line arguments for this main method.
     * @throws Exception
     */
    public static void main(String args[]) throws Exception {

        if (args.length <2) {
            System.out.println("Usage: DecodeESPFromPassword <input file> <password>");
            System.out.println("   where <input file> is the .pp7m encrypted file generated by Entrust Entelligence Security Provider");
            System.out.println("   where <password> is the password to the encrypted input file");
            return;
        }

        Initializer.getInstance().setProviders(Initializer.MODE_NORMAL);

        // The encrypted file
        String filename = args[0];

        // The password to the encrypted file
        SecureStringBuffer password = new SecureStringBuffer(new StringBuffer(args[1]));
        // The Password can only be used once, and then it gets wiped when decrypting the contents,
        // so make a copy of the password since it is used twice (once to decrypt the contents, and once to
        // display details about the recipient's PBKDF2).
        SecureStringBuffer passwordCopy = new SecureStringBuffer(new StringBuffer(args[1]));

        InputStream inFile = new FileInputStream(filename);

        decodeEnvelopedDataStreamFromFile(inFile, password, passwordCopy);
    }


    /**
     * Code that does the work of decoding the ESP generated file
     *
     * @param fileData the InputStream containing the ESP generated file
     * @param pass The Password protecting the file
     * @param passCopy A copy of the password
     * @throws Exception
     */
    private static void decodeEnvelopedDataStreamFromFile(InputStream fileData, SecureStringBuffer pass, SecureStringBuffer passCopy) throws Exception {

        // ESP generates a CMS enveloped data of type S/MIME
        SMimeEncrypted sMimeEncrypted = new SMimeEncrypted(fileData);

        // Get information about the included RecipientInfos
        RecipientInfo[] recipients = sMimeEncrypted.getRecipientInfos();
        boolean recipientFound = false;

        System.out.println("Extracting Recipient Info");
        if (recipients != null) {
            for (int recipientIndex = 0; recipientIndex < recipients.length; recipientIndex++) {
                if (recipients[recipientIndex].isPasswordRequired()) {
                    System.out.println("\nPassword RecipientInfo type");

                    // This call sets up the internal EncryptedContentInfoCipher based on the supplied password
                    // Decode parameters
                    PasswordRecipientInfo prf = (PasswordRecipientInfo) recipients[recipientIndex];
                    System.out.println("Alg Param Id: " + prf.getKeyDerivationAlgorithm());
                    AlgorithmID id = prf.getKeyDerivationAlgorithm();

                    PBKDF2KeyAndParameterSpec spec = new PBKDF2KeyAndParameterSpec(id.getParameter(), passCopy, 1);
                    // Display some details about the PBKDF2 key
                    System.out.println("salt: " + Util.toString(spec.getSalt()));
                    System.out.println("iteration: " + spec.getIterationCount());
                    System.out.println("key length: " + spec.getDerivedKeyLength());
                    System.out.println("password: " + Util.toString(spec.getPassword()));
                    System.out.println("Key derivation Algorithm: " + spec.getKDF());

                    // Use the password to derive the key encryption key (which will then give access to the content
                    // encryption key). This action will wipe the contents of 'pass'.
                    sMimeEncrypted.setupCipher(pass, recipientIndex);
                    recipientFound = true;
                    break;
                } else {
                    System.out.println("\nThis message also contains Recipient of type:\n" + recipients[recipientIndex]);
                }
            }
        }

        if (!recipientFound) {
            System.out.println("Password Recipient not found");
            return;
        }

        InputStream data_is = sMimeEncrypted.getInputStream();

        // Decode the Mime Layer Data
        Object o = new MimeBodyPart(data_is);

        System.out.println("CONTENT-TYPE of file is:\n" + ((Part)o).getContentType());
        // Strip off the Mime layer
        o = ((Part)o).getContent();
        // Content should be an input stream
        InputStream is = (InputStream)o;

        // InputStream should be an AuthenticatedDataStream
        AuthenticatedDataStream authenticatedData = new AuthenticatedDataStream(is);
        System.out.println("\nThis is SMime Authenticated Data");
        Attribute[] atts = authenticatedData.getAuthenticatedAttributes();
        System.out.println("Authenticated Attributes size: " + atts.length);
        Attribute[] unatts = authenticatedData.getUnauthenticatedAttributes();
        System.out.println("UnAuthenticated Attributes size: " + unatts.length);
        OriginatorInfo info = authenticatedData.getOriginatorInfo();
        System.out.println("OriginatorInfo: " + info);

        // Get the PasswordRecipientInfo used
        RecipientInfo[] recipient = authenticatedData.getRecipientInfos(RecipientInfo.PASSWORD_RECIPIENT_INFO);
        PasswordRecipientInfo pwri = (PasswordRecipientInfo )recipient[0];
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        try {
            SecretKey macKey = pwri.decryptKey(passCopy);
            System.out.println("The Recipient Key: " + macKey);
            // Setup the Mac
            authenticatedData.setupMac(macKey);

            // Read the remaining of the stream
            InputStream contentIs = authenticatedData.getInputStream();

            byte[] buf = new byte[1024];
            int r;
            while ((r = contentIs.read(buf)) > 0) {
                // do something useful
                bos.write(buf, 0, r);
            }
            contentIs.close();
            bos.flush(); bos.close();

            // verify the MAC
            if (!authenticatedData.verifyMac()) {
                System.out.println("Invalid MAC value!");
            }
            else {
                System.out.println("MAC Verified!");
            }
        }
        catch (Exception e) {
            e.printStackTrace();
        }

        // Now unwrap the Compressed Content inside the AuthenticatedData:
        o = new MimeBodyPart(new ByteArrayInputStream(bos.toByteArray()));

        System.out.println("\nCONTENT-TYPE of compressed content inside authenticated data is:\n" + ((Part)o).getContentType());
        // Strip of Mime layer
        o = ((Part)o).getContent();
        // Content is just an input stream
        is = (InputStream)o;

        try {
            CompressedDataStream cd = new CompressedDataStream(is);
            System.out.println("\nThis is a Compressed Data Stream");
            o = new MimeBodyPart(cd.getInputStream());
            System.out.println("CONTENT-TYPE:\n"+((Part)o).getContentType());
            // Strip off Mime layer
            o = ((Part)o).getContent();

            // Data should be a String
            if (o instanceof String) {
                System.out.println("\nUnencrypted contents is of type String.\nThe original file content is:\n");
                System.out.println((String)o);
            } else {
                System.out.println("\nContents is not a String");
            }
        }
        catch (Exception e) {
            System.out.println("Not a Compressed Data.... " + e.getMessage());
        }
    }
}

