//===========================================================================
//
// Copyright (c)  2023-2024 Entrust.  All rights reserved.
//
//===========================================================================

package com.entrust.toolkit.examples.cms;

import com.entrust.toolkit.cms.EntrustProvider;
import com.entrust.toolkit.security.provider.Initializer;
import com.entrust.toolkit.util.SecureStringBuffer;
import iaik.asn1.structures.AlgorithmID;
import iaik.cms.*;
import iaik.security.spec.PBEKeyAndParameterSpec;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;


/**
 * This sample application demonstrates the use of password based encryption (RFC 3211) for
 * Cryptographic Message Syntax (CMS) (RFC 2630). It shows how to create a CMS <code>EnvelopedData</code>
 * with a <code>PasswordRecipientInfo</code>, and how to read a CMS <code>EnvelopedData</code> that has
 * a <code>PasswordRecipientInfo</code>
 */
public class PasswordBasedEncryption {

    /**
     * The main program.
     *
     * @param args
     *        Program arguments. See the displayHelp method below for the expected command line arguments for
     *        this main method.
     * @throws Exception
     */
    public static void main(String[] args) throws Exception {

        if (args.length < 3) {
            displayHelp();
            return;
        }

        Initializer.getInstance().setProviders(Initializer.MODE_NORMAL);

        String action = args[0];
        String file = args[1];
        SecureStringBuffer password = new SecureStringBuffer(new StringBuffer(args[2]));

        if (action.equalsIgnoreCase("encryptFile")) {
            writeEnvelopedData(file, password);
        } else if (action.equalsIgnoreCase("decryptFile")) {
            readEnvelopedData(file, password);
        } else {
            System.out.println("Invalid action specified.");
            displayHelp();
        }
    }


    /**
     * A description of the expected command line arguments into the main method.
     */
    private static void displayHelp() {
        System.out.println("Usage: PasswordBasedEncryption <action> <input file> <password>");
        System.out.println("   where <action> is one of: encryptFile, decryptFile");
        System.out.println("   where <input file> is the file to be processed:\n" +
                           "       When using the 'encryptFile' action, <input file> is the plaintext file that is to be encrypted.\n" +
                           "       When using the 'decryptFile' action, <input file> is the encrypted file that is to be decrypted.");
        System.out.println("   where <password> is the password that is used to protect the file\n" +
                           "       (for reading the encrypted file or when generating the encrypted file)");
    }


    private static void writeEnvelopedData(String fileName, SecureStringBuffer password) throws Exception {
        // The algorithm used to create the content encryption key (cek)
        AlgorithmID contentEncryptionAlgorithm = AlgorithmID.aes_256_CBC;

        // The algorithm used to convert the (user supplied) password to a key encryption key (kek)
        AlgorithmID keyDeriviationAlgorithm  = AlgorithmID.pbkdf2;

        // The algorithm used to encrypt the content encryption key using the key encryption key
        AlgorithmID kekEncryptionAlgorithm = AlgorithmID.aes_256_CBC;

        // Parameters used for generating the key encryption key from the password
        int kekLengthInBytes = 32;
        // RFC 2898: a minimum of 1000 iterations is recommended
        int nIterations = 1000;
        // RFC 2898: recommended for salt selection should be at least eight octets (64 bits) long
        byte[] salt = new byte[64];
        // generate a random salt
        EntrustProvider.getSecurityProvider().getSecureRandom().nextBytes(salt);

        // Parameters used for creating the enveloped data
        int blocksize = 256;

        // Generate the key encryption key from the password
        PBEKeyAndParameterSpec derivationParameters = new PBEKeyAndParameterSpec(password.toByteArray(), salt, nIterations, kekLengthInBytes);
        // Create a recipientInfo for this key encryption key
        RecipientInfo recipientInfo = new PasswordRecipientInfo(password, keyDeriviationAlgorithm, derivationParameters, kekEncryptionAlgorithm, null);

        System.out.println("Reading plaintext source file: " + fileName);
        BufferedInputStream plaintextFile = new BufferedInputStream(new FileInputStream(fileName));

        System.out.println("Creating enveloped data with a password recipient info");
        EnvelopedDataStream envelopedDataStream = new EnvelopedDataStream(plaintextFile,contentEncryptionAlgorithm);
        envelopedDataStream.addRecipientInfo(recipientInfo);

        // Encrypt the data and Write it to a new file. (The output is an ASN.1 encoded CMS message.)
        // The new file has the same file name as the source file but with .p7 appended to it.
        File fileOut = new File(fileName.concat(".p7"));
        System.out.println("Writing enveloped data to file: " + fileOut);
        BufferedOutputStream encryptedFile = new BufferedOutputStream(new FileOutputStream(fileOut));
        envelopedDataStream.writeTo(encryptedFile, blocksize);

        plaintextFile.close();
        encryptedFile.close();
    }


    private static void readEnvelopedData(String fileName, SecureStringBuffer password) throws Exception {
        System.out.println("Reading encrypted source file: " + fileName);
        FileInputStream encryptedFile = new FileInputStream(fileName);
        EnvelopedDataStream envelopedDataStream = new EnvelopedDataStream(encryptedFile);
        EncryptedContentInfoStream eci = envelopedDataStream.getEncryptedContentInfo();

        // Retrieve the first password recipient
        RecipientInfo[] recipients = envelopedDataStream.getRecipientInfos();
        boolean recipientFound = false;
        int recipientIndex = 0;
        if (recipients != null) {
            for (recipientIndex=0; recipientIndex < recipients.length; recipientIndex++) {
                if (recipients[recipientIndex].isPasswordRequired()) {
                    recipientFound = true;
                    break;
                }
            }
        }

        if (!recipientFound) {
            System.out.println("PasswordRecipientInfo not found");
            return;
        }

        System.out.println("Decrypting enveloped data for recipient");
        envelopedDataStream.setupCipher(password, recipientIndex);

        System.out.println("Decrypted contents:\n");
        InputStream data = envelopedDataStream.getInputStream();
        byte[] buf = new byte[1024];
        int bytesRead;
        while ((bytesRead = data.read(buf)) > 0) {
            // Do something useful
            // The following simply displays the original, plaintext contents
            System.out.print( new String(Arrays.copyOfRange(buf, 0, bytesRead), StandardCharsets.UTF_8));
        }
        System.out.println();

        data.close();
        encryptedFile.close();
    }
}
