//===========================================================================
//
// Copyright (c)  1998-2024 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.credentials;

import com.entrust.toolkit.User;
import com.entrust.toolkit.credentials.CredentialCreator;
import com.entrust.toolkit.credentials.CredentialReader;
import com.entrust.toolkit.credentials.CredentialWriter;
import com.entrust.toolkit.credentials.FilenameProfileWriter;
import com.entrust.toolkit.util.AuthorizationCode;
import com.entrust.toolkit.util.ManagerTransport;
import com.entrust.toolkit.util.SecureStringBuffer;
import com.entrust.toolkit.x509.directory.JNDIDirectory;

/**
 * This sample class shows how to create a new Entrust digital identity
 * and write it to an epf file. It accepts command line arguments
 * that specify the address of the CA and Directory, the reference number
 * and authorization code, name of the epf file, and a password.
 * 
 * <p>To run this sample, you need an Entrust PKI, and a reference number and
 * authorization code issued by that PKI.</p>
 *  
 * <p>For example, from the examples directory:</p>
 * <pre>
 * javac -sourcepath source -d classes -classpath ../lib/enttoolkit.jar source/com/entrust/toolkit/examples/credentials/CreateCredential.java
 * java -classpath ..\lib\enttoolkit.jar;classes com.entrust.toolkit.examples.credentials.CreateCredential
 * 127.0.0.1 829 127.0.0.1 389 51997123 FAKE-AUTH-CODE MyNewIdentity.epf MyPassword!
 * </pre>
 */
public class CreateCredential {
    public static void main(String[] args) {
        // Ensure that the required command-line arguments were provided
        if (args.length != 8) {
            System.out.println("\nCreateCredential - Creates a new user");
            System.out.println(
                "\nUsage:  java CreateCredential <manager IP> <manager port> <directory IP> <directory port> <refnum> <authcode> <epf name> <password>\n");
            return;
        }

        // Create the user
        System.out.print("Creating a User... ");
        try {

            //  Extract all of the command-line arguments.
            int index = 0;
            String managerIP = args[index++];
            int managerPort = Integer.parseInt(args[index++]);
            String directoryIP = args[index++];
            int directoryPort = Integer.parseInt(args[index++]);
            SecureStringBuffer secureRefNum = new SecureStringBuffer(new StringBuffer(args[index++]));
            AuthorizationCode secureAuthCode = new AuthorizationCode(new StringBuffer(args[index++]));
            String epf = args[index++];
            SecureStringBuffer securePassword = new SecureStringBuffer(new StringBuffer(args[index++]));

            // Create a user object that has a connection to both the Security 
            // Manager and the Directory
            User user = new User();
            ManagerTransport transport = new ManagerTransport(managerIP, managerPort);
            JNDIDirectory directory = new JNDIDirectory(directoryIP, directoryPort);
            user.setConnections(directory, transport);

            // Set-up the user with a credential-writer that is used to write the 
            // user's Digital Identity to an Entrust file-based Digital Identity 
            // store
            CredentialWriter credentialWriter = new FilenameProfileWriter(epf);
            user.setCredentialWriter(credentialWriter);

            // Log-in the user with credential-reader that is used to create the 
            // user's Digital Identity in software (creates the User)
            CredentialReader credentialReader = new CredentialCreator(secureRefNum, secureAuthCode);
            user.login(credentialReader, securePassword);
            System.out.println("Done");
        } catch (Exception ex) {

            // An error occurred
            System.out.println("Failed");
            ex.printStackTrace();
        }
    }
}