//===========================================================================
//
// Copyright (c)  2003-2024 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.credentials;

import java.util.Date;

import iaik.x509.X509Certificate;
import iaik.x509.X509ExtensionInitException;
import iaik.x509.extensions.PrivateKeyUsagePeriod;

import com.entrust.toolkit.User;
import com.entrust.toolkit.credentials.CredentialReader;
import com.entrust.toolkit.credentials.FilenameProfileReader;
import com.entrust.toolkit.util.SecureStringBuffer;

/**
 * Outputs information about when a digital identity's verification certificate, encryption 
 * certificate and private signing key expire and start trying to update.
 * <p>For example, from the examples directory:</p>
 * <pre>
 * javac -sourcepath source -d classes -classpath ../lib/enttoolkit.jar source/com/entrust/toolkit/examples/credentials/PrintKeyDates.java
 * java -classpath ..\lib\enttoolkit.jar;classes com.entrust.toolkit.examples.credentials.PrintKeyDates
 * data/userdata/RSAUser1.epf ~Sample7~
 * </pre>
 */
public class PrintKeyDates
{
    final static long ONEDAYINMILLIS = 86400000L;
    final static long ONEHUNDREDDAYSINMILLIS = 8640000000L;
    final static double FIFTYPERCENT = 0.50;

    /**
     * The main driver for the PrintKeyDates example.
     * 
     * @param args
     *     Program arguments, see class documentation for details on running this sample.
     */
    public static void main (String[] args)
    {

        // First, parse the command line options.
        System.out.println();
        System.out.println("Sample to print out key date information for a user.");
        if (args.length != 2)
        { 
            System.out.println("Usage: PrintKeyDates <epf file> <password>");
            return;
        }

        try
        { 
            int i = 0;
            String epfFile = args[i++];
            String password = args[i++];

            User user = new User();
            CredentialReader reader = new FilenameProfileReader(epfFile);
            
            System.out.println("Logging in to Entrust identity: " + epfFile + "...");            
            user.login(reader, new SecureStringBuffer(new StringBuffer(password)));
            
            // Now let's print out the validity periods... 
            System.out.println();
            System.out.println("Verification Certificate Information");
            System.out.println();
            printSigningCertValidity(user.getVerificationCertificate());

            System.out.println();
            System.out.println("Encryption Certificate Information");
            System.out.println();
            printEncryptionCertValidity(user.getEncryptionCertificate());

            System.out.println();
            System.out.println("Done printing information, logging out... ");
            
            user.logout();
            
        }
        catch (Exception e) 
        {
            e.printStackTrace();
        }

    }

    /**
     * Prints out the verification certificate validity dates and the signing private
     * key validity dates. The signing key's lifetime is calculated and output, as well 
     * as when the signing private key will start trying to update.
     * 
     * @throws X509ExtensionInitException
     *     if there is a problem finding the PrivateKeyUsagePeriod extension within the certificate
     */
    public static void printSigningCertValidity(X509Certificate sigCert) throws X509ExtensionInitException
    {
        if (sigCert == null)
        {
            return;
        }

        Date certValidFrom = sigCert.getNotBefore();
        Date certValidTo = sigCert.getNotAfter();
        
        System.out.println("The signing certificate is valid from: " + certValidFrom.toString());
        System.out.println("The signing certificate is valid to: " + certValidTo.toString());
        
        PrivateKeyUsagePeriod usagePeriod=(PrivateKeyUsagePeriod)(sigCert.getExtension(PrivateKeyUsagePeriod.oid));
        
        Date privValidFrom = usagePeriod.getNotBefore();
        Date privValidTo = usagePeriod.getNotAfter();
        
        System.out.println();
        System.out.println("The signing private key is valid from: " + privValidFrom.toString());
        System.out.println("The signing private key is valid to: " + privValidTo.toString());
        
        //here are the calculations. First, find the lifetime of the private signing key
        Date privKeyLifeTime = new Date(privValidTo.getTime() - certValidFrom.getTime());
        long lifeLength = privKeyLifeTime.getTime();
        
        System.out.println();
        System.out.println("The signing private key has a lifetime length of (in millis): " + lifeLength);
        System.out.println("The signing private key has a lifetime length of (in days): " + lifeLength/ONEDAYINMILLIS);
        
        //The transition period is the lesser of one hundred days or fifty percent of the private key lifetime  
        long halfLife = (long)(privKeyLifeTime.getTime() * FIFTYPERCENT );
        long validLifeLength;
        if (ONEHUNDREDDAYSINMILLIS < halfLife)
        {
            validLifeLength = ONEHUNDREDDAYSINMILLIS;
        }
        else
        {
            validLifeLength = halfLife;
        }
        
        //Now we have how long it is valid, find out when it will need an update.
        //The key starts trying to update at the expiry date less the transition period length
        Date dateToUpdate = new Date();
        dateToUpdate.setTime(privValidTo.getTime() - validLifeLength);
        
        System.out.println();
        System.out.println("This is when the signing private key will start trying to update: " + dateToUpdate.toString());
    }

    /**
     * Prints out the encryption certificate validity dates. The public encryption key's
     * lifetime is calculated, as well as when the encryption public key will start 
     * trying to update.
     */
    public static void printEncryptionCertValidity(X509Certificate encCert)
    {
        if( encCert == null )
        {
            return;
        }

        Date certValidFrom = encCert.getNotBefore();
        Date certValidTo = encCert.getNotAfter();

        System.out.println("The encryption certificate is valid from: " + certValidFrom.toString());
        System.out.println("The encryption certificate is valid to: " + certValidTo.toString());

        //here are the calculations...
        Date certLifeTime = new Date(certValidTo.getTime() - certValidFrom.getTime());

        long halfLife = (long)(certLifeTime.getTime() * FIFTYPERCENT );

        long validLifeLength;

        if (ONEHUNDREDDAYSINMILLIS < halfLife )
        {
            validLifeLength = ONEHUNDREDDAYSINMILLIS;
        }
        else
        {
            validLifeLength = halfLife;
        }

        //Now we have how long it is valid, find out when it will need an update...
        Date dateToUpdate = new Date();
        dateToUpdate.setTime(certValidTo.getTime() - validLifeLength);

        System.out.println();
        System.out.println("This is when the encryption cert will start trying to update: " + dateToUpdate.toString());

    }

}

