//===========================================================================
//
// Copyright (c)  2003-2024 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.crypto;

import iaik.asn1.ObjectID.Algorithms;
import iaik.asn1.structures.AlgorithmID;
import iaik.security.cipher.PBEKeyBMP;
import iaik.security.spec.PBES2ParameterSpec;
import iaik.utils.Util;

import java.security.spec.AlgorithmParameterSpec;

import javax.crypto.Cipher;
import javax.crypto.SecretKey;
import javax.crypto.spec.PBEParameterSpec;

import com.entrust.toolkit.security.provider.Entrust;
import com.entrust.toolkit.security.provider.Initializer;

/**
 * This example demonstrates how to use Entrust's implementation of a 
 * password-based encryption (PBES2) Cipher algorithm from the Java Cryptography 
 * Architecture (JCA).
 * 
 * <p>
 * It shows how to create a PBE key from a password, create the PBES2 parameters, 
 * and then how to use the key and parameters in conjunction with an PBES2 Cipher 
 * to encrypt and decrypt data.
 * </p>
 * 
 * <p>
 * The JCA refers to a framework for accessing and developing cryptographic 
 * functionality for the Java platform.  For further information on how the JCA
 * is used, please refer to Java's 
 * <a href=http://download.oracle.com/javase/1.5.0/docs/guide/security/index.html>
 * Security Documentation</a> documentation.
 * </p>
 */
public class CipherExamplePBES2 {

    /** Test data. */
    private static final byte[] TEST_DATA = new byte[] { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 };

    /**
     * The main program.
     * 
     * @param args
     *      program arguments
     */
    public static void main(String args[]) {

        // Ensure that the arguments are valid
        int size = args.length;
        int iterationCount = 10;
        if (size < 1 || size > 2) {
            System.out.println("Usage: CipherExamplePBES2 <password> [<iterationCount>]");
            return;
        } else if (size == 2) {
            try {
                int i = Integer.parseInt(args[1]);
                if (i > 0)
                    iterationCount = i;
            } catch (NumberFormatException e) {
                // Second argument does not represent an integer iteration 
                // count.. ignore and use default iterator count
            }
        }
        String password = args[0];
        // Ensure that Entrust's providers have been installed
        System.out.print("Installing providers... ");
        Initializer.getInstance().setProviders(Initializer.MODE_NORMAL);
        System.out.println("DONE");
        System.out.println();

        try {
            // Request an instance of a PBES2 cipher from the Entrust provider.
            Cipher cipher = Cipher.getInstance("PBES2", "Entrust");

            // Prepare the the necessary PBE parameters and a key created from 
            // the password
            System.out.print("Creating key and parameters... ");
            SecretKey key = new PBEKeyBMP(password);
            byte[] salt = new byte[8];
            Entrust.getDefaultSecureRandomInstance().nextBytes(salt);

            // Test PBES2 with PBKDF2 (HmacSha256) and AES-CBC-128 to generate secret key
            byte[] iv = new byte[16]; 
            Entrust.getDefaultSecureRandomInstance().nextBytes(iv);
            AlgorithmParameterSpec paramSpec = new PBES2ParameterSpec(salt, iterationCount, AlgorithmID.pbkdf2,
					new AlgorithmID(Algorithms.id_hmacWithSHA256), AlgorithmID.aes_128_CBC, iv);
            System.out.println("DONE");
            System.out.println("salt: " + Util.toString(salt));
            System.out.println("iterationCount: " + iterationCount);
            System.out.println("iv: " + Util.toString(iv));
            System.out.println();
           
            // Initialize the cipher algorithm for encryption and encrypt
            // some data
            System.out.print("Encrypting data... ");
            cipher.init(Cipher.ENCRYPT_MODE, key, paramSpec);
            byte[] ciphertext = cipher.doFinal(TEST_DATA);
            System.out.println("DONE");
            System.out.println("input plaintext:   " + Util.toString(TEST_DATA));
            System.out.println("output ciphertext: " + Util.toString(ciphertext));
            System.out.println();

            // Initialize the cipher algorithm for decryption and decrypt the
            // ciphertext 
            System.out.print("Decrypting data... ");
            cipher.init(Cipher.DECRYPT_MODE, key, paramSpec);            
            byte[] plaintext = cipher.doFinal(ciphertext);
            System.out.println("DONE");
            System.out.println("input ciphertext: " + Util.toString(ciphertext));
            System.out.println("output plaintext: " + Util.toString(plaintext));
        } catch (Exception e) {
            e.printStackTrace();
        }
    }
}