//===========================================================================
//
// Copyright (c)  2003-2024 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.crypto;

import iaik.asn1.structures.AlgorithmID;
import iaik.utils.Util;

import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.spec.AlgorithmParameterSpec;

import javax.crypto.Cipher;

import com.entrust.toolkit.security.provider.Initializer;
import com.entrust.toolkit.security.spec.RSAOAEPParameterSpec;

/**
 * This example demonstrates how to use Entrust's 'RSA-OAEP' implementation of a Cipher 
 * algorithm from the Java Cryptography Architecture (JCA).
 * 
 * <p>
 * It shows how to generate an RSA public/private key pair, and then how to use
 * this key pair in conjunction with an RSA Cipher to encrypt and decrypt data.
 * </p>
 * 
 * <p>
 * The JCA refers to a framework for accessing and developing cryptographic 
 * functionality for the Java platform.  For further information on how the JCA
 * is used, please refer to Java's 
 * <a href=http://download.oracle.com/javase/1.5.0/docs/guide/security/index.html>
 * Security Documentation</a> documentation.
 * </p>
 */
public class CipherExampleRSAOAEP {

    /** Test data. */
    private static final byte[] TEST_DATA = new byte[] { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 };

    /**
     * The main program.
     * 
     * 
     * @param args
     *      program arguments
     */
    public static void main(String[] args) {

        // Ensure that the arguments are valid
        int size = args.length;
        int strength = 1024;
        if (size == 1) {
            try {
                strength = Integer.parseInt(args[0]);
            } catch (NumberFormatException e) {
                // First argument does not represent an integer strength.. ignore
                // and use default strength
            }
        } else if (size > 1) {
            System.out.println("Usage: CipherExampleRSA [<strength>]");
            return;
        }

        // Ensure that Entrust's providers have been installed
        System.out.print("Installing providers... ");
        Initializer.getInstance().setProviders(Initializer.MODE_NORMAL);
        System.out.println("DONE");
        System.out.println();

        try {
            // Request an instance of the RSA cipher and key-pair generation 
            // algorithms from the Entrust provider.  The requested RSA cipher
            // will use block-type 2, and PKCS1 padding.
            Cipher cipher = Cipher.getInstance("RSA/OAEP/PKCS1Padding", "Entrust");

            // Customized RSA OAEP parms
            RSAOAEPParameterSpec oaepParameters = new RSAOAEPParameterSpec();
            oaepParameters.setHashAlgorithm(AlgorithmID.sha256);

            AlgorithmID maskGenerationAlgorithm = AlgorithmID.mgf1;
            maskGenerationAlgorithm.setParameter(AlgorithmID.sha256.toASN1Object());
            oaepParameters.setMaskGenAlgorithm(maskGenerationAlgorithm);
            oaepParameters.setPParameter(new byte[] { 0, 2, 0 });

            AlgorithmParameterSpec spec = oaepParameters;
            
            KeyPairGenerator keyGen = KeyPairGenerator.getInstance("RSA", "Entrust");

            // Generate a public/private key pair that can be used with the 
            // RSA cipher
            System.out.print("Generating key pair... ");
            keyGen.initialize(strength);
            KeyPair keyPair = keyGen.generateKeyPair();
            System.out.println("DONE");
            System.out.println();
            
            // Note: Entrust's implementation does not support any of the             
            // update() methods for the RSA cipher.

            // Initialize the cipher algorithm for encryption and encrypt
            // some data
            System.out.print("Encrypting data... ");            
            cipher.init(Cipher.ENCRYPT_MODE, keyPair.getPublic(), spec);          
            byte[] ciphertext = cipher.doFinal(TEST_DATA);
            System.out.println("DONE");
            System.out.println("input plaintext:   " + Util.toString(TEST_DATA));            
            System.out.println("output ciphertext: " + Util.toString(ciphertext));
            System.out.println();

            // Initialize the cipher algorithm for decryption and decrypt the
            // ciphertext 
            System.out.print("Decrypting data... ");
            cipher.init(Cipher.DECRYPT_MODE, keyPair.getPrivate(), spec);
            byte[] plaintext = cipher.doFinal(ciphertext);
            System.out.println("DONE");
            System.out.println("input ciphertext: " + Util.toString(ciphertext));
            System.out.println("output plaintext: " + Util.toString(plaintext));            
        } catch (Exception e) {
            e.printStackTrace();
        }
    }
}