//===========================================================================
//
// Copyright (c)  2003-2024 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.crypto;

import iaik.utils.Util;

import javax.crypto.KeyGenerator;
import javax.crypto.Mac;
import javax.crypto.SecretKey;

import com.entrust.toolkit.security.provider.Initializer;

/**
 * This example demonstrates how to use Entrust's 'DESede' implementation of a
 * Message Authentication Code (MAC) algorithm from the Java Cryptography Architecture (JCA).
 * 
 * <p>
 * It shows how to generate a DESede key, and then how to use this key in 
 * conjunction with a DESede MAC to create message authentication codes.
 * </p>
 * 
 * <p>
 * The JCA refers to a framework for accessing and developing cryptographic 
 * functionality for the Java platform.  For further information on how the JCA
 * is used, please refer to Java's 
 * <a href=http://download.oracle.com/javase/1.5.0/docs/guide/security/index.html>
 * Security Documentation</a> documentation.
 * </p>
 */
public class MacExampleDESede {

    /** Test data. */
    private static final byte[] TEST_DATA = new byte[] { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 };

    /**
     * The main program.
     * 
     * 
     * @param args
     *      not used
     */
    public static void main(String[] args) {

        // the strength of the key to generate.
        int strength = 192;

        // Ensure that Entrust's providers have been installed
        System.out.print("Installing providers... ");
        Initializer.getInstance().setProviders(Initializer.MODE_NORMAL);
        System.out.println("DONE");
        System.out.println();

        try {
            // Request an instance of the DESede mac and key generation 
            // algorithms from the Entrust provider
            Mac mac = Mac.getInstance("DESede", "Entrust");
            KeyGenerator keyGen = KeyGenerator.getInstance("DESede", "Entrust");

            // Generate a key that can be used with the DESede mac
            System.out.print("Generating key... ");
            keyGen.init(strength);
            SecretKey key = keyGen.generateKey();
            System.out.println("DONE");
            System.out.println();

            // Initialize the mac algorithm and calculate a mac over some data
            System.out.print("Calculating MAC... ");
            mac.init(key);
            mac.update(TEST_DATA);
            byte[] mac1 = mac.doFinal();
            System.out.println("DONE");
            System.out.println("MAC: " + Util.toString(mac1));
            System.out.println();

            // Re-calculate the mac over the same data
            System.out.print("Re-calculating MAC... ");
            byte[] mac2 = mac.doFinal(TEST_DATA);
            System.out.println("DONE");
            System.out.println("MAC: " + Util.toString(mac2));
        } catch (Exception e) {
            e.printStackTrace();
        }
    }
}