//===========================================================================
//
// Copyright (c)  1999-2024 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.ecdsa;

import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.Signature;
import java.security.spec.ECParameterSpec;

import com.entrust.toolkit.security.crypto.ec.EcParameterFactory;
import com.entrust.toolkit.security.provider.Initializer;

/**
 * Demonstrates how to generate an EC key pair in a specific EC domain and then
 * how to use that key pair with the ECDSA digital signature algorithm to
 * generate and then verify a signature over a message.
 */
public class ECDSAExample
{

    /**
     * Sample Application.
     */
    public static void main(String[] args)
    {
        try {
            // Ensure the Entrust and IAIK cryptographic security providers have
            // been installed
            Initializer.getInstance().setProviders(Initializer.MODE_NORMAL);

            // Select an EC domain in which the EC key pair will be generated -
            // we choose NIST P-256 curve (OID: "1.2.840.10045.3.1.7")
            //
            // Note: all named EC domains from ANSI X9.62-2005 are supported;
            // refer to EcParameterFactory for a complete list of supported EC
            // domains.
            ECParameterSpec parms = EcParameterFactory.getInstance();
            // Alternatively, you could specify the above EC domains using the
            // standard ECGenParameterSpec class (from the JCE) as follows:
            //
            // ECGenParameterSpec parms = new java.security.spec.ECGenParameterSpec("ansix9p256r1");

            // Generate the key pair
            KeyPairGenerator keyPairGen = KeyPairGenerator.getInstance("EC", "Entrust");
            keyPairGen.initialize(parms);
            KeyPair keyPair = keyPairGen.generateKeyPair();
            System.out.println("DONE generating keypair");

            // Generate a signature (using the private key)
            System.out.print("Creating signature... ");
            Signature signer = Signature.getInstance("SHA1withECDSA", "Entrust");
            byte[] message = "This message will be signed.".getBytes();
            signer.initSign(keyPair.getPrivate());
            signer.update(message);
            byte[] signature = signer.sign();
            System.out.println("DONE");

            // Verify the signature (using the public key)
            System.out.print("Verifying signature... ");
            signer.initVerify(keyPair.getPublic());
            signer.update(message);
            if (signer.verify(signature)) {
                System.out.println("VALID");
            } else {
                System.out.println("INVALID");
                System.out.println("  The message has been tampered");
            }
        } catch (Exception e) {
            System.out.println("An unexpected error occurred");
            e.printStackTrace();
        }
    }
}
