//===========================================================================
//
// Copyright (c)  2001-2024 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.keystore;

import java.io.FileInputStream;
import java.security.KeyStore;
import java.security.cert.Certificate;

import iaik.pkcs.pkcs12.CertificateBag;
import iaik.pkcs.pkcs12.PKCS12;

import com.entrust.toolkit.credentials.EntrustKeyStore;
import com.entrust.toolkit.security.provider.Initializer;
import com.entrust.toolkit.util.SecureStringBuffer;

/**
 * This is a simple example of how a KeyStore can be used without a KeyStore
 * INI file (KeyStore in memory). 
 * 
 * <p>
 * To run the example, an Entrust Profile (epf) and its password must be 
 * provided as well as a PKCS# 12 file and its password.
 * </p>
 *
 * <p>
 * The creates and loads a KeyStore from an input stream attached to an EPF 
 * file.  This loads all keys/certificates from the EPF into the KeyStore.  It
 * then extracts all certificates in a p12 file and adds them to the KeyStore.  
 * Finally, it demonstrates how to extract certificates from the KeyStore.
 * </p>
 */
public class KeyStoreInMemory {

    /**
     * Sample Application
     */
    public static void main(String args[]) {
        // Read the application arguments
        if (args.length != 4) {
            System.out.println("Usage: KeyStoreInMemory <epf> <epf password> <p12> <p12 password>");
            System.exit(1);
        }
        String epf = args[0];
        SecureStringBuffer epfPassword = new SecureStringBuffer(new StringBuffer(args[1]));
        String p12 = args[2];
        SecureStringBuffer p12Password = new SecureStringBuffer(new StringBuffer(args[3]));

        try {
            // Ensure the IAIK and Entrust cryptographic service providers are installed
            System.out.print("Initializing Toolkit... ");
            Initializer.getInstance().setProviders(Initializer.MODE_NORMAL);
            System.out.println("DONE");

            // Load a KeyStore from an EPF file
            System.out.print("Create and load KeyStore from '" + epf + "'... ");
            KeyStore keyStore = KeyStore.getInstance("Entrust");
            keyStore.load(new FileInputStream(epf), epfPassword.toCharArray());
            System.out.println("DONE");

            // Extract certificates from PKCS#12 file and add them to the KeyStore
            // (a read/write certificate store is automatically created)
            System.out.print("Extracting certificate from '" + p12 + "' and adding them to the KeyStore... ");
            PKCS12 pkcs12 = new PKCS12(new FileInputStream(p12));
            pkcs12.decrypt(p12Password.toCharArray());
            Certificate[] certs = CertificateBag.getCertificates(pkcs12.getCertificateBags());
            for (int i = 0; i < certs.length; ++i) {
                keyStore.setCertificateEntry("cert" + i, certs[i]);
            }
            System.out.println("DONE");
            
            // Extract certificates from KeyStore
            System.out.println("\n-- CA certificate: \n" + keyStore.getCertificate(EntrustKeyStore.CA_CERTIFICATE));
            System.out.println("\n-- Encryption certificate: \n"
                + keyStore.getCertificate(EntrustKeyStore.ENCRYPTION_CERTIFICATE));
            System.out.println("\n-- Verification certificate: \n"
                + keyStore.getCertificate(EntrustKeyStore.VERIFICATION_CERTIFICATE));
            for (int i = 0; i < certs.length; ++i) {
                System.out.println("\n-- Certificate 'cert" + i + "': \n" + keyStore.getCertificate("cert" + i));
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            // Ensure the passwords are wiped from memory
            epfPassword.wipe();
            p12Password.wipe();
        }
    }
}