//===========================================================================
//
// Copyright (c)  2006-2024 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.ocsp;

import iaik.x509.X509Certificate;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.URL;
import java.security.cert.CertificateException;

import com.entrust.toolkit.exceptions.CertificationException;
import com.entrust.toolkit.security.provider.Initializer;
import com.entrust.toolkit.x509.CertVerifier;
import com.entrust.toolkit.x509.UserRevocationInfo;
import com.entrust.toolkit.x509.revocation.OCSPConfiguration;

/**
 * This sample demonstrates how to validate a certificate using OCSP revocation checking.  It does not
 * require that a user be logged into the toolkit, as the CertVerifier is used to establish the
 * root of trust.
 * 
 * To enable this checking to be done, the following are required:
 * <p>
 * 1.  The Certificate which will be used to check revocation
 * <p>
 * 2.  The OCSP responders certificate.  This should be the certificate of a trusted OCSPResponder which
 *     has been configured to respond to revocation requests for the certificate being checked for revocation.
 * <p>
 * 3.  The trusted CA certificate.  Used to establish the root of trust.  In most cases this would be
 *     the certificate that issued the certificate being checked for revocation.
 * <p>
 * 4.  If the certificate being checking for revocation contains an <code>AuthorityInfoAccess</code> (AIA)
 * extension which contains the location of the OCSP responder, then this option is not needed.
 * If it does not contain an AIA extensions, then this option is required otherwise revocation
 * checking will fail as there is no responder with which to check revocation.  Specify the accessLocation of the responder.  This should be an http location of the OCSP responder:
 * <p>
 * Usage:
 * <p>
 * java OCSPCheckExample (Cert_to_check_for_revocation) (responder_certificate) (trusted_CA_cert) [-loc (Access location of responder) ]'
 * <p>
 * For example:
 * <p>
 * java OCSPCheckExample User.cer OCSPServer.cer RootCAcert.cer -loc "http://ocsp.openvalidation.org:8080"
 *     
 * @author John Gray
 * @since 7.2
 */

public class OCSPCheckExample {

	/**
	 * @param args The arguments for the sample....
	 */
	public static void main(String[] args) {
		if (args.length < 3) {
			System.out.println("Usage: OCSPCheckExample <Cert to check for revocation> <responder certificate> <trusted CA cert> [-loc <Access location of responder> ]");
			return;
		}

		//Set up the Entrust security providers
		Initializer.getInstance().setProviders(Initializer.MODE_NORMAL);

		X509Certificate CertToCheck;
		X509Certificate responderCert;
		X509Certificate rootCA;
		URL accessLocation = null;

		int index = 0;
		try {
			CertToCheck = new X509Certificate(
					new FileInputStream(args[index++]));
			responderCert = new X509Certificate(new FileInputStream(
					args[index++]));
			rootCA = new X509Certificate(new FileInputStream(args[index++]));

			// Only parse if access location is supplied
			if (args.length > index && args[index].equals("-loc")) {
				accessLocation = new URL(args[++index]);
				index++;
			}

		} catch (CertificateException e) {
			e.printStackTrace();
			return;
		} catch (FileNotFoundException e) {
			e.printStackTrace();
			return;
		} catch (IOException e) {
			e.printStackTrace();
			return;
		}

		//If we got this far, then setup OCSP revocation configuration using API's
		OCSPConfiguration ocsp = new OCSPConfiguration();
		if (accessLocation != null) {
			ocsp.setURLLocation(accessLocation);
		}

		//enable AuthorityInfoAccess - by default this is already true, but set here for example
		ocsp.setAIAChecking(true);

		//Set the responder certificate
		ocsp.addResponderCertificate(responderCert);

		//Turn off the requestor Name - by default it is true.
		ocsp.setIncludeRequestorName(false);
		
		// Add the OCSP configuration setting into a UserRevocationInfo object.
		// This could have been done by reading in a config file as well
		// by using revinfo.parsePropertiesFile()
		UserRevocationInfo revinfo = new UserRevocationInfo();
		revinfo.addOCSPConfiguration(ocsp);

		//Create a certificate verifier, and add the revocation configuration
		CertVerifier certVer = new CertVerifier(rootCA, null, null, revinfo);

		try {
			certVer.validate(CertToCheck);
		}  
		   
		//A Revocation exception extends CertificateException, so catching the
		//Certification exception will also catch any Revocation Exceptions.  
		//This is a simple example of how the contents of a CertificationException
		//can be unwrapped to get more information on problems encountered during
		//revocation checking
		
		catch (CertificationException e) {
	       System.out.println("CertificationException Caught!  Details of Exception:");   
		   int i=1;
		   System.out.print(i++ +".  " );
		   System.out.println(e.getMessage());
		   
		   Exception inner = e.getInnerException();
		   while (inner instanceof CertificationException) {
			   CertificationException certification = (CertificationException)inner;
			   System.out.print(i++ +".  " );
			   System.out.println(inner.getMessage());
			   inner = certification.getInnerException();
		   }
		   System.out.println("Contents of stack trace: ");
		   e.printStackTrace();
		   return;
		}
		   
		// If any other errors occurred, then just print the tack trace
		catch (Exception e) {
	       e.printStackTrace();
		return;
		}

		//If we get this far, certificate was verified!
		System.out.println("Certificate verified and revocation checked using OCSP");
	}

}
