//===========================================================================
//
// Copyright (c)  2006-2024 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.ocsp;

import iaik.x509.X509Certificate;

import java.io.FileInputStream;
import java.io.IOException;
import java.security.cert.CertificateException;

import com.entrust.toolkit.User;
import com.entrust.toolkit.credentials.CredentialReader;
import com.entrust.toolkit.credentials.FilenameProfileReader;
import com.entrust.toolkit.credentials.UserConfigSettings;
import com.entrust.toolkit.exceptions.CertificationException;
import com.entrust.toolkit.exceptions.UserBadPasswordException;
import com.entrust.toolkit.exceptions.UserFatalException;
import com.entrust.toolkit.exceptions.UserNotLoggedInException;
import com.entrust.toolkit.exceptions.UserRecoverException;
import com.entrust.toolkit.util.SecureStringBuffer;
import com.entrust.toolkit.x509.UserRevocationInfo;
import com.entrust.toolkit.x509.revocation.OCSPConfiguration;

/**
 * Sample shows how to setup the Java toolkit to automatically use OCSP
 * for revocation checking of certificates. 
 * <p>
 * Setting up the revocation configuration is done through the UserRevocationInfo
 * object in the Java toolkit.  The API's provided by this object can be used, or
 * a properties file can be read in using the parsePropertiesFile() method.
 * <p>
 * Usage:
 * <p>
 * OCSPConfigExample (Entrust Profile) (password) [-prop (optional configuration file) ]
 * <p>
 * <dl>
 * <dt>profile</dt><dd>Path to EPF file. For example, data/userdata/RSAUser1.epf</dd>
 * <dt>password</dt><dd>The password for the EPF. For example, ~Sample7~</dd>
 * <dt>-prop config</dt><dd>location of optional revocation configuration file</dd>
 * </dl>
 * 
 * For example, from the examples directory, 
 * <pre>
 * java -classpath ../lib/enttoolkit.jar;classes com.entrust.toolkit.examples.ocsp.OCSPConfigExample data/userdata/RSAUser1.epf ~Sample7~ ocspconfig.property
 * </pre>
 * <p>
 * In the above example, the ocspconfig.property would need to contain the location of an OCSP Responder that
 * can be used to check revocation of the certificates contained in RSAUser1.epf.  Setup of the OCSP Responder
 * is beyond the scope of this sample.  A number of sample configuration files which can be used
 * with this sample are contained in the etjava/examples/data/ocsp/ directory.
 * </p> 
 */

public class OCSPConfigExample {

	public static void main(String args[]) throws UserNotLoggedInException {

		if (args.length < 2) {
			System.out.println("Usage: OCSPConfigExample <profile> <password> [-prop <optional configuration file> ]");
			return;
		}

		int index = 0;
		String profile = args[index++];
		SecureStringBuffer password = new SecureStringBuffer(args[index++].toCharArray());
		String filename = null;
		
        // Only parse properties file if it is supplied	
		if (args.length > index && args[index].equals("-prop")) {
            filename = args[++index];
            index++;
        }

		try {

			UserRevocationInfo revinfo = new UserRevocationInfo();

			// Only parse properties file if it is supplied
			if (args.length > index) {
				filename = args[index++];
			}

			if (filename != null) {
				revinfo.parsePropertiesFile(new FileInputStream(filename));
			}

			else {
				// Create a default OCSP configuration. A default
				// OCSPConfiguration has
				// AIA mode enabled
				OCSPConfiguration ocspconfig = new OCSPConfiguration();
				ocspconfig.setConnectTimeout(5000);
				revinfo.addOCSPConfiguration(ocspconfig);
			}

			// Add Revocation information into the User settings
			UserConfigSettings usersettings = new UserConfigSettings();
			usersettings.setRevocationInfo(revinfo);

			// Add User settings into the user
			User user = new User(usersettings);

			CredentialReader cr = new FilenameProfileReader(profile);
			System.out.println("login");
			user.login(cr, password);
			System.out.println("   done");
	
			//Check revocation status of the verification certificate
			System.out.println("Checking revocation of the users verification certificate");
			user.getCertVerifier().getRevocationManager().check(user.getVerificationCertificate());
			System.out.println("   Revocation check was successful!");
			

		} catch (IOException e) {
			System.out.println("An IOException was thrown with the following message: " + e.getMessage());
			e.printStackTrace();
		} catch (UserRecoverException e) {
			System.out.println("A UserRecoverException was thrown with the following message: " + e.getMessage());
			e.printStackTrace();
		} catch (UserBadPasswordException e) {
			System.out.println("A UserBadPasswordException was thrown with the following message: " + e.getMessage());
			e.printStackTrace();
		} catch (UserFatalException e) {
			System.out.println("A UserFatalException was thrown with the following message: " + e.getMessage());
			e.printStackTrace();
		} catch (CertificateException e) {
			System.out.println("A CertificateException was thrown with the following message: " + e.getMessage());
			e.printStackTrace();
		} catch (CertificationException e) {
		    System.out.println("CertificationException Caught!  Details of Exception:");   
			int i=1;
			System.out.print(i++ +".  " );
			System.out.println(e.getMessage());
			   
			Exception inner = e.getInnerException();
			while (inner instanceof CertificationException) {
			   CertificationException certification = (CertificationException)inner;
			   System.out.print(i++ +".  " );
			   System.out.println(inner.getMessage());
			   inner = certification.getInnerException();
		    }
		    System.out.println("Contents of stack trace: ");
		    e.printStackTrace();
		}
	}
}
