//===========================================================================
//
// Copyright (c)  2003-2024 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.pkcs10;

import java.io.File;
import java.io.FileOutputStream;
import java.security.KeyPair;
import java.security.KeyPairGenerator;

import iaik.asn1.structures.AlgorithmID;
import iaik.asn1.structures.Name;
import iaik.pkcs.pkcs10.CertificateRequest;
import iaik.utils.RFC2253NameParser;
import iaik.utils.Util;

import com.entrust.toolkit.security.provider.Initializer;

/**
 * This example demonstrates how to create a PKCS#10 CertificateRequest ASN1
 * structure.
 * 
 * <p>A CertificateRequest provides a generic manner of requesting 
 * certification of a key pair from a Certificate Authority (CA).
 * 
 * <p>To run this sample, you need a subject DN, the name of a file where the 
 * private key will be written in DER encoded PKCS 8 format, and the name of a 
 * file where the certificate request will be written in DER encoded PKCS 10 
 * format.
 * 
 * <p>For example, from the examples directory:</p>
 * <pre>
 * javac -sourcepath source -d classes -classpath ../lib/enttoolkit.jar source/com/entrust/toolkit/examples/pkcs10/CreateP10CertificateRequest.java
 * java -classpath ..\lib\enttoolkit.jar;classes com.entrust.toolkit.examples.pkcs10.CreateP10CertificateRequest
 * cn=MyUser ./MyUser.p8 ./MyUser.p10
 * </pre>
 */
public class CreateP10CertificateRequest {

    public static void main(String[] args) {

        // Ensure that the required command-line arguments were provided
        if (args.length != 3) {
            System.out.println("\nCreateP10CertificateRequest - Create a PKCS #10 certificate request");
            System.out
                .println("\nUsage:  java CreateP10CertificateRequest <subject DN> <P8 private key output file> <P10 certReq output file>");
            return;
        }

        // Ensure that Entrust's providers have been installed
        System.out.print("Installing JCA providers... ");
        Initializer.getInstance().setProviders(Initializer.MODE_NORMAL);
        System.out.println("DONE");

        try {
            // Extract all of the command-line arguments.
            int index = 0;
            Name subjectDN = new RFC2253NameParser(args[index++]).parse();
            File p8File = new File(args[index++]);
            File p10File = new File(args[index++]);

            // Generate an RSA public/private key pair
            KeyPairGenerator keyGen = KeyPairGenerator.getInstance("RSA", "Entrust");
            System.out.print("Generating key pair... ");
            keyGen.initialize(1024);
            KeyPair keyPair = keyGen.generateKeyPair();
            System.out.println("DONE");

            // Create the certificate request by including the subject DN
            // and the public key, and then sign the certificate request
            // with a private key
            System.out.print("Creating the CertificateRequest... ");
            CertificateRequest certReq = new CertificateRequest(keyPair.getPublic(), subjectDN);
            certReq.sign(AlgorithmID.sha1WithRSAEncryption, keyPair.getPrivate());
            System.out.println("DONE");
            System.out.println("\nDER encoded CertificateRequest structure:");
            System.out.println(Util.toString(certReq.toByteArray()));

            // Output P10 certificate request and P8 private key
            System.out.println();
            FileOutputStream fos = new FileOutputStream(p8File);
            System.out.println("Private key in P8 format written to: " + p8File.getAbsolutePath());
            fos.write(keyPair.getPrivate().getEncoded());
            fos.close();
            fos = new FileOutputStream(p10File);
            certReq.writeTo(fos);
            fos.close();
            System.out.println("Certificate request in P10 format written to: " + p10File.getAbsolutePath());
        } catch (Exception e) {
            System.out.println("Failed");
            e.printStackTrace();
        }
    }
}