//===========================================================================
//
// Copyright (c)  2003-2024 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.pkcs11;

import iaik.x509.X509Certificate;

import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.security.PrivateKey;
import java.security.Signature;

import javax.crypto.Cipher;

import com.entrust.toolkit.User;
import com.entrust.toolkit.credentials.CredentialReader;
import com.entrust.toolkit.credentials.CredentialWriter;
import com.entrust.toolkit.credentials.TokenCredentialRecoverer;
import com.entrust.toolkit.credentials.TokenWriter;
import com.entrust.toolkit.pkcs11.PKCS11LibraryConnection;
import com.entrust.toolkit.util.AuthorizationCode;
import com.entrust.toolkit.util.ManagerTransport;
import com.entrust.toolkit.util.SecureStringBuffer;
import com.entrust.toolkit.x509.directory.JNDIDirectory;

/*******************************************************************************
 * 
 * This class contains sample code for recovering a user on a Cryptoki device 
 * (token).
 * 
 * <P>
 * It requires several parameters prior to user recovery, which are either read 
 * from <CODE>System.in</CODE> or from command line arguments.  The following 
 * usages are supported:
 * </P>
 *
 * <P><DL>
 *      <DT><B>RecoverUser</B></DT>
 *      <DD>Parameters read from <CODE>System.in</CODE></DD>
 * </DL></P>
 *  
 * <P><DL>
 *      <DT><B>RecoverUser &#60;pki address&#62; &#60;pki port&#62; 
 *          &#60;dir address&#62; &#60;dir port&#62; &#60;ref num&#62; 
 *          &#60;auth code&#62; &#60;slot ID&#62; &#60;p11 lib&#62; 
 *          &#60;password&#62; &#60;apf dir&#62; &#60;apf name&#62;</B></DT>
 *      <DD>Parameters read from command line arguments</DD>
 * </DL></P>
 * 
 * <P>
 * The parameters provide all the information necessary to recover a user on a
 * token.  Below is an explanation of each parameter with example values:
 * </P>
 * 
 * <P><DL>
 *      <DT><B>&#60;pki address&#62;</B> ex: 'myTestPki' or '1.2.3.4'</DT>
 *      <DD>The full domain name or IP address of the Security Manager, which the 
 *          user has been setup on</DD>
 *      <DT><B>&#60;pki port&#62;</B> ex: 829</DT>
 *      <DD>The PKIX-CMP communication port of the Security Manager</DD>
 *      <DT><B>&#60;dir address&#62;</B> ex: 'myTestDir' or '5.6.7.8'</DT>
 *      <DD>The full domain name or IP address of the Directory for the Security 
 *          Manager</DD>
 *      <DT><B>&#60;dir port&#62;</B> ex: 389</DT>
 *      <DD>The LDAP communication port of the Directory</DD>
 *      <DT><B>&#60;ref num&#62;</B> ex: 51711731</DT>
 *      <DD>The Reference Number for the user that was setup on the Security 
 *          Manager</DD>
 *      <DT><B>&#60;auth code&#62;</B> ex: U3AC-8NPH-8FKF</DT>
 *      <DD>The Authorization Code for the user that was setup on the Security
 *          Manager</DD>
 *      <DT><B>&#60;slot ID&#62;</B> ex: 2</DT>
 *      <DD>The slot identifier of the token that the user will be recovered 
 *          on</DD>
 *      <DT><B>&#60;p11 lib&#62;</B> ex: slbck.dll</DT>
 *      <DD>The PKCS11 Library of the token; contains the PKCS11 API necessary for
 *          communicating with the token</DD>
 *      <DT><B>&#60;password&#62;</B> ex: 00000000</DT>
 *      <DD>The password for the user, which is also the PIN of the Normal User on 
 *          the token</DD>
 *      <DT><B>&#60;apf dir&#62;</B> ex: c:/temp</DT>
 *      <DD>The directory where the Auxillary Profile will be stored</DD>
 *      <DT><B>&#60;apf name&#62;</B> ex: TestUserRecover</DT>
 *      <DD>The name of the Auxillary Profile (without the .apf extension)</DD>
 * </DL></P> 
 * 
 * <P>
 * After the user is recovered, an encryption/decryption test and a sign/verify 
 * test are run.  The encryption/decryption test demonstrates how the user's 
 * encryption certificate and decryption private key can be used together to 
 * protect sensitive information.  The sign/verify test demonstrates how the 
 * user's signing private key and verification certificate can be used together to 
 * guarantee data integrity.
 * </P>
 * 
 * <P><DL>
 *      <DT>Note</DT>
 *      <DD>For user's that are recovered against a 7.0 or later Entrust Security
 *          Manager, it is possible that the user may not have a encryption
 *          certificate and decryption private key.  In such cases, the 
 *          encryption/decryption test will NOT be run.</DD>
 *      <DT>Note</DT>
 *      <DD>This sample requires that the token already be initialized with
 *          a Normal User; the Normal User PIN is required as a parameter</DD>
 * </DL></P>
 */
public final class RecoverUser {

    /******************************************************************************/
    /*                               SAMPLE PROGRAM                               */
    /******************************************************************************/

    public static void main(String[] args) {

        // Read the user parameters
        System.out.println("User Recovery on a Token");
        System.out.println("------------------------");
        try {
            readParams(args);
        } catch (Exception e) {
            System.out.println("ERROR: invalid parameters.");
            e.printStackTrace();
            System.exit(0);
        }

        // Recover the credentials on the token
        System.out.println("\nAttempting to recover a user on the token...");
        User user = new User();
        try {
            // Set the user's connection to both the Security Manager and the 
            // Directory            
            JNDIDirectory directory = new JNDIDirectory(m_directoryAddress, m_directoryPort);
            ManagerTransport managerTransport = new ManagerTransport(m_managerAddress, m_managerPort);
            user.setConnections(directory, managerTransport);

            // Create a credential reader that is used to recover a set of
            // credentials on a token
            SecureStringBuffer secureRefNum = new SecureStringBuffer(new StringBuffer(m_refNum));
            AuthorizationCode secureAuthCode = new AuthorizationCode(new StringBuffer(m_authCode));
            PKCS11LibraryConnection p11LibConn = new PKCS11LibraryConnection(m_p11LibConn);
            CredentialReader tokenCredentialRecoverer =
                new TokenCredentialRecoverer(secureRefNum, secureAuthCode, m_slotId, p11LibConn);

            // Create a credential writer that is used to write a set of 
            // credentials to a token (both entrustPath and entrustUser parameters
            // are required when writing newly created or recovered credentials)
            CredentialWriter tokenWriter = new TokenWriter(m_entrustPath, m_entrustUser);
            user.setCredentialWriter(tokenWriter);

            // Log the user in (this recovers the credentials)
            SecureStringBuffer securePassword = new SecureStringBuffer(new StringBuffer(m_password));
            user.login(tokenCredentialRecoverer, securePassword);
        } catch (Exception e) {
            System.out.println("ERROR: user recovery failed.");
            e.printStackTrace();
            System.exit(0);
        }
        System.out.println("SUCCESS: user recovery complete");

        // Test Signing/Verification
        System.out.println("\n\nSign/Verify Test");
        System.out.println("----------------");
        try {
            PrivateKey signingKey = user.getSigningKey();
            X509Certificate verificationCertificate = user.getVerificationCertificate();
            if (signingKey != null && verificationCertificate != null) {
                Signature signer = Signature.getInstance("SHA-1/" + signingKey.getAlgorithm(), "Entrust");

                // Sign
                System.out.println("\nAttempt sign operation using the user's signing key...\n");
                signer.initSign(signingKey);
                signer.update(TEST_DATA);
                System.out.println("data to sign: " + iaik.utils.Util.toString(TEST_DATA));
                byte[] signature = signer.sign();
                System.out.println("signature:    " + iaik.utils.Util.toString(signature));

                // Verify
                System.out.println(
                    "\nAttempt verify operation using the user's verification certificate with serial number '"
                        + verificationCertificate.getSerialNumber()
                        + "'...\n");
                signer.initVerify(verificationCertificate);
                signer.update(TEST_DATA);
                System.out.println("data that was signed: " + iaik.utils.Util.toString(TEST_DATA));
                boolean verified = signer.verify(signature);
                System.out.println("signature verified:   " + verified);
            }
        } catch (Exception e) {
            System.out.println("ERROR: sign/verify test failed.");
            try {
                user.logout();
            } catch (Exception e1) {
                // Ignore any exceptions
            }
            e.printStackTrace();
            System.exit(0);
        }

        // Test Encryption/Decryption
        System.out.println("\n\nEncryption/Decryption Test");
        System.out.println("--------------------------");
        try {
            PrivateKey decryptionKey = user.getDecryptionKey();
            X509Certificate encryptionCertificate = user.getEncryptionCertificate();
            if (decryptionKey != null && encryptionCertificate != null) {
                Cipher cipher = Cipher.getInstance(decryptionKey.getAlgorithm(), "Entrust");

                // Encrypt
                System.out.println(
                    "\nAttempt encrypt operation using the user's encryption certificate with serial number '"
                        + encryptionCertificate.getSerialNumber()
                        + "'...\n");
                cipher.init(Cipher.ENCRYPT_MODE, encryptionCertificate.getPublicKey());
                System.out.println("plaintext:  " + iaik.utils.Util.toString(TEST_DATA));
                byte[] ciphertext = cipher.doFinal(TEST_DATA);
                System.out.println("ciphertext: " + iaik.utils.Util.toString(ciphertext));

                // Decrypt
                System.out.println(
                    "\nAttempt decrypt operation using the user's corresponding decryption private key...\n");
                cipher.init(Cipher.DECRYPT_MODE, decryptionKey);
                System.out.println("ciphertext: " + iaik.utils.Util.toString(ciphertext));
                byte[] plaintext = cipher.doFinal(ciphertext);
                System.out.println("plaintext:  " + iaik.utils.Util.toString(plaintext));
            }
        } catch (Exception e) {
            System.out.println("ERROR: encryption/decryption test failed.");
            try {
                user.logout();
            } catch (Exception e1) {
                // Ignore any exceptions
            }
            e.printStackTrace();
            System.exit(0);
        }

        // Log the user out
        try {
            user.logout();
        } catch (Exception e) {
            e.printStackTrace();
            System.exit(0);
        }

        System.out.println("\n\nDONE");
    }

    /******************************************************************************/
    /*                                  CONSTANTS                                 */
    /******************************************************************************/

    private static byte[] TEST_DATA = "Test Data".getBytes();

    /******************************************************************************/
    /*                                 PARAMETERS                                 */
    /******************************************************************************/

    /** The full domain name or IP address of the Security Manager, which the user 
     *  has been setup on. */
    private static String m_managerAddress;

    /** The PKIX-CMP communication port of the Security Manager. */
    private static int m_managerPort;

    /** The full domain name or IP address of the Directory for the Security 
     *  Manager. */
    private static String m_directoryAddress;

    /** The LDAP communication port of the Directory. */
    private static int m_directoryPort;

    /** The Reference Number for the user that was setup on the Security 
     *  Manager. */
    private static String m_refNum;

    /** The Authorization Code for the user that was setup on the Security
     *  Manager. */
    private static String m_authCode;

    /** The slot identifier of the token that the user will be recovered on. */
    private static long m_slotId;

    /** The PKCS11 Library of the token; contains the PKCS11 API necessary for 
     *  communicating with the token. */
    private static String m_p11LibConn;

    /** The directory where the Auxillary Profile will be stored. */
    private static String m_entrustPath;

    /** The name of the Auxillary Profile (without the .apf extension). */
    private static String m_entrustUser;

    /** The password for the user, which is also the PIN of the Normal User on the
     *  token. */
    private static String m_password;

    /******************************************************************************/
    /*                                 PARAMETERS                                 */
    /******************************************************************************/

    /*******************************************************************************
     * 
     * The default constructor.
     * 
     */
    private RecoverUser() {
    }

    /******************************************************************************/
    /*                              PARAMETER PARSING                             */
    /******************************************************************************/

    /*******************************************************************************
     * 
     * Reads all the parameters necessary for recovering a user on a token.
     * 
     * 
     * @param args
     *      the program arguments read in by the main method
     * 
     * @exception Exception
     *      thrown if an error occurs while parsing the parameters
     */
    private static void readParams(String[] args) throws Exception {
        int size = args.length;
        if (size == 0) {
            readParamsFromUser();
        } else if (size == 11) {
            readParamsFromArgs(args);
        } else {
            System.out.println("Usages:\n");
            System.out.println("RecoverUser");
            System.out.println(
                "RecoverUser <pki address> <pki port> <dir address> <dir port> <ref num> <auth code> <slot ID> <p11 lib> <password> <apf dir> <apf name>");
            System.exit(0);
        }
    }

    /*******************************************************************************
     * 
     * Reads all the parameters necessary for recovering a user on a token from
     * a set of string arguments.
     * 
     * 
     * @param args
     *      the program arguments read in by the main method
     * 
     * @exception Exception
     *      thrown if an error occurs while parsing the parameters from the
     *      program arguments
     */
    private static void readParamsFromArgs(String[] args) throws Exception {

        int i = 0;

        // Security Manager location
        m_managerAddress = args[i++];
        m_managerPort = Integer.parseInt(args[i++]);

        // Directory location
        m_directoryAddress = args[i++];
        m_directoryPort = Integer.parseInt(args[i++]);

        // User information
        m_refNum = args[i++];
        m_authCode = args[i++];

        // Token information
        m_slotId = Long.parseLong(args[i++]);
        m_p11LibConn = args[i++];
        m_password = args[i++];

        // Auxillary Profile information
        m_entrustPath = args[i++];
        m_entrustUser = args[i++];
    }

    /*******************************************************************************
     * 
     * Reads all the parameters necessary for recovering a user on a token from
     * <CODE>System.in</CODE>.
     * 
     * 
     * @exception Exception
     *      thrown if an error occurs while parsing the parameters from the
     *      <CODE>System.in</CODE>
     */
    private static void readParamsFromUser() throws Exception {

        BufferedReader in = new BufferedReader(new InputStreamReader(System.in));

        System.out.println(
            "\nPlease enter the following information, all of which is required prior to recovery of a user on a token:\n");

        // Security Manager location
        System.out.println("Security Manager Address (ex: 'myTestPKI' or '1.2.3.4')");
        m_managerAddress = in.readLine();
        System.out.println("Security Manager Port (ex: 829)");
        m_managerPort = Integer.parseInt(in.readLine());

        // Directory location
        System.out.println("\nDirectory Address (ex: 'myTestDir' or '5.6.7.8')");
        m_directoryAddress = in.readLine();
        System.out.println("Directory Port (ex: 389)");
        m_directoryPort = Integer.parseInt(in.readLine());

        // User information
        System.out.println("\nReference Number (ex: 51711731)");
        m_refNum = in.readLine();
        System.out.println("Authorization Code (ex: U3AC-8NPH-8FKF)");
        m_authCode = in.readLine();

        // Token information
        System.out.println("\nToken Slot Identifier (ex: 2)");
        m_slotId = Long.parseLong(in.readLine());
        System.out.println("Token PKCS11 Library (ex: slbck.dll)");
        m_p11LibConn = in.readLine();
        System.out.println("User password (Normal User PIN on the token)");
        m_password = in.readLine();

        // Auxillary Profile information
        System.out.println("\nAuxillary Profile Path (ex: c:/temp)");
        m_entrustPath = in.readLine();
        System.out.println("Auxillary Profile Name (ex: TestRecoverUser)");
        m_entrustUser = in.readLine();
    }
}