//===========================================================================
//
// Copyright (c)  2003-2024 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.pkcs11;

import iaik.x509.X509Certificate;

import java.io.BufferedReader;
import java.io.File;
import java.io.InputStreamReader;
import java.security.PrivateKey;
import java.security.Signature;

import javax.crypto.Cipher;

import com.entrust.toolkit.User;
import com.entrust.toolkit.credentials.CredentialReader;
import com.entrust.toolkit.credentials.CredentialWriter;
import com.entrust.toolkit.credentials.FilenameProfileReader;
import com.entrust.toolkit.credentials.TokenInitializer;
import com.entrust.toolkit.credentials.TokenReader;
import com.entrust.toolkit.pkcs11.PKCS11LibraryConnection;
import com.entrust.toolkit.util.SecureStringBuffer;

/*******************************************************************************
 * 
 * This class contains sample code for writing a set of EPF credentials to
 * a Cryptoki device (token).
 * 
 * <P>
 * It requires several parameters prior to the user credential write operation, 
 * which are either read from <CODE>System.in</CODE> or from command line 
 * arguments.  The following usages are supported:
 * </P>
 *
 * <P><DL>
 *      <DT><B>WriteEpfToToken</B></DT>
 *      <DD>Parameters read from <CODE>System.in</CODE></DD>
 * </DL></P>
 *  
 * <P><DL>
 *      <DT><B>WriteEpfToToken &#60;slot ID&#62; &#60;p11 lib&#62; 
 *          &#60;password&#62; &#60;epf/apf dir&#62; &#60;epf/apf name&#62;</B></DT> 
 *      <DD>Parameters read from command line arguments</DD>
 * </DL></P>
 * 
 * <P>
 * The parameters provide all the information necessary to write the EPF onto
 * the token.  Below is an explanation of each parameter with example values:
 * </P>
 * 
 * <P><DL>
 *      <DT><B>&#60;slot ID&#62;</B> ex: 2</DT>
 *      <DD>The slot identifier of the token where the credentials will be
 *          written</DD>
 *      <DT><B>&#60;p11 lib&#62;</B> ex: slbck.dll</DT>
 *      <DD>The PKCS11 Library of the token; contains the PKCS11 API necessary for
 *          communicating with the token</DD>
 *      <DT><B>&#60;password&#62;</B> ex: 00000000</DT>
 *      <DD>The password for the user, which is also the PIN of the Normal User on 
 *          the token and the password protecting the EPF</DD>
 *      <DT><B>&#60;epf/apf dir&#62;</B> ex: c:/temp</DT>
 *      <DD>The directory where the Entrust Profile file (EPF) is stored and where
 *          the Auxillary Profile (APF) will be stored</DD>
 *      <DT><B>&#60;epf.apf name&#62;</B> ex: TestUser</DT>
 *      <DD>The name of the Entrust Profile file (EPF) and Auxillary Profile file
 *          (without the .epf or .apf extension)</DD>
 * </DL></P> 
 * 
 * <P>
 * After the credentials are written to the token, the token is logged into and
 * an encryption/decryption test and a sign/verify test are run.  The 
 * encryption/decryption test demonstrates how the user's encryption certificate 
 * and decryption private key can be used together to protect sensitive 
 * information.  The sign/verify test demonstrates how the user's signing private 
 * key and verification certificate can be used together to guarantee data 
 * integrity.
 * </P>
 * 
 * <P><DL>
 *      <DT>Note</DT>
 *      <DD>For user's that exist on a 7.0 or later Entrust Security Manager, 
 *          it is possible that the user may not have an encryption certificate 
 *          and decryption private key.  In such cases, the encryption/decryption 
 *          test will NOT be run.</DD>
 *      <DT>Note</DT>
 *      <DD>This sample requires that the token already be initialized with
 *          a Normal User; the Normal User PIN is required as a parameter.  For
 *          the write to be successful, the Normal User PIN must be identical to 
 *          the password protecting the user's EPF.</DD>
 * </DL></P>
 */
public final class WriteEpfToToken {

    /******************************************************************************/
    /*                               SAMPLE PROGRAM                               */
    /******************************************************************************/

    public static void main(String[] args) {

        // Read the user parameters
        System.out.println("Write an EPF to a Token");
        System.out.println("------------------------");
        try {
            readParams(args);
        } catch (Exception e) {
            System.out.println("ERROR: invalid parameters.");
            e.printStackTrace();
            System.exit(0);
        }

        // Write the EPF credentials to the token
        System.out.println("\nAttempting to write the EPF to the token...");
        User user = new User();
        PKCS11LibraryConnection p11LibConn = null;
        SecureStringBuffer securePassword = null;
        try {
            // Create a credential reader that is used to read a set of 
            // credentials from an EPF
            CredentialReader filenameProfileReader =
                new FilenameProfileReader(new File(m_entrustPath, m_entrustUser + ".epf").getAbsolutePath());

            // Create a credential writer that is used to write a set of 
            // EPF credentials to a token
            p11LibConn = new PKCS11LibraryConnection(m_p11LibConn);
            CredentialWriter tokenInitializer = new TokenInitializer(m_slotId, p11LibConn, m_entrustPath, m_entrustUser);
            user.setCredentialWriter(tokenInitializer);

            // Log the user in
            securePassword = new SecureStringBuffer(new StringBuffer(m_password));
            user.login(filenameProfileReader, securePassword);
            
            // Writ the credentials and log out
            user.write();
            user.logout();
        } catch (Exception e) {
            System.out.println("ERROR: write EPF to token failed.");
            e.printStackTrace();
            System.exit(0);
        }
        System.out.println("SUCCESS: write EPF to token complete");

        // Login to the token credentials
        System.out.println("\nAttempting to log in to the token...");
        try {
            // Create a credential read that is used to read a set of
            // credentials from a token
            CredentialReader tokenReader = new TokenReader(p11LibConn, m_slotId);
            
            // Reset the user's credential writer (no longer want to use the
            // token initializer)
            user.setCredentialWriter(null);
            
            // Log the user in
            user.login(tokenReader, securePassword);
        } catch(Exception e) {
            System.out.println("ERROR token login failed.");
            e.printStackTrace();
            System.exit(0);
        }
        System.out.println("SUCCESS: token login complete");
        

        // Test Signing/Verification
        System.out.println("\n\nSign/Verify Test");
        System.out.println("----------------");
        try {
            PrivateKey signingKey = user.getSigningKey();
            X509Certificate verificationCertificate = user.getVerificationCertificate();
            if (signingKey != null && verificationCertificate != null) {
                Signature signer = Signature.getInstance("SHA-1/" + signingKey.getAlgorithm(), "Entrust");

                // Sign
                System.out.println("\nAttempt sign operation using the user's signing key...\n");
                signer.initSign(signingKey);
                signer.update(TEST_DATA);
                System.out.println("data to sign: " + iaik.utils.Util.toString(TEST_DATA));
                byte[] signature = signer.sign();
                System.out.println("signature:    " + iaik.utils.Util.toString(signature));

                // Verify
                System.out.println(
                    "\nAttempt verify operation using the user's verification certificate with serial number '"
                        + verificationCertificate.getSerialNumber()
                        + "'...\n");
                signer.initVerify(verificationCertificate);
                signer.update(TEST_DATA);
                System.out.println("data that was signed: " + iaik.utils.Util.toString(TEST_DATA));
                boolean verified = signer.verify(signature);
                System.out.println("signature verified:   " + verified);
            }
        } catch (Exception e) {
            System.out.println("ERROR: sign/verify test failed.");
            try {
                user.logout();
            } catch (Exception e1) {
                // Ignore any exceptions
            }
            e.printStackTrace();
            System.exit(0);
        }

        // Test Encryption/Decryption
        System.out.println("\n\nEncryption/Decryption Test");
        System.out.println("--------------------------");
        try {
            PrivateKey decryptionKey = user.getDecryptionKey();
            X509Certificate encryptionCertificate = user.getEncryptionCertificate();
            if (decryptionKey != null && encryptionCertificate != null) {
                Cipher cipher = Cipher.getInstance(decryptionKey.getAlgorithm(), "Entrust");

                // Encrypt
                System.out.println(
                    "\nAttempt encrypt operation using the user's encryption certificate with serial number '"
                        + encryptionCertificate.getSerialNumber()
                        + "'...\n");
                cipher.init(Cipher.ENCRYPT_MODE, encryptionCertificate.getPublicKey());
                System.out.println("plaintext:  " + iaik.utils.Util.toString(TEST_DATA));
                byte[] ciphertext = cipher.doFinal(TEST_DATA);
                System.out.println("ciphertext: " + iaik.utils.Util.toString(ciphertext));

                // Decrypt
                System.out.println(
                    "\nAttempt decrypt operation using the user's corresponding decryption private key...\n");
                cipher.init(Cipher.DECRYPT_MODE, decryptionKey);
                System.out.println("ciphertext: " + iaik.utils.Util.toString(ciphertext));
                byte[] plaintext = cipher.doFinal(ciphertext);
                System.out.println("plaintext:  " + iaik.utils.Util.toString(plaintext));
            }
        } catch (Exception e) {
            System.out.println("ERROR: encryption/decryption test failed.");
            try {
                user.logout();
            } catch (Exception e1) {
                // Ignore any exceptions
            }
            e.printStackTrace();
            System.exit(0);
        }

        // Log the user out
        try {
            user.logout();
        } catch (Exception e) {
            e.printStackTrace();
            System.exit(0);
        }

        System.out.println("\n\nDONE");
    }

    /******************************************************************************/
    /*                                  CONSTANTS                                 */
    /******************************************************************************/

    private static byte[] TEST_DATA = "Test Data".getBytes();

    /******************************************************************************/
    /*                                 PARAMETERS                                 */
    /******************************************************************************/

    /** The slot identifier of the token that the EPF will be written to. */
    private static long m_slotId;

    /** The PKCS11 Library of the token; contains the PKCS11 API necessary for 
     *  communicating with the token. */
    private static String m_p11LibConn;

    /** The directory where the Auxillary Profile will be stored. */
    private static String m_entrustPath;

    /** The name of the Auxillary Profile (without the .apf extension). */
    private static String m_entrustUser;

    /** The password for the user, which is also the PIN of the Normal User on the
     *  token and the password protecting the EPF. */
    private static String m_password;

    /******************************************************************************/
    /*                                 PARAMETERS                                 */
    /******************************************************************************/

    /*******************************************************************************
     * 
     * The default constructor.
     * 
     */
    private WriteEpfToToken() {
    }

    /******************************************************************************/
    /*                              PARAMETER PARSING                             */
    /******************************************************************************/

    /*******************************************************************************
     * 
     * Reads all the parameters necessary for writing an EPF to a token.
     * 
     * 
     * @param args
     *      the program arguments read in by the main method
     * 
     * @exception Exception
     *      thrown if an error occurs while parsing the parameters
     */
    private static void readParams(String[] args) throws Exception {
        int size = args.length;
        if (size == 0) {
            readParamsFromUser();
        } else if (size == 5) {
            readParamsFromArgs(args);
        } else {
            System.out.println("Usages:\n");
            System.out.println("WriteEpfToToken");
            System.out.println(
                "WriteEpfToToken <slot ID> <p11 lib> <password> <epf/apf dir> <epf/apf name>");
            System.exit(0);
        }
    }

    /*******************************************************************************
     * 
     * Reads all the parameters necessary for writing an EPF token from a set of 
     * string arguments.
     * 
     * 
     * @param args
     *      the program arguments read in by the main method
     * 
     * @exception Exception
     *      thrown if an error occurs while parsing the parameters from the
     *      program arguments
     */
    private static void readParamsFromArgs(String[] args) throws Exception {

        int i = 0;

        // Token information
        m_slotId = Long.parseLong(args[i++]);
        m_p11LibConn = args[i++];
        m_password = args[i++];

        // Auxillary Profile information
        m_entrustPath = args[i++];
        m_entrustUser = args[i++];
    }

    /*******************************************************************************
     * 
     * Reads all the parameters necessary for writing an EPF to a token from
     * <CODE>System.in</CODE>.
     * 
     * 
     * @exception Exception
     *      thrown if an error occurs while parsing the parameters from the
     *      <CODE>System.in</CODE>
     */
    private static void readParamsFromUser() throws Exception {

        BufferedReader in = new BufferedReader(new InputStreamReader(System.in));

        System.out.println(
            "\nPlease enter the following information, all of which is required prior writing the EPF to a token:\n");

        // Token information
        System.out.println("Token Slot Identifier (ex: 2)");
        m_slotId = Long.parseLong(in.readLine());
        System.out.println("Token PKCS11 Library (ex: slbck.dll)");
        m_p11LibConn = in.readLine();
        System.out.println("User password (Normal User PIN on the token and password to the EPF)");
        m_password = in.readLine();

        // Auxillary Profile information
        System.out.println("\nEntrust Auxillary Profile Path (ex: c:/temp)");
        m_entrustPath = in.readLine();
        System.out.println("Entrust Auxillary Profile Name (ex: TestUser)");
        m_entrustUser = in.readLine();
    }
}