//===========================================================================
//
// Copyright (c)  2003-2024 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.pkits;

import java.io.File;
import java.io.FileInputStream;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * Class to run PKITS.
 * <p>
 * The Public Key Interoperability Test Suite (PKITS) is a set of tests
 * available from <a href="http://csrc.ncsl.nist.gov/pki/testing/x509paths.html">
 * NIST</a> that attempts to cover most certificate path validation requirements
 * from X509 and RFC3280. This class, and the associated classes
 * <code>PKITSTest</code>, <code>PKITSTestConfiguration</code>, and
 * <code>PKITSTestResult</code> are used to run the test suite. The
 * configuration information is contained in the file config.xml, which
 * describes the inputs to each test and the expected result. 
 * </p>
 * <p>
 * To run the tests, you must download the test data from NIST, available
 * <a href="http://csrc.ncsl.nist.gov/pki/testing/PKITS_data.zip">here</a>.
 * Unzip the data file, preserving the directory structure. You can unzip
 * to the same directory as you run this program from, or you can put it
 * somewhere else and specify the location on the command line. 
 * </p>
 * <p>
 * From the Toolkit, either entbase.jar and entuser.jar, or enttoolkit.jar 
 * must be on the classpath. entjavaxcrypto.jar is required if a recent JCE has 
 * not been installed. A JAXP implementation must also be available, the 
 * default one provided with J2SE 1.5 is fine. 
 * </p>
 * <p>
 * The <code>PKITSTest</code> class is the only one that actually uses the
 * Toolkit, the rest of the code is merely to set up the tests and report the 
 * results. If your interest is in how to use the Toolkit APIs, this is the 
 * only class that should be looked at.
 * </p>
 * <p>
 * It is important to note that the Toolkit will not pass all of the
 * tests. In the interests of full disclosure, all of the tests described by
 * PKITS are run. At the time of this writing, the Toolkit does not support
 * DSA parameter chaining, indirect CRLs, or CRL reason code partitioning. 
 * PKITS has tests for all of these, and the results reported about success 
 * or failure have little meaning. There are 250 individual tests, 240 should 
 * pass, and 10 should fail.
 * </p>
 * <p>
 * The provided test configurations are based on PKITS as of October 2003.
 * New revisions of the suite may invalidate the configuration file or the
 * expected results described above.
 * </p>
 * 
 * <p>
 * Possible options for running the tests are:
 * <dl>
 * <dt>-d</dt><dd>Extra verbose output for debugging purposes.</dd>
 * <dt>-f configFile</dt><dd>Use the XML test configuration file found at configFile.</dd>
 * <dt>-p pkitsDataLocation</dt><dd>Find all of the PKITS data files in pkitsDataLocation.</dd>
 * <dt>-t testPrefix</dt><dd>Run all tests that start with testPrefix.</dd>
 * <dt>-T testPrefix</dt><dd>Run the first test that start with testPrefix.</dd>
 * </dl>
 * For example, from the examples directory:
 * <pre>
 * java -classpath ..\lib\enttoolkit.jar;classes com.entrust.toolkit.examples.pkits.PKITSTestRunner -t 4.6 -p c:\pkits_data
 * </pre>
 * Would run all tests that start with "4.6" (the basicConstraints tests),
 * using data files from c:\pkits_data. 
 * </p>
 * 
 * @see PKITSTest
 * @see PKITSTestConfiguration
 */
public class PKITSTestRunner
{

    /**
     * The main program that runs PKITS.
     * 
     * @param args
     *     the program arguments.
     */
    public static void main(String[] args)
    {
        try
        {
            PKITSTestRunner runner = new PKITSTestRunner();
            runner.run(args);
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }
    }

    ////////////////////////////////////////////////////////////////////////

    /** The name of the configuration file. */
    private String m_configFile = "data/pkits/config.xml";

    /** The directory that contains the PKITS data file. If no directory
     * is specified, it is assumed that the data files have been
     * extracted to the current directory.
     */
    private String m_pkitsDataFileLocation = "data/pkits";

    /** true if just a single test is being run, false otherwise. */
    private boolean m_runSingleTest = false;

    /** A prefix that test titles must match. */
    private String m_testPrefix = null;

    /** Extra debugging information. */
    private boolean m_debug = false;

	private boolean m_online;

	private String m_directory;

    /**
     * Run the tests. Parse the command line arguments,
     * parse the test configuration, run the tests, then
     * report the results.
     * 
     * @param args
     *     The arguments passed to the program.
     * 
     * @throws Exception
     */
    private void run(String args[]) throws Exception
    {
        parseCommandLine(args);
        FileInputStream in = new FileInputStream(m_configFile);
        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
        DocumentBuilder parser = factory.newDocumentBuilder();
        Document configDoc = parser.parse(in);
        List testConfigurations = parseTestConfigurations(configDoc);

        long start = System.currentTimeMillis();
        List testResults = runAllTests(testConfigurations);
        displayTestResults(testResults);
        long end = System.currentTimeMillis();
        long total = end - start;
        System.out.println("Total Run time: " + total);
        
    }

    /**
     * Parse the command line arguments.
     * 
     * @param args
     *     the command line arguments.
     */
    private void parseCommandLine(String args[])
    {
        for (int i = 0; i < args.length; ++i)
        {
            if (args[i].equals("-t") && i + 1 < args.length)
            {
                // Run tests beginning with given prefix;
                m_testPrefix = args[++i];
            }
            else if (args[i].equals("-T") && i + 1 < args.length)
            {
                // Run only first test with matching prefix;
                m_runSingleTest = true;
                m_testPrefix = args[++i];
            }
            else if (args[i].equals("-dir") && i + 1 < args.length)
            {
                // Run only first test with matching prefix;
                m_online = true;
                m_directory = args[++i];
            }
            
            else if (args[i].equals("-f") && i + 1 < args.length)
            {
                // Config file;
                m_configFile = args[++i];
            }
            else if (args[i].equals("-p") && i + 1 < args.length)
            {
                // Config file;
                m_pkitsDataFileLocation = args[++i];
            }
            else if (args[i].equals("-d"))
            {
                // Debug mode
                m_debug = true;
            }
        }
    }

    /**
     * Parse all of test configurations out of the given Document.
     * If any options about which tests to run were specified on the
     * command line, they are processed in this method.
     * 
     * @param configDoc
     *     The Document containing the test configurations.
     * 
     * @return
     *     A <code>List</code> of <code>PKITSTestConfiguration</code>
     * objects that were parsed from the given Document.
     */
    private List parseTestConfigurations(Document configDoc)
    {
        List testConfigurations = new LinkedList();
        Element doc = configDoc.getDocumentElement();
        NodeList testNodes = doc.getElementsByTagName("test");
        for (int i = 0; i < testNodes.getLength(); ++i)
        {
            Node node = testNodes.item(i);
            short type = node.getNodeType();
            if (type == Node.ELEMENT_NODE)
            {
                PKITSTestConfiguration testConfig = parseTest((Element) node);
                
                //Add online and Directory information if set
                if (m_directory != null) {
                	testConfig.setDirectory(m_directory, 389);
                	//testConfig.
                }
                if (!testConfig.isValidTest())
                {
                    throw new IllegalArgumentException();
                }
                if (m_testPrefix == null)
                {
                    testConfigurations.add(testConfig);
                }
                else
                {
                    if (testConfig.getTitle().startsWith(m_testPrefix))
                    {
                        testConfigurations.add(testConfig);
                        if (m_runSingleTest == true)
                        {
                            break;
                        }
                    }
                }
            }
        }
        return testConfigurations;
    }

    /**
     * Parse an individual test configuration.
     * Note to self: this probably would have been easier with SAX.
     * 
     * @param testElement
     *    the <code>Element</code> containing the test configuration.
     * 
     * @return
     *    a <code>PKITSTestConfiguration</code> object parsed from the Element.
     * 
     * @throws IllegalArgumentException
     *     if the given Element does not represent a valid configuration.
     */
    private PKITSTestConfiguration parseTest(Element testElement)
    {
        PKITSTestConfiguration testConfig =
            new PKITSTestConfiguration(m_pkitsDataFileLocation + File.separator);
        Element el;
        NodeList nl = testElement.getElementsByTagName("title");
        if (nl.getLength() == 1)
        {
            el = (Element) nl.item(0);
            testConfig.setTitle(getText(el));
        }

        nl = testElement.getElementsByTagName("description");
        if (nl.getLength() == 1)
        {
            el = (Element) nl.item(0);
            testConfig.setDescription(getText(el));
        }

        nl = testElement.getElementsByTagName("comment");
        if (nl.getLength() == 1)
        {
            el = (Element) nl.item(0);
            testConfig.setComment(getText(el));
        }

        nl = testElement.getElementsByTagName("shouldValidate");
        if (nl.getLength() == 1)
        {
            el = (Element) nl.item(0);
            String boolText = getText(el);
            if (boolText.equalsIgnoreCase("true"))
            {
                testConfig.setShouldValidate(true);
            }
            else if (boolText.equalsIgnoreCase("false"))
            {
                testConfig.setShouldValidate(false);
            }
            else
            {
                throw new IllegalArgumentException();
            }
        }
        else
        {
            throw new IllegalArgumentException();
        }

        nl = testElement.getElementsByTagName("inhibitAnyPolicy");
        if (nl.getLength() == 1)
        {
            el = (Element) nl.item(0);
            String boolText = getText(el);
            if (boolText.equalsIgnoreCase("true"))
            {
                testConfig.setInhibitAnyPolicy(true);
            }
            else if (boolText.equalsIgnoreCase("false"))
            {
                testConfig.setInhibitAnyPolicy(false);
            }
            else
            {
                throw new IllegalArgumentException();
            }
        }
        else
        {
            throw new IllegalArgumentException();
        }

        nl = testElement.getElementsByTagName("inhibitPolicyMapping");
        if (nl.getLength() == 1)
        {
            el = (Element) nl.item(0);
            String boolText = getText(el);
            if (boolText.equalsIgnoreCase("true"))
            {
                testConfig.setInhibitPolicyMapping(true);
            }
            else if (boolText.equalsIgnoreCase("false"))
            {
                testConfig.setInhibitPolicyMapping(false);
            }
            else
            {
                throw new IllegalArgumentException();
            }
        }
        else
        {
            throw new IllegalArgumentException();
        }

        nl = testElement.getElementsByTagName("requirePolicy");
        if (nl.getLength() == 1)
        {
            el = (Element) nl.item(0);
            String boolText = getText(el);
            if (boolText.equalsIgnoreCase("true"))
            {
                testConfig.setRequireExplicitPolicy(true);
            }
            else if (boolText.equalsIgnoreCase("false"))
            {
                testConfig.setRequireExplicitPolicy(false);
            }
            else
            {
                throw new IllegalArgumentException();
            }
        }
        else
        {
            throw new IllegalArgumentException();
        }

        nl = testElement.getElementsByTagName("acceptablePolicies");
        if (nl.getLength() == 1)
        {
            List policyList = new ArrayList();
            el = (Element) nl.item(0);
            nl = el.getElementsByTagName("policy");
            for (int j = 0; j < nl.getLength(); ++j)
            {
                policyList.add(getText(nl.item(j)));
            }
            testConfig.addAcceptablePolicies(policyList);
        }

        nl = testElement.getElementsByTagName("certPathCertificates");
        if (nl.getLength() == 1)
        {
            List certList = new ArrayList();
            el = (Element) nl.item(0);
            nl = el.getElementsByTagName("certificate");
            for (int j = 0; j < nl.getLength(); ++j)
            {
                certList.add(getText(nl.item(j)));
            }
            testConfig.addCertPathFiles(certList);
        }

        nl = testElement.getElementsByTagName("extraCertificates");
        if (nl.getLength() == 1)
        {
            List certList = new ArrayList();
            el = (Element) nl.item(0);
            nl = el.getElementsByTagName("certificate");
            for (int j = 0; j < nl.getLength(); ++j)
            {
                certList.add(getText(nl.item(j)));
            }
            testConfig.addExtraCertFiles(certList);
        }

        nl = testElement.getElementsByTagName("revocationLists");
        if (nl.getLength() == 1)
        {
            List rlList = new ArrayList();
            el = (Element) nl.item(0);
            nl = el.getElementsByTagName("revocationList");
            for (int j = 0; j < nl.getLength(); ++j)
            {
                rlList.add(getText(nl.item(j)));
            }
            testConfig.addRevocationListFiles(rlList);
        }

        nl = testElement.getElementsByTagName("trustRoot");
        if (nl.getLength() == 1)
        {
            el = (Element) nl.item(0);
            testConfig.setTrustRootFile(getText(el));
        }

        nl = testElement.getElementsByTagName("targetCertificate");
        if (nl.getLength() == 1)
        {
            el = (Element) nl.item(0);
            testConfig.setTargetCertificateFile(getText(el));
        }

        return testConfig;
    }

    /**
     * Return the full text of a node and all its children
     */
    private String getText(Node node)
    {
        if (node.getNodeType() == Node.COMMENT_NODE
            || node.getNodeType() == Node.PROCESSING_INSTRUCTION_NODE)
        {
            return "";
        }

        StringBuffer text = new StringBuffer();
        String value = node.getNodeValue();
        if (value != null)
        {
            text.append(value);
        }
        if (node.hasChildNodes())
        {
            NodeList children = node.getChildNodes();
            for (int i = 0; i < children.getLength(); ++i)
            {
                Node child = children.item(i);
                text.append(getText(child));
            }
        }
        return text.toString();
    }

    /**
     * Runs all of the tests in the given list.
     * 
     * @param testConfigurations
     *     A <code>List</code> of <code>PKITSTestConfiguration</code>
     * objects.
     * 
     * @return
     *     A <code>List</code> of <code>PKITSTestResult</code> objects,
     * one for each test in <code>testConfigurations</code>.
     */
    private List runAllTests(List testConfigurations)
    {
        List testResults = new LinkedList();
        Iterator it = testConfigurations.iterator();
        while (it.hasNext())
        {
            PKITSTestConfiguration testConfig = (PKITSTestConfiguration) it.next();
            PKITSTest theTest = new PKITSTest(testConfig, m_debug);
            Exception theException = null;
            try
            {
                System.out.println("Running test " + testConfig.getTitle());
                theTest.runTest();
            }
            catch (Exception e)
            {
                // Catch all exceptions from running the test, and store
                // them for use in the test result.
                theException = e;
            }
            PKITSTestResult result =
                new PKITSTestResult(
                    testConfig.getTitle(),
                    testConfig.getComment(),
                    testConfig.shouldValidate(),
                    theException);
            testResults.add(result);
        }
        return testResults;
    }

    /**
     * Display the results of the tests.
     * 
     * @param testResults
     *     A list of <code>PKITSTestResult</code> objects.
     */
    private void displayTestResults(List testResults)
    {
        System.out.println();
        System.out.println("Results of tests:");
        Iterator it = testResults.iterator();
        int i = 0;
        int passes = 0;
        int failures = 0;
        int possibleFailures = 0;
        while (it.hasNext())
        {
            PKITSTestResult result = (PKITSTestResult) it.next();
            System.out.println(result.getTestTitle() + ": " + result.getResultString(true));
            if (result.isSuccess())
            {
                ++passes;
            }
            if (result.isFailure())
            {
                ++failures;
            }
            if (result.isPossibleFailure())
            {
                ++possibleFailures;
            }
            ++i;
        }
        System.out.println("Total tests run: " + i);
        System.out.println("Total passes: " + passes);
        System.out.println("Total failures: " + failures);
        System.out.println("Total possible failures: " + possibleFailures);
    }
}
