//===========================================================================
//
// Copyright (c)  1998-2024 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.servlet;

import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.security.cert.X509Certificate;

import iaik.asn1.structures.AlgorithmID;

import com.entrust.toolkit.CertificateSet;
import com.entrust.toolkit.PKCS7DecodeStream;
import com.entrust.toolkit.PKCS7EncodeStream;
import com.entrust.toolkit.User;
import com.entrust.toolkit.exceptions.UserFatalException;

/**
 * A basic PKCS #7 utility class; provides APIs for encrypting/signing and 
 * decrypting/verifying messages using PKCS #7 format.
 */
public class PKCS7Util {

    /**
     * Encrypt and sign a message using PKCS #7 format.
     * 
     * 
     * @param user
     *      the user's whose credentials with be used to encrypt/sign
     * 
     * @param message
     *      the message being encrypted/signed
     * 
     * @param outputStream
     *      an output stream were the encrypted/signed message will be written
     * 
     * @throws UserFatalException
     *      in any error occurs while encrypting/signing the message
     */
    public static void writeEncryptSignedMessage(User user, String message, OutputStream outputStream) throws Exception {
        try {
            System.out.print("  Protecting message as encrypted/signed PKCS #7 message... ");

            // Prepare to encrypt/sign using PKCS #7
            PKCS7EncodeStream encodeStream = new PKCS7EncodeStream(user, outputStream,
                PKCS7EncodeStream.SIGN_AND_ENCRYPT);

            // Optionally, set the block size; the message will be processed in 
            // pieces of this size
            encodeStream.setBlockSize(1024); // Optional set the

            // Optionally, specify the digest algorithm that should be used
            // during signing (otherwise a default algorithm is used)
            encodeStream.setDigestAlgorithm(AlgorithmID.sha1);

            // Optionally, specify a symmetric cipher algorithm that should be
            // used during encryption  (otherwise a default algorithm is used
            encodeStream.setEncryptionAlgorithm(AlgorithmID.cast5_CBC);

            // Encrypt and sign the message
            encodeStream.write(message.getBytes());
            encodeStream.close();
            outputStream.flush();
            outputStream.close();
            System.out.println("DONE");
        } catch (Exception e) {
            System.out.println("FAILED");
            throw new UserFatalException("Failed while processing/sending PKCS #7 encrypted/signed message", e);
        }
    }

    /**
     * Decrypt and verify a PKCS #7 encrypted/signed message.
     * 
     * 
     * @param user
     *      the user's whose credentials with be used to encrypt/sign
     * 
     * @param is
     *      the input stream from which the encrypted/signed message can be read
     * 
     * @return
     *      the original message
     */
    public static String readEncryptedSignedMessage(User user, InputStream is) throws UserFatalException {
        System.out.print("  Extracting message from encrypted/signed PKCS#7 message... ");
        try {
            PKCS7DecodeStream decodeStream = new PKCS7DecodeStream(user, is);

            // Read the message (decrypt message verify/validate signature)
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            int bytesRead = 0;
            byte[] buffer = new byte[1024];
            while ((bytesRead = decodeStream.read(buffer)) >= 0) {
                baos.write(buffer, 0, bytesRead);
            }

            // Print encryption and signature information
            System.out.println("DONE");
            printData(decodeStream);
            return baos.toString();
        } catch (Exception e) {
            System.out.println("FAILED");
            throw new UserFatalException("Failed while receiving/processing PKCS #7 encrypted/signed message", e);
        }
    }

    /**
     * Prints information about the PKCS #7 operation that was done.
     * 
     * 
     * @param pkcs7DecodeStream
     *      the PKCS #7 decode stream that was used to do the PKCS #7 operation
     */
    private static void printData(PKCS7DecodeStream pkcs7DecodeStream) {
        System.out.println("  Signature and encryption information:");
        System.out.println("  - operation: " + pkcs7DecodeStream.getOperation());
        AlgorithmID algID = pkcs7DecodeStream.getEncryptionAlgorithm();
        String aglName = "null - undefined";
        if (algID != null)
            aglName = algID.getName();
        System.out.println("  - encryption alg: " + aglName);
        int numOfSignatures = pkcs7DecodeStream.getNumberOfSignatures();
        System.out.println("  - number of signatures: " + numOfSignatures);
        for (int i = 0; i < numOfSignatures; i++) {
            System.out.println("  - signature " + i);
            System.out.println("    - digest algorithm: " + pkcs7DecodeStream.getDigestAlgorithm(i));
            System.out.println("    - signer certificate subject name: "
                + pkcs7DecodeStream.getSignerCertificate(i).getSubjectDN().getName());
        }
        System.out.println("  - included certificates:");
        CertificateSet certSet = pkcs7DecodeStream.getIncludedCertificates();
        X509Certificate certs[] = null;
        if (certSet != null)
            certs = certSet.getCertificates();
        if (certs == null) {
            System.out.println("    - none");
        } else {
            int numOfCerts = certs.length;
            for (int i = 0; i < numOfCerts; i++) {
                System.out.println("    - " + certs[i].getSubjectDN().getName());
            }
        }
    }
}