//===========================================================================
//
// Copyright (c)  2005-2024 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.smimev3;

/**
 * @author jogray
 * 
 * TODO To change the template for this generated type comment go to Window -
 * Preferences - Java - Code Style - Code Templates
 */

import iaik.asn1.ASN1;
import iaik.asn1.CodingException;

import iaik.cms.CompressedDataStream;

import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.FileNotFoundException;
import java.io.IOException;

public class CompressedContentDecoder {

	/**
	 * This Sample Program reads in the contents of a CMS CompressedContent
	 * ASN.1 type as defined in RFC 3274. It requires an ASN.1 Compression
	 * content as well as a Filename where to write the decompressed output.
	 * <pre>
	 * Usage: CompressedContentRipper &lt;ASN.1 CompressedContent FileName&gt; &lt;outputfile&gt;
	 * </pre>
	 * 
	 * @param args
	 *            args[0] is the Filename of an exiting DER encoded CMS
	 *            CompressedContent
	 * @param args
	 *            args[1] is where the decompressed data should be written
	 * 
	 * Note: This sample also uses the sample
	 * com.entrust.toolkit.examples.smimev3.NullCompression object which
	 * provides the alg_null algorithm so that any CMS CompressedContent
	 * compressed with alg_null (no compression) can be decoded properly.
	 *  
	 */

	public static void main(String[] args) {

		if (args.length < 2) {
			System.out
					.println("Usage:  CompressedContentRipper <ASN.1 CompressedContent FileName> <output file>");
			return;
		}

		// Assign the Input arguments to string names, and print them out:
		String inputFile = args[0];
		String outputFile = args[1];
		System.out.println("The inputFile to read:   " + inputFile);
		System.out.println("The outputFile to write: " + outputFile);

		try {

			// Display the ASN1 components of this CompressedContent ASN.1 object.
			displayASN1Structure(inputFile);
	
			//Read the CompressedDataStream object
			System.out.println("Attempting to read supplied data as a CMS CompressedContent ASN.1 type....");
			CompressedDataStream cdatastream = new CompressedDataStream(
					new FileInputStream(inputFile));
			System.out.println("Successfully read CMS CompressedContent!");
            
			// For messages that are compressed with algorithms defined in NullCompression.
			// This is not needed if default ZLIB compression is used.
		    NullCompression ncompression = new NullCompression();
			cdatastream.setCompressionProvider(ncompression);

			// Print out some information about the CompressedContent
			System.out.println("\nInformation in CompressedContent");
			System.out.println("The Version is: " + cdatastream.getVersion());
			System.out.println("The algorithm used is: "
					+ cdatastream.getCompressionAlgorithm());
			System.out.println("The encapsulated content is: "
					+ cdatastream.getEncapsulatedContentType());

			//Get the InputStream from the compressed data Stream.
			InputStream dataStream = cdatastream.getInputStream();
            
			//Write decoded contents of the CompressedInputStream to outputFile
			System.out.println("Writing decompressed encapsulated content to " + outputFile);
			FileOutputStream fos = new FileOutputStream(outputFile);
			byte[] b2 = new byte[128];
			int i2 = dataStream.read(b2);
			while (i2 >= 0) {
				fos.write(b2, 0, i2);
				i2 = dataStream.read(b2);
			}
			
			System.out.println("Done!");

		} catch (Exception e) {
			e.printStackTrace();
		}

	}

	/**
	 * Display the ASN1 components of this CompressedContent ASN.1 object.
	 * The Structure should be as follows:
	 * 
	 * <pre>
	 * There are 2 components in this ASN.1 structure
	 * OBJECT ID = 1.2.840.113549.1.9.16.1.9
	 * 
	 * CONTEXTSPECIFIC[C][I] = [0] EXPLICIT
	 *   SEQUENCE[C][I] = 3 elements
	 *     INTEGER = 0
	 *     SEQUENCE[C] = 1 elements
	 *       OBJECT ID = 1.2.840.113549.1.9.16.3.9
	 *     SEQUENCE[C][I] = 2 elements
	 *      OBJECT ID = PKCS#7 data
	 *      CONTEXTSPECIFIC[C][I] = [0] EXPLICIT
	 *        OCTET STRING[C][I] = 1 elements
	 *          OCTET STRING = 144 bytes: 43:6F:6E:74:65...
	 *</pre>
	 * @param inputFile the filename of the CompressedContents Der encoded ASN.1 object
	 * @throws CodingException
	 * @throws FileNotFoundException
	 * @throws IOException
	 */
	public static void displayASN1Structure(String inputFile)
			throws CodingException, FileNotFoundException, IOException {
		
		System.out.println("Attempting to display ASN.1 structure...");
		ASN1 asn_obj = new ASN1(new FileInputStream(inputFile));
		int components = asn_obj.countComponents();
		System.out.println("There are " + components
				+ " components in this ASN.1 structure");
		for (int i = 0; i < components; i++) {
			System.out.println(ASN1.print(asn_obj.getComponentAt(i)));
		}

	}
}
