//===========================================================================
//
// Copyright (c)  2003-2024 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.smimev3;

import iaik.asn1.structures.AlgorithmID;
import iaik.smime.EncryptedContent;
import iaik.smime.SignedContent;
import iaik.utils.Base64OutputStream;
import iaik.x509.X509Certificate;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import jakarta.activation.CommandMap;
import jakarta.activation.DataHandler;
import jakarta.activation.DataSource;
import jakarta.activation.MailcapCommandMap;
import jakarta.mail.Header;

import com.entrust.toolkit.KeyAndCertificateSource;
import com.entrust.toolkit.PKCS7EncodeStream;
import com.entrust.toolkit.User;
import com.entrust.toolkit.credentials.CredentialReader;
import com.entrust.toolkit.credentials.FilenameProfileReader;
import com.entrust.toolkit.exceptions.UserNotLoggedInException;
import com.entrust.toolkit.util.SecureStringBuffer;

/**
 * <p>This sample shows how to create S/MIME protected files (files with
 * a .p7m extension) that can be read by Entrust Entelligence. It shows
 * show to create signed, encrypted, and signed and encrypted p7m
 * files. Two strategies are shown for each type, one which uses the
 * S/MIME classes in the iaik.smime and iaik.cms packages, and one which uses
 * the PKCS7 classes in the entp7 JAR file.</p>
 * 
 * <p>For each protection method (signed, encrypted, signed and encrypted),
 * the sample can create files using both definite and indefinite length
 * ASN.1 encoding. Entelligence can only read files that use definite length
 * encoding, but using definite length encoding requires all data to be
 * buffered in memory and is thus not suitable for large files. Indefinite
 * length encoding can be used to create p7m files of arbitrary size, but they
 * cannot be read by Entelligence. The <code>Asn1EncodingConverter</code>
 * class in the Toolkit can be used to convert large p7m files to an
 * encoding format that Entelligence can read.</p>
 * 
 * <p>The sample is very simplistic around character encoding and content types.
 * It uses 7 bit ASCII for all data that is being protected, and text/plain
 * as the content type. In practice, this will
 * likely not be the case, which means more sophisticated handling will
 * be required. The <code>MimeUtility.encode()</code> methods from the JavaMail
 * API can be used to help with encoding, and the <code>MimetypesFileTypeMap</code>
 * from the JavaBeans activation framework can be used to help with content
 * types.</p>
 * 
 * <p>This sample does not show how to decode the files which are output,
 * they are expected to be used with Entelligence.</p>
 * 
 * <p>To run the sample, from the <code>etjava\examples</code> directory,
 * run the following:</p>
 * <pre>java -classpath classes;../etjava/lib/enttoolkit.jar com.entrust.toolkit.examples.smime.EntelligenceInterop
 *       "epf file" "password" ["path to write output files"]</pre>
 * <p>For example:</p>
 * <pre>java -classpath classes;../etjava/lib/enttoolkit.jar com.entrust.toolkit.examples.smime.EntelligenceInterop
 *       data/userdata/RSAUser1.epf ~Sample7~</pre>
 * 
 */
public class EntelligenceInterop
{
    private KeyAndCertificateSource m_keyAndCertificateSource;
    private String m_fileLocation;
    private boolean m_doIndefiniteLength;
    private boolean m_doDefiniteLength;


    /**
     * The main driver for the sample.
     * 
     * @param args
     *     The program arguments. See the class description for details.
     */
    public static void main(String[] args)
    {
        if (args.length != 2 && args.length != 3)
        {
            System.out.println("Usage: EntelligenceInterop <epf file> <password> [<path to write output files>]");
            return;
        }

        // Create a mailcap file and set it as the default command map.
        // This is required so the S/MIME classes can get the correct content
        // handler when encrypting a signed message in createSmimeSignedAndEncryptedMessage
        MailcapCommandMap mailcapCommandMap = new MailcapCommandMap();
        String mailCap =
            "#\r\n"
                + "# 'mailcap' file\r\n"
                + "#\r\n"
                + "multipart/signed;;               x-java-content-handler=iaik.smime.signed_content\r\n"
                + "application/x-pkcs7-signature;;  x-java-content-handler=iaik.smime.signed_content\r\n"
                + "application/x-pkcs7-mime;;       x-java-content-handler=iaik.smime.encrypted_content\r\n"
                + "application/x-pkcs10;;           x-java-content-handler=iaik.smime.pkcs10_content\r\n"
                + "application/pkcs7-signature;;    x-java-content-handler=iaik.smime.signed_content\r\n"
                + "application/pkcs7-mime;;         x-java-content-handler=iaik.smime.encrypted_content\r\n"
                + "application/pkcs10;;             x-java-content-handler=iaik.smime.pkcs10_content\r\n";
        mailcapCommandMap.addMailcap(mailCap);
        CommandMap.setDefaultCommandMap(mailcapCommandMap);

        // Log in to the supplied epf file
        User user = new User();
        KeyAndCertificateSource keyAndCertificateSource;
        try
        {
            System.out.println("Logging in");
            CredentialReader reader = new FilenameProfileReader(args[0]);
            user.login(reader, new SecureStringBuffer(new StringBuffer(args[1])));
            keyAndCertificateSource = new KeyAndCertificateSource(user);
        }
        catch (Exception e)
        {
            e.printStackTrace();
            return;
        }

        // Set up the arguments that control which files are created and where.
        boolean doDefiniteLength = true;
        boolean doIndefiniteLength = true;
        String outputFileLocation = ""; // default location is current working directory
        if( args.length == 3 )
        {
            outputFileLocation = args[2] + File.separator;
        }
        // Create the object that will create the p7m files.
        EntelligenceInterop interop = new EntelligenceInterop(keyAndCertificateSource, outputFileLocation, doIndefiniteLength, doDefiniteLength);

        // Run the code to make the different varieties of signed and encrypted files.
        boolean doSmimeSigned = true;
        boolean doSmimeEncrypted = true;
        boolean doSmimeSignedEncrypted = true;
        boolean doPkcs7Signed = true;
        boolean doPkcs7Encrypted = true;
        boolean doPkcs7SignedEncrypted = true;

        if (doSmimeSigned)
        {
            interop.doSmimeSigned();
        }

        if (doSmimeEncrypted)
        {
            interop.doSmimeEncrypted();
        }

        if (doSmimeSignedEncrypted)
        {
            interop.doSmimeSignedEncrypted();
        }

        if (doPkcs7Signed)
        {
            interop.doPkcs7Signed();
        }

        if (doPkcs7Encrypted)
        {
            interop.doPkcs7Encrypted();
        }

        if (doPkcs7SignedEncrypted)
        {
            interop.doPkcs7SignedEncrypted();
        }

        // Log out the user now that everything is done.
        try
        {
            user.logout();
        }
        catch (UserNotLoggedInException e)
        {
        }

        System.out.println("Done");

    }

    /**
     * Create an EntelligenceInterop object.
     * 
     * @param keyAndCertificateSource
     *     a source of certificates and private keys.
     * 
     * @param outputFileLocation
     *     the location to write the p7m files that are created. Passing in
     * "" will cause the files to be written to the current working directory.
     * 
     * @param doIndefiniteLength
     *     whether or not files should be created using indefinite length
     * encoding. An arbitrary amount of data can be protected when using
     * indefinite length, but the files created are not compatible with
     * Entelligence.
     * 
     * @param doDefiniteLength
     *     whether or not files should be created using definite length
     * encoding. The files created can be read by Entelligence, but only
     * a limited amount of data can be protected as it must all be
     * read in to memory.
     */
    EntelligenceInterop(KeyAndCertificateSource keyAndCertificateSource, 
        String outputFileLocation,
        boolean doIndefiniteLength,
        boolean doDefiniteLength)
    {
        m_keyAndCertificateSource = keyAndCertificateSource;
        m_fileLocation = outputFileLocation;
        m_doIndefiniteLength = doIndefiniteLength;
        m_doDefiniteLength = doDefiniteLength;
    }

    /**
     * Runs the code that creates signed p7m files using the code from the smimev3 JAR file.
     */
    void doSmimeSigned()
    {
        System.out.println("doSmimeSigned begins");
        try
        {
            if (m_doDefiniteLength)
            {
                createSmimeSignedMessage(new DataHandler(new SampleDataSource()), 0);
            }
            if (m_doIndefiniteLength)
            {
                createSmimeSignedMessage(new DataHandler(new SampleDataSource()), 4001);
            }

        }
        catch (Exception e)
        {
            e.printStackTrace();
        }
        System.out.println("doSmimeSigned DONE");
    }

    /**
     * Creates a signed p7m file using the classes in the smimev3 and cms JAR files.
     * 
     * @param dataHandler
     *     the data handler that provides the data to be signed/ 
     * @param blockSize
     *     the block size to use during ASN.1 encoding.
     * 
     * @throws Exception
     */
    void createSmimeSignedMessage(DataHandler dataHandler, int blockSize) throws Exception
    {
        String fileName = getFileName("smimeSigned", blockSize);

        SignedContent signedContent = new SignedContent(true);
        signedContent.setBlockSize(blockSize);

        // Create headers to add to the message, which override the defaults.
        // This is important to preserve the name of the file being signed.
        List headerList = new ArrayList();
        headerList.add(
            new Header(
                "Content-Type",
                "text/plain;\r\n\tcharset=\"US-ASCII\";\r\n\tname=" + fileName + ".txt"));
        headerList.add(new Header("Content-Disposition", "attachment;\r\n\tfilename=" + fileName + ".txt"));
        Header[] headers = new Header[headerList.size()];
        headerList.toArray(headers);
        signedContent.setContentContentHeaders(headers);

        // Set the data handler of the content so that it has somewhere to
        // get the data to sign from. 
        signedContent.setDataHandler(dataHandler);

        // Set the certificates to include with the message, and add the
        // logged in user as a signer.
        X509Certificate signerCertificate = m_keyAndCertificateSource.getVerificationCertificate();
        X509Certificate[] certChain = m_keyAndCertificateSource.getCertificateChain();
        X509Certificate[] signerCertificates = new X509Certificate[certChain.length + 1];
        signerCertificates[0] = signerCertificate;
        System.arraycopy(certChain, 0, signerCertificates, 1, certChain.length);
        signedContent.setCertificates(signerCertificates);
        signedContent.addSigner(m_keyAndCertificateSource.getSigningKey(), signerCertificate);

        // Finally, write the signed data to a file. 
        OutputStream os =
            new BufferedOutputStream(new FileOutputStream(m_fileLocation + fileName + ".p7m"), 100000);
        signedContent.writeTo(os);
        os.close();

        /*
         * The following code demonstrates how the decoding might be done.
         * Without using the full capabilities of S/MIME it is a bit awkward,
         * as there is no easy way to tell whether is a signed  or
         * encrypted message, and the content encoding is not handled either.
         * However, when using full S/MIME, the JavaMail API will read the
         * whole message in to memory, which does not work for large files.
        InputStream is = new BufferedInputStream(new FileInputStream(m_fileLocation + fileName + ".p7m"));
        SMimeSigned verifier = new SMimeSigned(is);
        InputStream contentInputStream = verifier.getInputStream();
        byte[] decoded = new byte[1024];
        int bytesRead = contentInputStream.read(decoded);
        while (bytesRead > 0)
        {
            // Do something with the decoded bytes.
            bytesRead = contentInputStream.read(decoded);
        }
        contentInputStream.close();
        
        X509Certificate signerCert = verifier.verify();
        m_user.validate(signerCert);
        */
    }

    /**
     * Runs the code that creates encrypted p7m files using the code from the smimev3 JAR file.
     */
    void doSmimeEncrypted()
    {
        System.out.println("doSmimeEncrypted begins");
        try
        {
            if (m_doDefiniteLength)
            {
                createSmimeEncryptedMessage(new DataHandler(new SampleDataSource()), 0);
            }
            if (m_doIndefiniteLength)
            {
                createSmimeEncryptedMessage(new DataHandler(new SampleDataSource()), 4001);
            }
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }
        System.out.println("doSmimeEncrypted DONE");
    }

    /**
     * Creates an encrypted p7m file using the classes in the smimev3 and cms JAR files.
     * 
     * @param dataHandler
     *     the data handler that provides the data to be encrypted 
     * @param blockSize
     *     the block size to use during ASN.1 encoding.
     * 
     * @throws Exception
     */
    void createSmimeEncryptedMessage(DataHandler dataHandler, int blockSize) throws Exception
    {
        String fileName = getFileName("smimeEncrypted", blockSize);

        EncryptedContent encryptedContent = new EncryptedContent();
        encryptedContent.setBlockSize(blockSize);

        // Create headers to add to the message, which override the defaults.
        // This is important to preserve the name of the file being encrypted.
        List headerList = new ArrayList();
        headerList.add(
            new Header(
                "Content-Type",
                "text/plain;\r\n\tcharset=\"US-ASCII\";\r\n\tname=" + fileName + ".txt"));
        headerList.add(new Header("Content-Disposition", "attachment;\r\n\tfilename=" + fileName + ".txt"));
        Header[] headers = new Header[headerList.size()];
        headerList.toArray(headers);
        encryptedContent.setContentContentHeaders(headers);

        // Set the data handler of the content so that it has somewhere to
        // get the data to encrypt. 
        encryptedContent.setDataHandler(dataHandler);

        // Set the encryption algorithm to be used.
        encryptedContent.setEncryptionAlgorithm(AlgorithmID.des_EDE3_CBC, 128);

        // Add the logged in user as the only recipient of the message.
        encryptedContent.addRecipient(m_keyAndCertificateSource.getEncryptionCertificate(), AlgorithmID.rsaEncryption);

        // Now encrypt the data and write it to a file.
        OutputStream os = new FileOutputStream(m_fileLocation + fileName + ".p7m");
        encryptedContent.writeTo(os);
        os.close();

        /*
         * The following code illustrates how simple decoding might be done.
         * This method will not work for large files as the JavaMail classes
         * read the whole file in to memory. To work with large files, the
         * SMimeEncrypted class can be used.
        FileDataSource fileDataSource = new FileDataSource(m_fileLocation + fileName + ".p7m");
        EncryptedContent decryptor = new EncryptedContent(fileDataSource);
        X509Certificate recipientCertificate = m_user.getEncryptionCertificate();
        PrivateKey decryptionKey =
            m_user.getDecryptionKey(
                (Name) recipientCertificate.getIssuerDN(),
                recipientCertificate.getSerialNumber().toString());
        decryptor.decryptSymmetricKey(decryptionKey, recipientCertificate);
        InputStream contentInputStream = decryptor.getContentInputStream();
        byte[] decoded = new byte[1024];
        int bytesRead = contentInputStream.read(decoded);
        while (bytesRead > 0)
        {
            bytesRead = contentInputStream.read(decoded);
        }
        contentInputStream.close();
        */
    }

    /**
     * Runs the code that creates signed and encrypted p7m files using
     * the code from the smimev3 JAR file.
     */
    void doSmimeSignedEncrypted()
    {
        System.out.println("doSmimeSignedEncrypted begins");
        try
        {
            if (m_doDefiniteLength)
            {
                createSmimeSignedAndEncryptedMessage(new DataHandler(new SampleDataSource()), 0);
            }
            if (m_doIndefiniteLength)
            {
                createSmimeSignedAndEncryptedMessage(new DataHandler(new SampleDataSource()), 4001);
            }
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }
        System.out.println("doSmimeSignedEncrypted DONE");
    }

    /**
     * Creates a signed and encrypted p7m file using the classes in the smimev3 and cms JAR files.
     * 
     * @param dataHandler
     *     the data handler that provides the data to be signed and encrypted 
     * @param blockSize
     *     the block size to use during ASN.1 encoding.
     * 
     * @throws Exception
     */
    void createSmimeSignedAndEncryptedMessage(DataHandler dataHandler, int blockSize) throws Exception
    {
        String fileName = getFileName("smimeSignedEncrypted", blockSize);
        SignedContent signedContent = new SignedContent(true);
        signedContent.setBlockSize(blockSize);

        // Create headers to add to the message, which override the defaults.
        // This is required to preserve the name of the file being signed.
        List signedHeaderList = new ArrayList();
        signedHeaderList.add(
            new Header(
                "Content-Type",
                "text/plain;\r\n\tcharset=\"US-ASCII\";\r\n\tname=" + fileName + ".txt"));
        signedHeaderList.add(
            new Header("Content-Disposition", "attachment;\r\n\tfilename=" + fileName + ".txt"));
        Header[] signedHeaders = new Header[signedHeaderList.size()];
        signedHeaderList.toArray(signedHeaders);
        signedContent.setContentContentHeaders(signedHeaders);

        // Set the data handler of the content so that it has somewhere to
        // get the data to sign from. 
        signedContent.setDataHandler(dataHandler);

        X509Certificate signerCertificate = m_keyAndCertificateSource.getVerificationCertificate();
        X509Certificate[] certChain = m_keyAndCertificateSource.getCertificateChain();
        X509Certificate[] signerCertificates = new X509Certificate[certChain.length + 1];
        signerCertificates[0] = signerCertificate;
        System.arraycopy(certChain, 0, signerCertificates, 1, certChain.length);

        // Set the certificates to include with the message, and add the
        // logged in user as a signer.
        signedContent.setCertificates(signerCertificates);
        signedContent.addSigner(m_keyAndCertificateSource.getSigningKey(), signerCertificate);

        // Now create the object that will encrypt the signed data stream.
        EncryptedContent encryptedContent = new EncryptedContent(signedContent);
        encryptedContent.setBlockSize(blockSize);

        // encrypt for the currently logged in user.
        encryptedContent.addRecipient(m_keyAndCertificateSource.getEncryptionCertificate(), AlgorithmID.rsaEncryption);
        encryptedContent.setEncryptionAlgorithm(AlgorithmID.des_EDE3_CBC, 128);

        // Create headers to add to the message, which override the defaults.
        // This is required to preserve the name of the file being encrypted.
        List encryptionHeaderList = new ArrayList();
        encryptionHeaderList.add(
            new Header("Content-Type", "application/x-pkcs7-mime;\r\n\tname=" + fileName + ".p7m"));
        encryptionHeaderList.add(
            new Header("Content-Disposition", "attachment;\r\n\tfilename=" + fileName + ".p7m"));
        Header[] encryptionHeaders = new Header[encryptionHeaderList.size()];
        encryptionHeaderList.toArray(encryptionHeaders);
        encryptedContent.setContentContentHeaders(encryptionHeaders);

        // Now do the encrypting and signing.
        OutputStream os = new BufferedOutputStream(new FileOutputStream(m_fileLocation + fileName + ".p7m"));
        encryptedContent.writeTo(os);
        os.close();
    }

    /**
     * Runs the code that creates signed p7m files using the code
     * from the p7 JAR file.
     */
    void doPkcs7Signed()
    {
        System.out.println("doPkcs7Signed begins");
        try
        {
            if (m_doDefiniteLength)
            {
                createPkcs7SignedMessage(new SampleDataSource(), 0);
            }
            if (m_doIndefiniteLength)
            {
                createPkcs7SignedMessage(new SampleDataSource(), 2323);
            }
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }
        System.out.println("doPkcs7Signed DONE");
    }

    /**
     * Creates a signed p7m file using the classes in the p7 JAR file.
     * 
     * @param dataSource
     *     the data handler that provides the data to be signed. 
     * @param blockSize
     *     the block size to use during ASN.1 encoding.
     * 
     * @throws Exception
     */
    void createPkcs7SignedMessage(DataSource dataSource, int blockSize) throws Exception
    {
        String fileName = getFileName("pkcs7Signed", blockSize);

        // Create the object that will sign the data
        OutputStream os = new FileOutputStream(m_fileLocation + fileName + ".p7m");
        PKCS7EncodeStream pkcs7Signer = new PKCS7EncodeStream(m_keyAndCertificateSource, os, PKCS7EncodeStream.SIGN_ONLY);
        pkcs7Signer.setBlockSize(blockSize);

        // Write the MIME header fields. This must be done using ASCII encoding.
        String header =
            "Content-Type: text/plain; charset=\"US-ASCII\"; name=" + fileName + ".txt\r\n"
          + "Content-Transfer-Encoding: 7bit\r\n"
          + "Content-Disposition: attachment; filename=" + fileName + ".txt\r\n"
          + "\r\n";
        pkcs7Signer.write(header.getBytes("US-ASCII"));

        // Now sign the data
        InputStream is = dataSource.getInputStream();
        byte[] toSign = new byte[1024];
        int bytesRead = is.read(toSign);
        while (bytesRead > 0)
        {
            pkcs7Signer.write(toSign, 0, bytesRead);
            bytesRead = is.read(toSign);
        }
        pkcs7Signer.close();

        /*
         * Verification can be done using PKCS7DecodeStream, but code must be written
         * to parse the MIME headers to extract the file name and correct content
         * encoding. PKCS7DecodeStream works at the byte level only, and has
         * no concept of character encoding.
         */

    }

    /**
     * Runs the code that creates encrypted p7m files using the code
     * from the p7 JAR file.
     */
    void doPkcs7Encrypted()
    {
        System.out.println("doPkcs7Encrypted begins");
        try
        {
            if (m_doDefiniteLength)
            {
                createPkcs7EncryptedMessage(new SampleDataSource(), 0);
            }
            if (m_doIndefiniteLength)
            {
                createPkcs7EncryptedMessage(new SampleDataSource(), 333);
            }
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }
        System.out.println("doPkcs7Encrypted DONE");
    }

    /**
     * Creates an encrypted p7m file using the classes in the p7 JAR file.
     * 
     * @param dataSource
     *     the data handler that provides the data to be encrypted. 
     * @param blockSize
     *     the block size to use during ASN.1 encoding.
     * 
     * @throws Exception
     */
    void createPkcs7EncryptedMessage(DataSource dataSource, int blockSize) throws Exception
    {
        String fileName = getFileName("pkcs7Encrypted", blockSize);

        // Create the object that will encrypt the data
        OutputStream os = new FileOutputStream(m_fileLocation + fileName + ".p7m");
        PKCS7EncodeStream pkcs7Encryptor = new PKCS7EncodeStream(m_keyAndCertificateSource, os, PKCS7EncodeStream.ENCRYPT_ONLY);
        pkcs7Encryptor.setBlockSize(blockSize);
        pkcs7Encryptor.setEncryptionAlgorithm(AlgorithmID.des_EDE3_CBC);

        // Write the MIME header fields. This must be done using ASCII encoding.
        String header =
            "Content-Type: text/plain; charset=\"US-ASCII\"; name=" + fileName + ".txt\r\n"
          + "Content-Transfer-Encoding: 7bit\r\n"
          + "Content-Disposition: attachment; filename=" + fileName + ".txt\r\n"
          + "\r\n";
        pkcs7Encryptor.write(header.getBytes("US-ASCII"));

        // Now encrypt the data
        InputStream is = dataSource.getInputStream();
        byte[] toEncrypt = new byte[1024];
        int bytesRead = is.read(toEncrypt);
        while (bytesRead > 0)
        {
            pkcs7Encryptor.write(toEncrypt, 0, bytesRead);
            bytesRead = is.read(toEncrypt);
        }
        pkcs7Encryptor.close();

        /*
         * Decryption can be done using PKCS7DecodeStream, but code must be written
         * to parse the MIME headers to extract the file name and correct content
         * encoding. PKCS7DecodeStream works at the byte level only, and has
         * no concept of character encoding.
         */

    }

    /**
     * Runs the code that creates signed and encrypted p7m files using the code
     * from the p7 JAR file.
     */
    void doPkcs7SignedEncrypted()
    {
        System.out.println("doPkcs7SignedEncrypted begins");
        try
        {
            if (m_doDefiniteLength)
            {
                createPkcs7SignedEncryptedMessage(new SampleDataSource(), 0);
            }
            if (m_doIndefiniteLength)
            {
                createPkcs7SignedEncryptedMessage(new SampleDataSource(), 1234);
            }
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }
        System.out.println("doPkcs7SignedEncrypted DONE");
    }

    /**
     * Creates a signed and encrypted p7m file using the classes in the p7 JAR file.
     * 
     * @param dataSource
     *     the data handler that provides the data to be signed and encrypted. 
     * @param blockSize
     *     the block size to use during ASN.1 encoding.
     * 
     * @throws Exception
     */
    void createPkcs7SignedEncryptedMessage(DataSource dataSource, int blockSize) throws Exception
    {
        /*
         * There are two options for how to do this. One is to use
         * PKCS7EncodeStream in the SIGN_AND_ENCRYPT mode, which uses 
         * the PKCS7 SignedAndEnveloped data type to do the signing and
         * encryption in one shot. While this is compatible
         * with Entelligence, it is not part of the S/MIME standard.
         * 
         * The second approach is the one taken here, using two
         * PKCS7EncodeStream instances. The first stream signs the data.
         * The signed data is then Base64 encoded, and that data encrypted
         * with another stream, which has a slightly different header.
         * Decoding such a stream is decidedly non-trivial.
         */

        String fileName = getFileName("pkcs7SignedEncrypted", blockSize);

        // First, create the encrypting stream, and write the header to it.
        OutputStream os = new FileOutputStream(m_fileLocation + fileName + ".p7m");
        PKCS7EncodeStream pkcs7Encryptor = new PKCS7EncodeStream(m_keyAndCertificateSource, os, PKCS7EncodeStream.ENCRYPT_ONLY);
        pkcs7Encryptor.setEncryptionAlgorithm(AlgorithmID.des_EDE3_CBC);
        pkcs7Encryptor.setBlockSize(blockSize);
        String header =
            "Content-Type: application/x-pkcs7-mime; name=" + fileName + ".p7m\r\n"
          + "Content-Transfer-Encoding: base64\r\n"
          + "Content-Disposition: attachment; filename=" + fileName + ".p7m\r\n"
          + "\r\n";
        pkcs7Encryptor.write(header.getBytes("US-ASCII"));

        // Second, create the stream that will sign the data.
        os = new Base64OutputStream(pkcs7Encryptor);
        PKCS7EncodeStream pkcs7Signer = new PKCS7EncodeStream(m_keyAndCertificateSource, os, PKCS7EncodeStream.SIGN_ONLY);
        pkcs7Signer.setBlockSize(blockSize);

        // Write the MIME header fields for the signed content.
        header =
            "Content-Type: text/plain; charset=\"US-ASCII\"; name=" + fileName + ".txt\r\n"
          + "Content-Transfer-Encoding: 7bit\r\n"
          + "Content-Disposition: attachment; filename=" + fileName + ".txt\r\n"
          + "\r\n";
        pkcs7Signer.write(header.getBytes("US-ASCII"));

        // Now sign and encrypt the data
        InputStream is = dataSource.getInputStream();
        byte[] toEncrypt = new byte[1024];
        int bytesRead = is.read(toEncrypt);
        while (bytesRead > 0)
        {
            pkcs7Signer.write(toEncrypt, 0, bytesRead);
            bytesRead = is.read(toEncrypt);
        }
        pkcs7Signer.close();
    }

    /**
     * Private helper method to create a file name based on the block size being used.
     * @param originalName the original file name
     * @param blockSize the block size being used during ASN.1 encoding
     * @return a file name specific to the block size.
     */
    private static String getFileName(String originalName, int blockSize)
    {
        if (blockSize > 0)
        {
            return originalName + "Indefinite";
        }
        else
        {
            return originalName + "Definite";
        }
    }

}

/**
 * This class is a data source that will return a fixed message.
 * It is not particularly useful, it is just for demonstration purposes.
 */
class SampleDataSource implements DataSource
{
    /**
     * An input stream to be returned from the data source.
     * This stream will just output the same message a fixed
     * number of times. 
     */
    static class SampleDataSourceInputStream extends InputStream
    {
        /** The number of times that the message will be returned. */
        private static final int MAX_READ_COUNT = 10;

        /** The message to return. */
        private static byte[] MESSAGE;
        static {
            try
            {
                MESSAGE =
                    (
                        "Classification of animals:\r\n"
                            + "1.  those that belong to the Emperor.\r\n"
                            + "2.  embalmed ones.\r\n"
                            + "3.  those that are trained.\r\n"
                            + "4.  suckling pigs.\r\n"
                            + "5.  mermaids.\r\n"
                            + "6.  fabulous ones.\r\n"
                            + "7.  stray dogs.\r\n"
                            + "8.  those included in the present classification.\r\n"
                            + "9.  those that tremble as if they were mad.\r\n"
                            + "10. innumerable ones.\r\n"
                            + "11. those drawn with a very fine camelhair brush.\r\n"
                            + "12. others.\r\n"
                            + "13. those that have just broken a flower vase.\r\n"
                            + "14. those that from a long way off look like flies.\r\n").getBytes(
                        "US-ASCII");
            }
            catch (UnsupportedEncodingException e)
            {
                // Platforms that don't support ASCII as they are supposed to
                // get a boring message. 
                MESSAGE = new byte[100];
                Arrays.fill(MESSAGE, (byte) 'C');
            }
        }

        /** Number of times read() has been called. */
        int m_readCount = 0;

        /*
         * The following methods override those from InputStream.
         */
        public int available()
        {
            return MESSAGE.length;
        }

        public int read()
        {
            if (m_readCount >= MAX_READ_COUNT)
            {
                return -1;
            }
            else
            {
                ++m_readCount;
                return 'C';
            }
        }

        public int read(byte[] b, int offset, int length) throws IOException
        {
            if (b == null)
            {
                throw new NullPointerException();
            }
            else if (
                (offset < 0)
                    || (offset > b.length)
                    || (length < 0)
                    || ((offset + length) > b.length)
                    || ((offset + length) < 0))
            {
                throw new IndexOutOfBoundsException();
            }
            else if (m_readCount >= MAX_READ_COUNT)
            {
                return -1;
            }
            else if (length == 0)
            {
                return 0;
            }

            if (m_readCount >= MAX_READ_COUNT)
            {
                return -1;
            }
            else
            {
                int bytesToCopy = Math.min(MESSAGE.length, length);
                System.arraycopy(MESSAGE, 0, b, offset, bytesToCopy);
                ++m_readCount;
                return bytesToCopy;
            }
        }
    }

    /**
     * Returns the content type of this data source.
     * @return the content type of this data source.
     */
    public String getContentType()
    {
        return "text/plain";
    }

    /**
     * Returns the input stream associated with this source.
     * @return the input stream associated with this source.
     */
    public InputStream getInputStream()
    {
        return new SampleDataSourceInputStream();
    }

    /**
     * Returns the name of this data source
     * @return the name of this data source.
     */
    public String getName()
    {
        return "SampleDataSource";
    }

    /**
     * @throws IOException always, this operation is not supported.
     */
    public OutputStream getOutputStream() throws IOException
    {
        throw new IOException("getOutputStream() is not supported");
    }
}
