//===========================================================================
//
// Copyright (c)  2005-2024 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.smimev3;

import iaik.asn1.structures.AlgorithmID;
import iaik.cms.CompressionProvider;
import iaik.cms.InputStreamCompressEngine;

import java.io.InputStream;
import java.security.NoSuchAlgorithmException;

/**
 * This class demonstrates the How the SMime Compression architecture can be
 * extended to enable any type of compression algorithm. This implementation is
 * just an example of a compress algorithm that does not compress the data.
 *  
 */
public class NullCompression extends CompressionProvider {

	/**
	 * This method returns an InputStreamCompressEngine for compressing or
	 * decompressing data read from some input stream. An
	 * InputStreamCompressEngine is used by class CompressedDataStream for
	 * content compression or decompression.
	 * 
	 * @param algorithmid
	 *            the compression/decompression algorithm to be used
	 * @param inputstream
	 *            the input stream supplying the data to be
	 *            compressed/decompressed
	 * @param CompressMode
	 *            the compression mode to be used (COMPRESS or DECOMPRESS)
	 * @param blocksize
	 *            the size of the internal buffer, if used by the compress
	 *            engine.
	 * 
	 * @return the InputStreamCompressEngine just created
	 */

	public InputStreamCompressEngine getInputStreamCompressEngine(
			AlgorithmID algorithmid, InputStream inputstream, int CompressMode,
			int blocksize) throws NoSuchAlgorithmException {
		//Select the new type of compress algorithm
		if (algorithmid.equals(NullCompression.alg_null))
			return new NULLCompressionInputStream(inputstream, CompressMode,
					blocksize);

		//If not the algorithm we just defined, then call superclass
		return super.getInputStreamCompressEngine(algorithmid, inputstream,
				CompressMode, blocksize);

	}

	//Now provide the Compression Algorithms which are used to

	/**
	 * Creates an AlgorithmID for the <code>ALG_NULL</code> compression
	 * algorithm. This algorithm does not apply any compression.. The ObjectID
	 * number used here is not an official ObjectID tag, it is just use here as
	 * an example.
	 * <ul>
	 * <li>ObjectID = "1.2.840.113549.1.9.16.3.36"
	 * <li>name = "ALG_NULL"
	 * <li>implementationName = "ALG_NULL"
	 * </ul>
	 * 
	 * For addressing it, use <code>AlgorithmID.alg_null</code>.
	 */

	public static AlgorithmID alg_null = new AlgorithmID(
			"1.2.840.113549.1.9.16.3.9", "alg_null", "alg_null");

}
