//===========================================================================
//
// Copyright (c)  2002-2024 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.xml.decryptionTransform;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.util.Vector;

import com.entrust.toolkit.User;
import com.entrust.toolkit.credentials.StreamProfileReader;
import com.entrust.toolkit.util.SecureStringBuffer;
import com.entrust.toolkit.exceptions.UserFatalException;
import com.entrust.toolkit.exceptions.UserNotLoggedInException;
import com.entrust.toolkit.xencrypt.exceptions.EncryptionHandlerException;
import com.entrust.toolkit.xencrypt.exceptions.XMLEInitException;
import com.entrust.toolkit.xencrypt.init.XMLEInit;
import com.entrust.toolkit.examples.xml.utils.ElementEncryptor;
import com.entrust.toolkit.examples.xml.utils.Utils;

import iaik.x509.X509Certificate;
import iaik.ixsil.init.IXSILInit;
import iaik.ixsil.util.URI;
import iaik.ixsil.util.ExternalReferenceResolverImpl;
import iaik.ixsil.exceptions.InitException;
import iaik.ixsil.exceptions.URIException;
import iaik.ixsil.exceptions.ExternalReferenceResolverException;

/**
 * This class encrypts all elements of a given name in an XML document.
 *<P>
 * The "Decryption Transform" demonstration uses this class to create a sample
 * document that contains a mixture of encrypted and/or signed XML content.
 * The verifier determines which elements to decrypt by using a Decryption Transform.
 *</P>
 */
public class EncryptElements {

    /**
    * main method
    */
	public static void main(String args[])
	throws FileNotFoundException,
	       UserNotLoggedInException,
	       EncryptionHandlerException,
	       XMLEInitException,
	       InitException,
	       ExternalReferenceResolverException,
	       URIException
	{
        int index = 0 ;
		String profile        = args[index++];
		String password       = args[index++];
		String propertiesURL  = args[index++];
		String documentFile   = args[index++];
		String encryptedFile  = args[index++] ;
		String elementName    = args[index++];

		Vector recipientCertificates = new Vector();
        while (args.length > index) {
            if (args[index].equals("-cert")) {
                recipientCertificates.addElement(args[++index]);
            }
            ++index;
        }

        // Display the command line arguments
		System.out.println("\nInput document: \"" + documentFile + "\"");
		System.out.println("Encrypt elements named: \"" + elementName + "\"");
		System.out.println("Output document: \"" + encryptedFile + "\"");

		User user = null  ;
		X509Certificate[] encryptionCertificates = new X509Certificate[recipientCertificates.size()] ;
		try {

		    // Log in to the Toolkit
            user = Utils.login(profile, new SecureStringBuffer(new StringBuffer(password)));

            // Read the encryption certificates.
            for(int i=0; i<recipientCertificates.size(); i++) {
                String cert = (String)recipientCertificates.elementAt(i);
                System.out.println("Recipient: " + cert);
                encryptionCertificates[i] = new X509Certificate( new FileInputStream(cert));
            }

            // If no encryption certificates were provided, the user will encrypt for themself.
            if(encryptionCertificates.length == 0) {
                encryptionCertificates = new X509Certificate[1];
                encryptionCertificates[0] = user.getEncryptionCertificate();
            }
        }
        catch (Exception e)
        {
            System.out.println("Failed to log in: " + e.toString());
            System.exit(0);
        }

	    // Get the properties for the XMLE system.
        URI initProps = new URI(propertiesURL);

	    // Initialize IXSIL
        System.out.println("Initializing IXSIL properties from \"" + initProps + "\"...");
        IXSILInit.init(initProps);

	    // Initialize XMLE.
        ExternalReferenceResolverImpl res = new ExternalReferenceResolverImpl(initProps);
        XMLEInit initializer = new XMLEInit(res.resolve(initProps));

	    // The ElementEncryptor utility encrypts all elements that have a particular name.
        System.out.println("Encrypting...");
        ElementEncryptor enc = new ElementEncryptor(user, encryptionCertificates, elementName);
        enc.setContentOnly(false);  // it's the default anyway
        enc.encrypt(initializer, documentFile) ;

        // Write the encrypted document to a file.
        enc.writeResult(encryptedFile);

        // Report the number of elements that were encrypted.
        System.out.println("Encrypted " + enc.getNumberOfElementsEncrypted() + " elements.");
	}
}
